<?php
session_start();
if (!isset($_SESSION['user_id'])) {
    header('Location: login');
    exit;
}
include '../includes/header.php';
include '../config/database.php';

$statusMessage = '';
if (isset($_GET['status'])) {
    if ($_GET['status'] === 'submitted') {
        $statusMessage = 'Your details have been submitted.';
    } elseif ($_GET['status'] === 'saved') {
        $statusMessage = 'Your details have been saved. You can continue later.';
    } elseif ($_GET['status'] === 'error_upload') {
        $statusMessage = 'There was an error uploading your file. Please try again.';
    } elseif ($_GET['status'] === 'invalid_file_type') {
        $statusMessage = 'Invalid file type. Only PDF and image files (JPG, PNG) are allowed.';
    }
}

// Fetch the logged-in user's group and existing details if any
$userId = $_SESSION['user_id'];
$stmt = $pdo->prepare('SELECT `group`, (SELECT status FROM student_details WHERE user_id = ?) AS submission_status FROM users WHERE id = ?');
$stmt->execute([$userId, $userId]);
$user = $stmt->fetch();
$group = htmlspecialchars($user['group'], ENT_QUOTES, 'UTF-8');
$submissionStatus = htmlspecialchars($user['submission_status'], ENT_QUOTES, 'UTF-8');

$isSubmitted = $submissionStatus === 'submitted';
$isSaved = $submissionStatus === 'saved';

$lecturerName = ''; // Initialize lecturer name variable

if ($isSaved || $isSubmitted) {
    $stmt = $pdo->prepare('SELECT * FROM student_details WHERE user_id = ?');
    $stmt->execute([$userId]);
    $studentDetails = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Check if a lecturer_id is set and fetch the supervisor's (lecturer's) name
    if (!empty($studentDetails['lecturer_id'])) {
        $stmt = $pdo->prepare('SELECT name FROM lecturer_login WHERE id = ?');
        $stmt->execute([$studentDetails['lecturer_id']]);
        $lecturer = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($lecturer) {
            $lecturerName = htmlspecialchars($lecturer['name'], ENT_QUOTES, 'UTF-8');
        }
    }
} else {
    $studentDetails = [
        'surname' => '',
        'first_name' => '',
        'other_names' => '',
        'telephone' => '',
        'email' => '',
        'course' => '',
        'organisation_name' => '',
        'months' => '',
        'location' => '',
        'gps_location' => '',
        'region' => '',
        'district' => '',
        'acceptance_letter' => ''
    ];
}

// Fetch regions from the database
$stmt = $pdo->prepare('SELECT DISTINCT region FROM districts');
$stmt->execute();
$regions = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Define courses of study based on the group
$courses = [];
if ($group === 'Industrial Internship') {
    $courses = [
        'Accounting',
        'Procurement and Supply Chain Management',
        'Business Information Systems',
        'Human Resource Management',
        'Banking and Finance',
        'Insurance and Risk Management',
        'Marketing and Entrepreneurship'
    ];
} elseif ($group === 'Teaching Practice') {
    $courses = [
        'Management Education',
        'Accounting Education'
    ];
}
?>

<!-- Inline styling for dashboard elements -->
<style>
  .btn-custom {
      background-color: #007bff;
      color: white !important;
      border: none;
      border-radius: 0;
      padding: 10px 20px;
      text-align: center;
      text-decoration: none;
      display: inline-block;
      font-size: 16px;
      margin: 4px 2px;
      cursor: pointer;
      box-shadow: 0 2px 4px rgba(0,0,0,0.2);
      transition: background-color 0.3s, box-shadow 0.3s;
  }
  .btn-custom:hover {
      background-color: #0056b3;
      box-shadow: 0 4px 8px rgba(0,0,0,0.2);
  }
  .spinner {
      display: none;
      margin-left: 10px;
      width: 20px;
      height: 20px;
      border: 3px solid #f3f3f3;
      border-top: 3px solid #007bff;
      border-radius: 50%;
      animation: spin 1s linear infinite;
  }
  @keyframes spin {
      0% { transform: rotate(0deg); }
      100% { transform: rotate(360deg); }
  }
  /* Styling for the swipe button */
  .swipe-button-container {
      margin-top: 20px;
  }
  #swipeButton {
      background-color: #007bff;
      color: white;
      padding: 15px 30px;
      font-size: 18px;
      border-radius: 0;
      display: inline-block;
      user-select: none;
      cursor: pointer;
      touch-action: pan-y;
  }
  #swipeButton i {
      margin-left: 10px;
      animation: bounce 1s infinite;
  }
  @keyframes bounce {
      0%, 20%, 50%, 80%, 100% { transform: translateX(0); }
      40% { transform: translateX(5px); }
      60% { transform: translateX(3px); }
  }
  /* Supervisor name styling below swipe button */
  .supervisor-name {
      margin-top: 15px;
      text-align: center;
      font-size: 18px;
      color: #007bff;
      font-weight: bold;
  }
  /* Mobile adjustments for supervisor name */
  @media (max-width: 576px) {
      .supervisor-name {
          font-size: 16px;
      }
  }
  @media (max-width: 480px) {
      h2, h4 {
          font-size: 1.5rem;
      }
      .btn-custom {
          font-size: 14px;
          padding: 8px 16px;
      }
      #swipeButton {
          font-size: 16px;
          padding: 10px 20px;
      }
  }
</style>

<div class="container mt-4">
  <?php if ($statusMessage): ?>
      <div class="alert alert-info" id="statusMessage">
          <?php echo htmlspecialchars($statusMessage, ENT_QUOTES, 'UTF-8'); ?>
      </div>
  <?php endif; ?>

  <!-- Wrap both cards in a single form -->
  <form method="POST" action="save_student_details" enctype="multipart/form-data" id="studentForm">
    <div class="row justify-content-center">
      <div class="col-md-8 col-lg-6">
          <h2>University of Education, Winneba</h2>
          <h4>School of Business.</h4>

          <?php if ($isSubmitted): ?>
              <!-- Download Card -->
              <div class="card mb-4">
                  <div class="card-header">
                      <h3>Download Supervisor's Evaluation Form</h3>
                  </div>
                  <div class="card-body text-center">
                      <a href="download_evaluation" class="btn btn-custom">
                          <i class="fas fa-download"></i> Download
                      </a>
                  </div>
              </div>
              <!-- Swipe Button Under Download Card (Activated) -->
              <div class="swipe-button-container text-center">
                  <div id="swipeButton">
                      Swipe to Post-Internship Report <i class="fas fa-arrow-right"></i>
                  </div>
              </div>
              <!-- Display Supervisor Name Below Swipe Button -->
              <?php if (!empty($lecturerName)): ?>
                  <div class="supervisor-name">
                      Supervisor: <?php echo $lecturerName; ?>
                  </div>
              <?php endif; ?>
          <?php else: ?>
              <!-- Student's Details Card -->
              <div class="card mb-4">
                  <div class="card-header">
                      <h3>Student's Details</h3>
                  </div>
                  <div class="card-body">
                          <fieldset <?php echo $isSubmitted ? 'disabled' : ''; ?>>
                              <div class="mb-3">
                                  <label for="surname" class="form-label">Surname</label>
                                  <input type="text" class="form-control" id="surname" name="surname" value="<?php echo htmlspecialchars($studentDetails['surname'], ENT_QUOTES, 'UTF-8'); ?>" required>
                              </div>
                              <div class="mb-3">
                                  <label for="firstName" class="form-label">First Name</label>
                                  <input type="text" class="form-control" id="firstName" name="firstName" value="<?php echo htmlspecialchars($studentDetails['first_name'], ENT_QUOTES, 'UTF-8'); ?>" required>
                              </div>
                              <div class="mb-3">
                                  <label for="otherNames" class="form-label">Other Names</label>
                                  <input type="text" class="form-control" id="otherNames" name="otherNames" value="<?php echo htmlspecialchars($studentDetails['other_names'], ENT_QUOTES, 'UTF-8'); ?>">
                              </div>
                              <div class="mb-3">
                                  <label for="telephone" class="form-label">Telephone Number</label>
                                  <input type="tel" class="form-control" id="telephone" name="telephone" pattern="[0-9]{10,}" value="<?php echo htmlspecialchars($studentDetails['telephone'], ENT_QUOTES, 'UTF-8'); ?>" required>
                              </div>
                              <div class="mb-3">
                                  <label for="email" class="form-label">Email</label>
                                  <input type="email" class="form-control" id="email" name="email" value="<?php echo htmlspecialchars($studentDetails['email'], ENT_QUOTES, 'UTF-8'); ?>" required>
                              </div>
                              <div class="mb-3">
                                  <label for="course" class="form-label">Course of Study</label>
                                  <select class="form-control" id="course" name="course" required>
                                      <option value="">Select Course</option>
                                      <?php foreach ($courses as $course): ?>
                                          <option value="<?php echo htmlspecialchars($course, ENT_QUOTES, 'UTF-8'); ?>" <?php echo htmlspecialchars($studentDetails['course'], ENT_QUOTES, 'UTF-8') === $course ? 'selected' : ''; ?>>
                                              <?php echo htmlspecialchars($course, ENT_QUOTES, 'UTF-8'); ?>
                                          </option>
                                      <?php endforeach; ?>
                                  </select>
                              </div>
                          </fieldset>
                  </div>
              </div>

              <!-- Organisation/School Card -->
              <div class="card mb-4">
                  <div class="card-header">
                      <h3><?php echo $group === 'Industrial Internship' ? 'Organisation' : 'School'; ?></h3>
                  </div>
                  <div class="card-body">
                          <fieldset <?php echo $isSubmitted ? 'disabled' : ''; ?>>
                              <div class="mb-3">
                                  <label for="organisationName" class="form-label"><?php echo $group === 'Industrial Internship' ? 'Name of Organisation' : 'Name of School'; ?></label>
                                  <input type="text" class="form-control" id="organisationName" name="organisationName" value="<?php echo htmlspecialchars($studentDetails['organisation_name'], ENT_QUOTES, 'UTF-8'); ?>" required>
                              </div>
                              <div class="mb-3">
                                  <label for="months" class="form-label">Number of Months</label>
                                  <select class="form-control" id="months" name="months" required>
                                      <option value="">Select Months</option>
                                      <?php for ($i = 1; $i <= 6; $i++): ?>
                                          <option value="<?php echo htmlspecialchars($i, ENT_QUOTES, 'UTF-8'); ?>" <?php echo htmlspecialchars($studentDetails['months'], ENT_QUOTES, 'UTF-8') == $i ? 'selected' : ''; ?>>
                                              <?php echo htmlspecialchars($i, ENT_QUOTES, 'UTF-8'); ?> month<?php echo $i > 1 ? 's' : ''; ?>
                                          </option>
                                      <?php endfor; ?>
                                  </select>
                              </div>
                              <div class="mb-3">
                                  <label for="location" class="form-label"><?php echo $group === 'Industrial Internship' ? 'Location of Organisation' : 'Location of School'; ?></label>
                                  <input type="text" class="form-control" id="location" name="location" value="<?php echo htmlspecialchars($studentDetails['location'], ENT_QUOTES, 'UTF-8'); ?>" required>
                              </div>
                              <div class="mb-3">
                                  <label for="gpsLocation" class="form-label">
                                      Ghana Post GPS / 
                                      <a href="https://www.google.com/maps/" target="_blank">Google location</a> of the Organisation
                                  </label>
                                  <input type="text" class="form-control" id="gpsLocation" name="gpsLocation" value="<?php echo htmlspecialchars($studentDetails['gps_location'], ENT_QUOTES, 'UTF-8'); ?>" required>
                                  <small id="gpsError" class="form-text text-danger" style="display:none;">Please enter a valid GPS location (e.g., coordinate, Google Maps URL, or Ghana Post GPS code).</small>
                              </div>
                              <div class="mb-3">
                                  <label for="region" class="form-label">Region</label>
                                  <select class="form-control" id="region" name="region" required>
                                      <option value="">Select Region</option>
                                      <?php foreach ($regions as $region): ?>
                                          <option value="<?php echo htmlspecialchars($region['region'], ENT_QUOTES, 'UTF-8'); ?>" <?php echo htmlspecialchars($studentDetails['region'], ENT_QUOTES, 'UTF-8') === $region['region'] ? 'selected' : ''; ?>>
                                              <?php echo htmlspecialchars($region['region'], ENT_QUOTES, 'UTF-8'); ?>
                                          </option>
                                      <?php endforeach; ?>
                                  </select>
                              </div>
                              <div class="mb-3">
                                  <label for="district" class="form-label">District</label>
                                  <select class="form-control" id="district" name="district" required>
                                      <option value="">Select District</option>
                                      <?php if ($studentDetails['region']): ?>
                                          <?php
                                          $stmt = $pdo->prepare('SELECT district FROM districts WHERE region = ?');
                                          $stmt->execute([htmlspecialchars($studentDetails['region'], ENT_QUOTES, 'UTF-8')]);
                                          $districts = $stmt->fetchAll(PDO::FETCH_ASSOC);
                                          foreach ($districts as $district):
                                          ?>
                                              <option value="<?php echo htmlspecialchars($district['district'], ENT_QUOTES, 'UTF-8'); ?>" <?php echo htmlspecialchars($studentDetails['district'], ENT_QUOTES, 'UTF-8') === $district['district'] ? 'selected' : ''; ?>>
                                                  <?php echo htmlspecialchars($district['district'], ENT_QUOTES, 'UTF-8'); ?>
                                              </option>
                                          <?php endforeach; ?>
                                      <?php endif; ?>
                                  </select>
                              </div>
                              <div class="mb-3">
                                  <label for="acceptanceLetter" class="form-label">Upload Acceptance Letter</label>
                                  <input type="file" class="form-control" id="acceptanceLetter" name="acceptanceLetter" accept=".pdf, .jpg, .jpeg, .png" required>
                                  <small id="fileError" class="form-text text-danger" style="display:none;">Please upload a valid acceptance letter.</small>
                                  <div class="spinner" id="processingSpinner"></div>
                                  <?php if ($studentDetails['acceptance_letter']): ?>
                                      <p>Current file: <?php echo htmlspecialchars(basename($studentDetails['acceptance_letter']), ENT_QUOTES, 'UTF-8'); ?></p>
                                  <?php endif; ?>
                              </div>
                              <!-- Submit Button -->
                              <?php if (!$isSubmitted): ?>
                                  <div class="d-grid gap-2 d-md-flex justify-content-md-end">
                                      <button type="submit" name="submit" class="btn btn-primary" style="border-radius: 0;" id="submitBtn">Submit</button>
                                  </div>
                              <?php endif; ?>
                          </fieldset>
                  </div>
              </div>
          <?php endif; ?>
      </div>
    </div>
  </form>
</div>

<!-- JavaScript to auto-populate districts based on selected region -->
<script>
// Listen for changes on the region select element
document.getElementById('region').addEventListener('change', function() {
    var selectedRegion = this.value;
    var districtSelect = document.getElementById('district');
    
    // Clear existing options in the district select
    districtSelect.innerHTML = '<option value="">Select District</option>';
    
    if (selectedRegion !== "") {
        // Use AJAX to fetch districts for the selected region from fetch_districts.php
        var xhr = new XMLHttpRequest();
        xhr.open('GET', 'fetch_districts.php?region=' + encodeURIComponent(selectedRegion), true);
        xhr.onload = function() {
            if (xhr.status === 200) {
                try {
                    var districts = JSON.parse(xhr.responseText);
                    districts.forEach(function(item) {
                        var option = document.createElement('option');
                        option.value = item.district;
                        option.textContent = item.district;
                        districtSelect.appendChild(option);
                    });
                } catch(e) {
                    console.error('Error parsing JSON: ' + e);
                }
            } else {
                console.error('Error fetching districts: ' + xhr.status);
            }
        };
        xhr.send();
    }
});
</script>

<!-- Swipe detection for activated swipe button -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    var swipeButton = document.getElementById('swipeButton');
    if (swipeButton) {
        var startX = 0;
        var threshold = 100; // Minimum swipe distance in pixels

        swipeButton.addEventListener('pointerdown', function(e) {
            startX = e.clientX;
            swipeButton.setPointerCapture(e.pointerId);
        });

        swipeButton.addEventListener('pointerup', function(e) {
            var diffX = e.clientX - startX;
            if (diffX > threshold) {
                console.log('Swipe detected. Redirecting...');
                window.location.href = 'internship_report.php';
            }
            swipeButton.releasePointerCapture(e.pointerId);
        });

        swipeButton.addEventListener('pointercancel', function(e) {
            swipeButton.releasePointerCapture(e.pointerId);
        });
    }
});
</script>

<?php include '../includes/footer.php'; ?>
