<?php
session_start();
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../includes/error_handler.php';

// Enable error reporting for debugging
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Generate CSRF token if it doesn't exist
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

try {
    // Error handling for database connection
    if (!$pdo) {
        throw new Exception('Database connection failed.');
    }
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    if (!isset($_SESSION['user_id'])) {
        header('Location: login.php');
        exit;
    }

    // CSRF Protection
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        if (!isset($_POST['csrf_token']) || !isset($_SESSION['csrf_token']) || 
            !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
            throw new Exception('CSRF token validation failed. Please try submitting the form again.');
        }
    }

    $userId     = $_SESSION['user_id'];
    // $photoJson  = $_POST['photo_album_entries'] ?? '[]';

    // Validate required fields
    if (!isset($_FILES['presentation']) || $_FILES['presentation']['error'] !== UPLOAD_ERR_OK) {
        throw new Exception('Presentation upload failed. Error code: ' . ($_FILES['presentation']['error'] ?? 'unknown'));
    }

    // Remove photo album validation
    // $entries = json_decode($photoJson, true);
    // if (!is_array($entries) || count($entries) === 0) {
    //     throw new Exception('No photos received. Make sure you upload at least one image and caption it.');
    // }

    // Handle Presentation
    $allowedPresentationTypes = ['application/vnd.ms-powerpoint', 'application/vnd.openxmlformats-officedocument.presentationml.presentation'];
    $maxFileSize = 10 * 1024 * 1024; // 10MB

    if (!in_array($_FILES['presentation']['type'], $allowedPresentationTypes) || $_FILES['presentation']['size'] > $maxFileSize) {
        throw new Exception('Invalid presentation file type or size. Please upload a PPT or PPTX file under 10MB.');
    }

    // Create upload directories if they don't exist
    $presentationDir = __DIR__ . '/../uploads/presentations/';
    if (!is_dir($presentationDir)) {
        if (!mkdir($presentationDir, 0755, true)) {
            throw new Exception('Failed to create presentation upload directory.');
        }
    }

    // Handle CV
    if (!isset($_FILES['cv']) || $_FILES['cv']['error'] !== UPLOAD_ERR_OK) {
        throw new Exception('CV upload failed. Error code: ' . ($_FILES['cv']['error'] ?? 'unknown'));
    }

    // Validate file type and size
    $allowedTypes = ['application/pdf', 'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'];
    $maxFileSize = 5 * 1024 * 1024; // 5MB

    if (!in_array($_FILES['cv']['type'], $allowedTypes) || $_FILES['cv']['size'] > $maxFileSize) {
        throw new Exception('Invalid file type or size. Please upload a PDF or DOC file under 5MB.');
    }

    // Create upload directories if they don't exist
    $cvDir = __DIR__ . '/../uploads/cvs/';
    if (!is_dir($cvDir)) {
        if (!mkdir($cvDir, 0755, true)) {
            throw new Exception('Failed to create upload directory.');
        }
    }

    // Fetch the user's index_number from the users table
    $stmt = $pdo->prepare('SELECT index_number FROM users WHERE id = ?');
    $stmt->execute([$userId]);
    $user = $stmt->fetch();
    $indexNumber = $user['index_number'];

    // Generate new filename for presentation using index_number
    $presentationExt = pathinfo($_FILES['presentation']['name'], PATHINFO_EXTENSION);
    $presentationName = $indexNumber . '_presentation.' . $presentationExt;
    $presentationPath = 'uploads/presentations/' . $presentationName;

    // Move uploaded presentation file
    if (!move_uploaded_file($_FILES['presentation']['tmp_name'], $presentationDir . $presentationName)) {
        throw new Exception('Failed to save presentation. Please check directory permissions.');
    }

    // Generate new filename for CV using index_number
    $cvExt = pathinfo($_FILES['cv']['name'], PATHINFO_EXTENSION);
    $cvName = $indexNumber . '_report.' . $cvExt;
    $cvPath = 'uploads/cvs/' . $cvName;

    // Move uploaded file
    if (!move_uploaded_file($_FILES['cv']['tmp_name'], $cvDir . $cvName)) {
        throw new Exception('Failed to save CV. Please check directory permissions.');
    }

    // Save to DB
    $stmt = $pdo->prepare("
        INSERT INTO teaching_reports (user_id, presentation_path, cv_path, status, submitted_at)
        VALUES (?, ?, ?, 'submitted', NOW())
    ");
    
    $stmt->execute([
        $userId,
        $presentationPath,
        $cvPath
    ]);

    // Log successful submission
    error_log("Teaching report submitted successfully by user ID: $userId");

    header("Location: teaching_report.php?submit=1&success=1");
    exit;
} catch (Exception $e) {
    error_log("Error in submit_teaching_report.php: " . $e->getMessage());
    throw $e; // This will be caught by our exception handler
}
