<?php
ob_start(); 
session_start();

if (!isset($_SESSION['user_id'])) {
    header('Location: login');
    exit;
}

// -----------------------------------------------------
// Ensure you have a valid $pdo object
// -----------------------------------------------------

include '../includes/header.php';

// Block phones / narrow screens
echo <<<HTML
<script>
document.addEventListener('DOMContentLoaded', function() {
  if (window.innerWidth < 576) {
    document.body.innerHTML =
      '<div class="container mt-5"><div class="alert alert-danger">' +
      'Please access this page on a desktop screen to complete the form.' +
      '</div></div>';
  }
});
</script>
HTML;

// Retrieve the student's company/organization from student_details
$stmt = $pdo->prepare("SELECT organisation_name FROM student_details WHERE user_id = ?");
$stmt->execute([$_SESSION['user_id']]);
$studentDetail = $stmt->fetch(PDO::FETCH_ASSOC);
$companyFromDB = $studentDetail ? $studentDetail['organisation_name'] : '';

// Check if the user has saved progress (status = 'saved')
$savedData = null;
$stmt = $pdo->prepare("SELECT * FROM internship_reports WHERE user_id = ? AND status = 'saved' LIMIT 1");
$stmt->execute([$_SESSION['user_id']]);
$savedData = $stmt->fetch(PDO::FETCH_ASSOC);

// Check if the user has already submitted their report (status = 'submitted')
$stmt = $pdo->prepare("SELECT * FROM internship_reports WHERE user_id = ? AND status = 'submitted' LIMIT 1");
$stmt->execute([$_SESSION['user_id']]);
$submittedReport = $stmt->fetch(PDO::FETCH_ASSOC);

// Use GET parameters for PRG feedback
$submitted = isset($_GET['submit']);
$saved     = isset($_GET['save']);

// If the form is submitted
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // If already submitted, stop
    if ($submittedReport) {
        header("Location: internship_report.php?submit=1");
        exit;
    }
    
    // Company is autofilled
    $company     = $companyFromDB;
    $department  = htmlspecialchars(trim($_POST['department'] ?? ''));
    $startDate   = htmlspecialchars(trim($_POST['startDate'] ?? ''));
    $endDate     = htmlspecialchars(trim($_POST['endDate'] ?? ''));

    // Retrieve internship report sections
    $executiveSummary         = htmlspecialchars(trim($_POST['executiveSummary'] ?? ''));
    $introduction             = htmlspecialchars(trim($_POST['introduction'] ?? ''));
    $institutionProfile       = htmlspecialchars(trim($_POST['institutionProfile'] ?? ''));
    $goalsObjectives          = htmlspecialchars(trim($_POST['goalsObjectives'] ?? ''));
    $coreDuties               = htmlspecialchars(trim($_POST['coreDuties'] ?? ''));
    $organizationalChallenges = htmlspecialchars(trim($_POST['organizationalChallenges'] ?? ''));
    $studentChallenges        = htmlspecialchars(trim($_POST['studentChallenges'] ?? ''));
    $achievements             = htmlspecialchars(trim($_POST['achievements'] ?? ''));
    $lessonsLearned           = htmlspecialchars(trim($_POST['lessonsLearned'] ?? ''));
    $recommendations          = htmlspecialchars(trim($_POST['recommendations'] ?? ''));
    $conclusion               = htmlspecialchars(trim($_POST['conclusion'] ?? ''));
    $references               = htmlspecialchars(trim($_POST['references'] ?? ''));

    // Determine status
    if (isset($_POST['save'])) {
        $status = 'saved';
    } else {
        $status = 'submitted';
    }

    // INSERT ... ON DUPLICATE KEY UPDATE
    $sql = "INSERT INTO internship_reports 
            (user_id, company, department, start_date, end_date, executive_summary, introduction, institution_profile, goals_objectives, core_duties, organizational_challenges, student_challenges, achievements, lessons_learned, recommendations, conclusion, `references`, submission_date, status)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), ?)
            ON DUPLICATE KEY UPDATE 
                department = VALUES(department),
                start_date = VALUES(start_date),
                end_date = VALUES(end_date),
                executive_summary = VALUES(executive_summary),
                introduction = VALUES(introduction),
                institution_profile = VALUES(institution_profile),
                goals_objectives = VALUES(goals_objectives),
                core_duties = VALUES(core_duties),
                organizational_challenges = VALUES(organizational_challenges),
                student_challenges = VALUES(student_challenges),
                achievements = VALUES(achievements),
                lessons_learned = VALUES(lessons_learned),
                recommendations = VALUES(recommendations),
                conclusion = VALUES(conclusion),
                `references` = VALUES(`references`),
                submission_date = NOW(),
                status = VALUES(status)";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([
        $_SESSION['user_id'],
        $company,
        $department,
        $startDate,
        $endDate,
        $executiveSummary,
        $introduction,
        $institutionProfile,
        $goalsObjectives,
        $coreDuties,
        $organizationalChallenges,
        $studentChallenges,
        $achievements,
        $lessonsLearned,
        $recommendations,
        $conclusion,
        $references,
        $status
    ]);

    // Redirect using PRG
    if ($status === 'saved') {
        header("Location: internship_report.php?save=1");
    } else {
        header("Location: internship_report.php?submit=1");
    }
    exit;
}

// Should form be disabled?
$formDisabled = (bool)$submittedReport;
?>

<!-- Pagination Styles -->
<style>
  .step { display: none; }
  .step.active { display: block; }
</style>

<div class="container mt-5 mb-5" id="mainContent">
  <h1 class="mb-4">Internship Report Form</h1>
  
  <?php if($submitted): ?>
    <div class="alert alert-success" role="alert" id="submitAlert">
      <h4 class="alert-heading">Internship Report Submitted</h4>
      <p>Your internship report has been successfully submitted.</p>
      <hr>
      <p class="mb-0">Thank you for your submission.</p>
    </div>
  <?php elseif($saved): ?>
    <div class="alert alert-info" role="alert" id="saveAlert">
      <h4 class="alert-heading">Progress Saved</h4>
      <p>Your progress has been saved. You can continue later.</p>
    </div>
  <?php endif; ?>

  <?php if($formDisabled): ?>
    <div class="alert alert-warning" role="alert">
      <h4 class="alert-heading">Submission Locked</h4>
      <p>You have already submitted your internship report. Further changes are disabled.</p>
      <hr>
      <button type="button" class="btn btn-info" data-bs-toggle="modal" data-bs-target="#submittedReportModal">
        View Submitted Report
      </button>
    </div>
  <?php endif; ?>
  
  <?php if(!$formDisabled): ?>
    <div id="progressIndicator" class="mb-4">
      <p id="stepIndicator">Step 1 of 3</p>
      <div class="progress">
        <div id="progressBar" class="progress-bar" role="progressbar" style="width: 33%;" aria-valuenow="33" aria-valuemin="0" aria-valuemax="100"></div>
      </div>
    </div>
  <?php endif; ?>
  
  <form action="<?php echo htmlspecialchars($_SERVER["PHP_SELF"]); ?>" method="POST" enctype="multipart/form-data" class="needs-validation" novalidate>
    <fieldset <?php echo $formDisabled ? 'disabled' : ''; ?>>
      <!-- Step 1 -->
      <div class="step active" id="step-1">
        <h3>Internship Details</h3>
        <div class="row mb-3">
          <div class="col-md-6">
            <label for="company" class="form-label">Company/Organization</label>
            <input type="text" class="form-control" id="company" name="company"
                   value="<?php echo htmlspecialchars($companyFromDB); ?>" required readonly>
            <div class="invalid-feedback">Company/Organization is required.</div>
          </div>
          <div class="col-md-6">
            <label for="department" class="form-label">Department/Division</label>
            <input type="text" class="form-control" id="department" name="department"
                   value="<?php echo htmlspecialchars($savedData['department'] ?? ''); ?>">
          </div>
        </div>
        <div class="row mb-3">
          <div class="col-md-6">
            <label for="startDate" class="form-label">Internship Start Date</label>
            <input type="date" class="form-control" id="startDate" name="startDate"
                   value="<?php echo htmlspecialchars($savedData['start_date'] ?? ''); ?>" required>
            <div class="invalid-feedback">Please select a start date.</div>
          </div>
          <div class="col-md-6">
            <label for="endDate" class="form-label">Internship End Date</label>
            <input type="date" class="form-control" id="endDate" name="endDate"
                   value="<?php echo htmlspecialchars($savedData['end_date'] ?? ''); ?>" required>
            <div class="invalid-feedback">Please select an end date.</div>
          </div>
        </div>
        <div class="step-buttons d-flex justify-content-end">
          <?php if(!$formDisabled): ?>
            <button type="button" class="btn btn-primary" onclick="nextStep()">Next</button>
          <?php endif; ?>
        </div>
      </div>
      
      <!-- Step 2 -->
      <div class="step" id="step-2">
        <h3>Internship Report (Part 1)</h3>
        <div class="mb-3">
          <label for="executiveSummary" class="form-label">Executive Summary (Max 250 words)</label>
          <textarea class="form-control" id="executiveSummary" name="executiveSummary" rows="4"
            required oninput="checkWordCount(this)"><?php 
            echo htmlspecialchars($savedData['executive_summary'] ?? ''); ?></textarea>
          <div id="executiveSummaryWordCount" class="word-count-message"></div>
          <div class="invalid-feedback">Please provide an executive summary.</div>
        </div>
        <div class="mb-3">
          <label for="introduction" class="form-label">Introduction (Max 250 words)</label>
          <textarea class="form-control" id="introduction" name="introduction" rows="4"
            required oninput="checkWordCount(this)"><?php 
            echo htmlspecialchars($savedData['introduction'] ?? ''); ?></textarea>
          <div id="introductionWordCount" class="word-count-message"></div>
          <div class="invalid-feedback">Please provide an introduction.</div>
        </div>
        <div class="mb-3">
          <label for="institutionProfile" class="form-label">Profile of the Institution (Max 250 words)</label>
          <textarea class="form-control" id="institutionProfile" name="institutionProfile" rows="4"
            required oninput="checkWordCount(this)"><?php 
            echo htmlspecialchars($savedData['institution_profile'] ?? ''); ?></textarea>
          <div id="institutionProfileWordCount" class="word-count-message"></div>
          <div class="invalid-feedback">Please provide the institution profile.</div>
        </div>
        <div class="mb-3">
          <label for="goalsObjectives" class="form-label">Goals and Objectives (Max 250 words)</label>
          <textarea class="form-control" id="goalsObjectives" name="goalsObjectives" rows="4"
            required oninput="checkWordCount(this)"><?php 
            echo htmlspecialchars($savedData['goals_objectives'] ?? ''); ?></textarea>
          <div id="goalsObjectivesWordCount" class="word-count-message"></div>
          <div class="invalid-feedback">Please list your goals and objectives.</div>
        </div>
        <div class="mb-3">
          <label for="coreDuties" class="form-label">Core Duties (Max 250 words)</label>
          <textarea class="form-control" id="coreDuties" name="coreDuties" rows="4"
            required oninput="checkWordCount(this)"><?php 
            echo htmlspecialchars($savedData['core_duties'] ?? ''); ?></textarea>
          <div id="coreDutiesWordCount" class="word-count-message"></div>
          <div class="invalid-feedback">Please provide details on your duties.</div>
        </div>
        <div class="mb-3">
          <label for="organizationalChallenges" class="form-label">Organizational Challenges (Max 250 words)</label>
          <textarea class="form-control" id="organizationalChallenges" name="organizationalChallenges" rows="4"
            required oninput="checkWordCount(this)"><?php 
            echo htmlspecialchars($savedData['organizational_challenges'] ?? ''); ?></textarea>
          <div id="organizationalChallengesWordCount" class="word-count-message"></div>
          <div class="invalid-feedback">Please mention challenges and suggestions.</div>
        </div>
        <div class="mb-3">
          <label for="studentChallenges" class="form-label">Student Challenges (Max 250 words)</label>
          <textarea class="form-control" id="studentChallenges" name="studentChallenges" rows="4"
            required oninput="checkWordCount(this)"><?php 
            echo htmlspecialchars($savedData['student_challenges'] ?? ''); ?></textarea>
          <div id="studentChallengesWordCount" class="word-count-message"></div>
          <div class="invalid-feedback">Please describe challenges encountered.</div>
        </div>
        <div class="step-buttons d-flex justify-content-between">
          <?php if(!$formDisabled): ?>
            <button type="button" class="btn btn-secondary" onclick="prevStep()">Previous</button>
            <button type="button" class="btn btn-primary" onclick="nextStep()">Next</button>
          <?php endif; ?>
        </div>
      </div>
      
      <!-- Step 3 -->
      <div class="step" id="step-3">
        <h3>Internship Report (Part 2)</h3>
        <div class="mb-3">
          <label for="achievements" class="form-label">Achievements (Max 250 words)</label>
          <textarea class="form-control" id="achievements" name="achievements" rows="4"
            required oninput="checkWordCount(this)"><?php 
            echo htmlspecialchars($savedData['achievements'] ?? ''); ?></textarea>
          <div id="achievementsWordCount" class="word-count-message"></div>
          <div class="invalid-feedback">Please list your achievements.</div>
        </div>
        <div class="mb-3">
          <label for="lessonsLearned" class="form-label">Lessons Learned (Max 250 words)</label>
          <textarea class="form-control" id="lessonsLearned" name="lessonsLearned" rows="4"
            required oninput="checkWordCount(this)"><?php 
            echo htmlspecialchars($savedData['lessons_learned'] ?? ''); ?></textarea>
          <div id="lessonsLearnedWordCount" class="word-count-message"></div>
          <div class="invalid-feedback">Please share your insights.</div>
        </div>
        <div class="mb-3">
          <label for="recommendations" class="form-label">Recommendations (Max 250 words)</label>
          <textarea class="form-control" id="recommendations" name="recommendations" rows="4"
            required oninput="checkWordCount(this)"><?php 
            echo htmlspecialchars($savedData['recommendations'] ?? ''); ?></textarea>
          <div id="recommendationsWordCount" class="word-count-message"></div>
          <div class="invalid-feedback">Please provide recommendations.</div>
        </div>
        <div class="mb-3">
          <label for="conclusion" class="form-label">Conclusion (Max 250 words)</label>
          <textarea class="form-control" id="conclusion" name="conclusion" rows="4"
            required oninput="checkWordCount(this)"><?php 
            echo htmlspecialchars($savedData['conclusion'] ?? ''); ?></textarea>
          <div id="conclusionWordCount" class="word-count-message"></div>
          <div class="invalid-feedback">Please provide a conclusion.</div>
        </div>
        <div class="mb-3">
          <label for="references" class="form-label">References</label>
          <textarea class="form-control" id="references" name="references" rows="3"
            placeholder="List references or resources (optional)..."><?php 
            echo htmlspecialchars($savedData['references'] ?? ''); ?></textarea>
        </div>
        <div class="step-buttons d-flex justify-content-between">
          <?php if(!$formDisabled): ?>
            <button type="button" class="btn btn-secondary" onclick="prevStep()">Previous</button>
            <button type="submit" name="save" class="btn btn-warning" formnovalidate>Save and Continue</button>
            <button type="submit" name="submit" class="btn btn-success">Submit Report</button>
          <?php endif; ?>
        </div>
      </div>
    </fieldset>
  </form>
</div>

<!-- Bootstrap JS -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<!-- Include html2canvas and jsPDF libraries for PDF export -->
<script src="https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js"></script>

<script>
  var currentStep = 1;
  var totalSteps = 3;
  
  function showStep(step) {
    document.querySelectorAll('.step').forEach(function(el) {
      el.classList.remove('active');
    });
    document.getElementById('step-' + step).classList.add('active');
    document.getElementById('stepIndicator').innerText = "Step " + step + " of " + totalSteps;
    document.getElementById('progressBar').style.width = (step / totalSteps * 100) + "%";
  }
  
  function nextStep() {
    if (currentStep < totalSteps) {
      currentStep++;
      showStep(currentStep);
    }
  }
  
  function prevStep() {
    if (currentStep > 1) {
      currentStep--;
      showStep(currentStep);
    }
  }
  
  document.addEventListener('DOMContentLoaded', function() {
    showStep(currentStep);
    setTimeout(function(){
      var saveAlert = document.getElementById('saveAlert');
      if (saveAlert) saveAlert.style.display = 'none';
    }, 4000);
    
    // Initialize Bootstrap form validation
    (function () {
      'use strict';
      var forms = document.querySelectorAll('.needs-validation');
      Array.prototype.slice.call(forms).forEach(function (form) {
        form.addEventListener('submit', function (event) {
          // Only prevent submission for 'submit' button when form is invalid
          // Allow 'save' button to bypass validation
          if (!form.checkValidity() && event.submitter && event.submitter.name === 'submit') {
            event.preventDefault();
            event.stopPropagation();
          }
          // Always add validation classes for visual feedback
          form.classList.add('was-validated');
        }, false);
      });
    })();
  });
  
  // Word count limit (250 words)
  const maxWords = 250;
  function checkWordCount(textarea) {
    let words = textarea.value.trim().split(/\s+/).filter(w => w);
    if (words.length > maxWords) {
      textarea.value = words.slice(0, maxWords).join(' ');
      words = words.slice(0, maxWords);
    }
    let msg = document.getElementById(textarea.id + 'WordCount');
    msg.innerText = 'Word count: ' + words.length + ' / ' + maxWords;
  }
  
  // Export functions unchanged
  function exportToWord(elementId, filename = '') {
    var preHtml = "<html xmlns:o='urn:schemas-microsoft-com:office:office' "+
        "xmlns:w='urn:schemas-microsoft-com:office:word' "+
        "xmlns='http://www.w3.org/TR/REC-html40'><head><meta charset='utf-8'></head><body>";
    var postHtml = "</body></html>";
    var content = document.getElementById(elementId).innerHTML;
    var blob = new Blob(['\ufeff', preHtml + content + postHtml], { type: 'application/msword' });
    var url = URL.createObjectURL(blob);
    var link = document.createElement('a');
    link.href = url;
    link.download = filename ? filename + '.doc' : 'document.doc';
    document.body.appendChild(link);
    link.click();
    link.remove();
  }
  
  async function exportToPDF(elementId, filename = '') {
    const { jsPDF } = window.jspdf;
    await html2canvas(document.getElementById(elementId), { scale: 2 }).then(canvas => {
      const imgData = canvas.toDataURL('image/png');
      const pdf = new jsPDF('p','pt','a4');
      const margin = 20;
      const pdfWidth = pdf.internal.pageSize.getWidth() - margin * 2;
      const pdfHeight = pdf.internal.pageSize.getHeight() - margin * 2;
      pdf.addImage(imgData, 'PNG', margin, margin, pdfWidth, pdfHeight);
      pdf.save((filename || 'document') + '.pdf');
    });
  }
</script>

<?php
if ($submittedReport):
    function decryptData($val) { return htmlspecialchars($val); }
    $company    = decryptData($submittedReport['company']);
    $department = decryptData($submittedReport['department']);
    $startDate  = decryptData($submittedReport['start_date']);
    $endDate    = decryptData($submittedReport['end_date']);
    $fields = [
      'executive_summary' => 'Executive Summary',
      'introduction' => 'Introduction',
      'institution_profile' => 'Profile of the Institution',
      'goals_objectives' => 'Goals & Objectives',
      'core_duties' => 'Core Duties',
      'organizational_challenges' => 'Organizational Challenges',
      'student_challenges' => 'Student Challenges',
      'achievements' => 'Achievements',
      'lessons_learned' => 'Lessons Learned',
      'recommendations' => 'Recommendations',
      'conclusion' => 'Conclusion',
      'references' => 'References'
    ];
?>
<div class="modal fade" id="submittedReportModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-scrollable" style="max-width:800px">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title">Submitted Internship Report</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body">
        <p><strong>Company:</strong> <?= $company ?></p>
        <p><strong>Department:</strong> <?= $department ?></p>
        <p><strong>Period:</strong> <?= $startDate ?> – <?= $endDate ?></p>
        <hr>
        <?php foreach ($fields as $key => $label): ?>
          <h6><?= $label ?></h6>
          <p><?= nl2br(decryptData($submittedReport[$key] ?? '')) ?></p>
        <?php endforeach; ?>
      </div>
      <div class="modal-footer">
        <button class="btn btn-primary" onclick="exportToWord('submittedReportModal','Internship_Report')">Export to Word</button>
        <button class="btn btn-secondary" onclick="exportToPDF('submittedReportModal','Internship_Report')">Export to PDF</button>
      </div>
    </div>
  </div>
</div>
<?php
endif;
include '../includes/footer.php';
ob_end_flush();
?>
