<?php
session_start();
if (!isset($_SESSION['lecturer_id'])) {
    header('Location: login.php');
    exit;
}

include '../config/database.php';
require('../vendor/fpdf/fpdf.php'); // Adjust the path to FPDF if needed

$lecturerId = $_SESSION['lecturer_id'];

// Check if mapping has been submitted
$stmt = $pdo->prepare('SELECT mapping_submitted, name, staff_id, car_number, department FROM lecturer_login WHERE id = ?');
$stmt->execute([$lecturerId]);
$lecturer = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$lecturer) {
    die('Error: Lecturer not found.');
}

// Initialize lecturer details if they are not set
$staff_id = $lecturer['staff_id'] ?? 'N/A';
$car_number = $lecturer['car_number'] ?? 'N/A';
$department = $lecturer['department'] ?? 'N/A';

if ($lecturer['mapping_submitted']) {
    header('Location: mapping.php');
    exit;
}

if (isset($_POST['date_of_supervision'])) {
    $date_of_supervision = $_POST['date_of_supervision'];
    $student_name = $_POST['student_name'];
    $index_number = $_POST['index_number'];
    $from_location = $_POST['from_location'];
    $from_location_other = $_POST['from_location_other'];
    $to_location = $_POST['to_location'];
    $to_location_other = $_POST['to_location_other'];
    $distance = $_POST['distance'];
    $nights = $_POST['nights'];

    // Save ongoing mapping to supervision_mapping table
    // Delete existing data if any
    $stmt = $pdo->prepare('DELETE FROM supervision_mapping WHERE lecturer_id = ?');
    $stmt->execute([$lecturerId]);

    // Insert new data into supervision_mapping table
    for ($i = 0; $i < count($date_of_supervision); $i++) {
        $fromLocation = ($from_location[$i] === 'Other') ? (!empty($from_location_other[$i]) ? htmlspecialchars($from_location_other[$i]) : null) : htmlspecialchars($from_location[$i]);
        $toLocation = ($to_location[$i] === 'Other') ? (!empty($to_location_other[$i]) ? htmlspecialchars($to_location_other[$i]) : null) : htmlspecialchars($to_location[$i]);
        $distanceValue = !empty($distance[$i]) ? htmlspecialchars($distance[$i]) : null;
        $nightValue = (!empty($nights[$i]) || $nights[$i] === "0") ? htmlspecialchars($nights[$i]) : 0;

        $stmt = $pdo->prepare('INSERT INTO supervision_mapping (lecturer_id, date_of_supervision, student_name, index_number, from_location, to_location, distance, nights) VALUES (?, ?, ?, ?, ?, ?, ?, ?)');
        $stmt->execute([
            $lecturerId,
            htmlspecialchars($date_of_supervision[$i]),
            htmlspecialchars($student_name[$i]),
            htmlspecialchars($index_number[$i]),
            $fromLocation,
            $toLocation,
            $distanceValue,
            $nightValue
        ]);
    }

    // If submit button was pressed, finalize the mapping
    if (isset($_POST['submit'])) {
        // Delete existing data in mapping table if any
        $stmt = $pdo->prepare('DELETE FROM mapping WHERE lecturer_id = ?');
        $stmt->execute([$lecturerId]);

        // Insert finalized data into mapping table
        for ($i = 0; $i < count($date_of_supervision); $i++) {
            $studentId = ($student_name[$i] === 'Other' || empty($student_name[$i])) ? null : htmlspecialchars($student_name[$i]);
            $fromLocation = ($from_location[$i] === 'Other') ? (!empty($from_location_other[$i]) ? htmlspecialchars($from_location_other[$i]) : null) : htmlspecialchars($from_location[$i]);
            $toLocation = ($to_location[$i] === 'Other') ? (!empty($to_location_other[$i]) ? htmlspecialchars($to_location_other[$i]) : null) : htmlspecialchars($to_location[$i]);
            $distanceValue = !empty($distance[$i]) ? htmlspecialchars($distance[$i]) : null;
            $nightValue = (!empty($nights[$i]) || $nights[$i] === "0") ? htmlspecialchars($nights[$i]) : 0;

            $stmt = $pdo->prepare('INSERT INTO mapping (lecturer_id, date_of_supervision, student_id, from_location, to_location, distance, nights) VALUES (?, ?, ?, ?, ?, ?, ?)');
            $stmt->execute([
                $lecturerId,
                htmlspecialchars($date_of_supervision[$i]),
                $studentId,
                $fromLocation,
                $toLocation,
                $distanceValue,
                $nightValue
            ]);
        }

        // Generate PDF
        $stmt = $pdo->prepare('
            SELECT sm.date_of_supervision, sm.student_name, sm.index_number, sm.from_location, sm.to_location, sm.distance, sm.nights
            FROM supervision_mapping sm
            WHERE sm.lecturer_id = ?
        ');
        $stmt->execute([$lecturerId]);
        $supervisionData = $stmt->fetchAll(PDO::FETCH_ASSOC);

        $totalDistance = 0;
        $totalNights = 0;
        foreach ($supervisionData as $data) {
            $totalDistance += $data['distance'];
            $totalNights += $data['nights'];
        }

        class PDF extends FPDF {
            function Header() {
                $this->SetFont('Arial', 'B', 12);
                $this->Cell(0, 5, 'UNIVERSITY OF EDUCATION, WINNEBA', 0, 1, 'C');
                $this->Cell(0, 5, 'SCHOOL OF BUSINESS', 0, 1, 'C');
                $this->Cell(0, 5, 'INTERNSHIP CLAIM FOR THE 2024/25 ACADEMIC YEAR', 0, 1, 'C');
                $this->Ln(5);
            }

            function LecturerDetails($lecturer) {
                $this->SetFont('Arial', '', 10);
                $this->Cell(0, 5, 'Name: ' . htmlspecialchars($lecturer['name']), 0, 1);
                $this->Cell(0, 5, 'Staff No.: ' . htmlspecialchars($lecturer['staff_id']), 0, 1);
                $this->Cell(0, 5, 'Car Reg. No.: ' . htmlspecialchars($lecturer['car_number']), 0, 1);
                $this->Cell(0, 5, 'Department: ' . htmlspecialchars($lecturer['department']), 0, 1);
                $this->Ln(5);
            }

            function SupervisionTable($header, $data, $totalDistance, $totalNights) {
                $this->SetFont('Arial', 'B', 10);
                $widths = [30, 50, 30, 40, 40, 30, 20];
                for ($i = 0; $i < count($header); $i++) {
                    $this->Cell($widths[$i], 7, htmlspecialchars($header[$i]), 1, 0, 'C');
                }
                $this->Ln();
                $this->SetFont('Arial', '', 10);

                foreach ($data as $row) {
                    $maxHeight = $this->GetMultiCellHeight($row, $widths);

                    $x = $this->GetX();
                    $y = $this->GetY();

                    $this->MultiCell($widths[0], $maxHeight / $this->NbLines($widths[0], htmlspecialchars($row['date_of_supervision'])), htmlspecialchars($row['date_of_supervision']), 1, 'L');
                    $this->SetXY($x + $widths[0], $y);
                    $this->MultiCell($widths[1], $maxHeight / $this->NbLines($widths[1], htmlspecialchars($row['student_name'])), htmlspecialchars($row['student_name']), 1, 'L');
                    $this->SetXY($x + $widths[0] + $widths[1], $y);
                    $this->MultiCell($widths[2], $maxHeight / $this->NbLines($widths[2], htmlspecialchars($row['index_number'])), htmlspecialchars($row['index_number']), 1, 'L');
                    $this->SetXY($x + $widths[0] + $widths[1] + $widths[2], $y);
                    $this->MultiCell($widths[3], $maxHeight / $this->NbLines($widths[3], htmlspecialchars($row['from_location'])), htmlspecialchars($row['from_location']), 1, 'L');
                    $this->SetXY($x + $widths[0] + $widths[1] + $widths[2] + $widths[3], $y);
                    $this->MultiCell($widths[4], $maxHeight / $this->NbLines($widths[4], htmlspecialchars($row['to_location'])), htmlspecialchars($row['to_location']), 1, 'L');
                    $this->SetXY($x + $widths[0] + $widths[1] + $widths[2] + $widths[3] + $widths[4], $y);
                    $this->MultiCell($widths[5], $maxHeight / $this->NbLines($widths[5], htmlspecialchars(number_format($row['distance'], 2))), htmlspecialchars(number_format($row['distance'], 2)), 1, 'L');
                    $this->SetXY($x + $widths[0] + $widths[1] + $widths[2] + $widths[3] + $widths[4] + $widths[5], $y);
                    $this->MultiCell($widths[6], $maxHeight / $this->NbLines($widths[6], htmlspecialchars($row['nights'])), htmlspecialchars($row['nights']), 1, 'L');

                    $this->SetXY($x, $y + $maxHeight);
                }

                // Add totals
                $this->Cell(array_sum(array_slice($widths, 0, 5)), 6, 'Total', 1);
                $this->Cell($widths[5], 6, htmlspecialchars(number_format($totalDistance, 2)), 1);
                $this->Cell($widths[6], 6, htmlspecialchars($totalNights), 1);
                $this->Ln(10); // Space before the signature
                $this->Cell(0, 10, 'Signature: ........................................', 0, 1, 'L');
                
                // Add the current date
                $this->Cell(0, 10, 'Date: ' . htmlspecialchars(date('d-M-Y')), 0, 1, 'L');
            }

            function GetMultiCellHeight($row, $widths) {
                $nbLines = max(
                    $this->NbLines($widths[0], htmlspecialchars($row['date_of_supervision'])),
                    $this->NbLines($widths[1], htmlspecialchars($row['student_name'])),
                    $this->NbLines($widths[2], htmlspecialchars($row['index_number'])),
                    $this->NbLines($widths[3], htmlspecialchars($row['from_location'])),
                    $this->NbLines($widths[4], htmlspecialchars($row['to_location'])),
                    $this->NbLines($widths[5], htmlspecialchars(number_format($row['distance'], 2))),
                    $this->NbLines($widths[6], htmlspecialchars($row['nights']))
                );

                return 6 * $nbLines;
            }

            function NbLines($w, $txt) {
                // Calculates the number of lines a MultiCell of width w will take
                $cw = &$this->CurrentFont['cw'];
                if ($w == 0)
                    $w = $this->w - $this->rMargin - $this->x;
                $wmax = ($w - 2 * $this->cMargin) * 1000 / $this->FontSize;
                $s = str_replace("\r", '', $txt);
                $nb = strlen($s);
                if ($nb > 0 && $s[$nb - 1] == "\n")
                    $nb--;
                $sep = -1;
                $i = 0;
                $j = 0;
                $l = 0;
                $nl = 1;
                while ($i < $nb) {
                    $c = $s[$i];
                    if ($c == "\n") {
                        $i++;
                        $sep = -1;
                        $j = $i;
                        $l = 0;
                        $nl++;
                        continue;
                    }
                    if ($c == ' ')
                        $sep = $i;
                    $l += $cw[$c];
                    if ($l > $wmax) {
                        if ($sep == -1) {
                            if ($i == $j)
                                $i++;
                        } else
                            $i = $sep + 1;
                        $sep = -1;
                        $j = $i;
                        $l = 0;
                        $nl++;
                    } else
                        $i++;
                }
                return $nl;
            }
        }

        $pdf = new PDF('L', 'mm', 'A4');
        $pdf->AddPage();
        $pdf->LecturerDetails($lecturer);
        $pdf->SupervisionTable(['Date', 'Student', 'Index', 'From', 'To', 'Distance (KM)', 'Nights'], $supervisionData, $totalDistance, $totalNights);

        // Ensure the directory exists
        $directoryPath = '../admin/mapping/';
        if (!is_dir($directoryPath)) {
            mkdir($directoryPath, 0777, true);
        }

        // Create a filename based on the lecturer's name and current timestamp
        $filename = $directoryPath . $lecturerId . '_mapping_' . date('Ymd_His') . '.pdf';

        // Generate the PDF and save it to the file
        $pdf->Output('F', $filename);

        // Update lecturer_login to mark mapping as submitted and store the timestamp
        $stmt = $pdo->prepare('UPDATE lecturer_login SET mapping_submitted = 1, mapping_timestamp = ? WHERE id = ?');
        $stmt->execute([date('Y-m-d H:i:s'), $lecturerId]);

        // Notify admin (if needed)

        // Redirect to a confirmation page or back to the mapping page
        header('Location: mapping.php');
        exit;
    }
}
?>
