<?php
session_start();
ob_start(); // Start output buffering

if (!isset($_SESSION['lecturer_id'])) {
    header('Location: login.php');
    exit;
}

include 'lect_header.php';
include '../config/database.php';

$statusMessage = '';
$lecturerLoginId = $_SESSION['lecturer_id'];
$centralRegion = 'Central Region';

// Ensure Central Region is always in the preferences
$stmt = $pdo->prepare('SELECT COUNT(*) FROM lecturer_preferences WHERE lecturer_id = ? AND preferred_region = ?');
$stmt->execute([$lecturerLoginId, $centralRegion]);
$regionExists = $stmt->fetchColumn() > 0;

if (!$regionExists) {
    $stmt = $pdo->prepare('INSERT INTO lecturer_preferences (lecturer_id, preferred_region, preferred_districts) VALUES (?, ?, ?)');
    $stmt->execute([$lecturerLoginId, $centralRegion, '']);
}

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $preferredRegion = $_POST['preferredRegion'];

    // Check if the region is already in the preferences
    $stmt = $pdo->prepare('SELECT COUNT(*) FROM lecturer_preferences WHERE lecturer_id = ? AND preferred_region = ?');
    $stmt->execute([$lecturerLoginId, $preferredRegion]);
    $regionExists = $stmt->fetchColumn() > 0;

    if ($regionExists) {
        $statusMessage = 'This region is already in your preferences.';
    } else {
        try {
            // Insert new region preference
            $stmt = $pdo->prepare('INSERT INTO lecturer_preferences (lecturer_id, preferred_region, preferred_districts) VALUES (?, ?, ?)');
            $stmt->execute([$lecturerLoginId, $preferredRegion, '']);

            $statusMessage = 'Preference saved successfully.';

            // Clear output buffer and send header
            ob_end_clean();
            header('Location: region.php');
            exit;
        } catch (PDOException $e) {
            $statusMessage = 'Error saving preference: ' . $e->getMessage();
        }
    }
}

// Now output content safely
ob_end_flush();

// Fetch regions from the database
$stmt = $pdo->prepare('SELECT DISTINCT region FROM districts');
$stmt->execute();
$regions = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch preferred regions if any
$stmt = $pdo->prepare('SELECT id, preferred_region FROM lecturer_preferences WHERE lecturer_id = ?');
$stmt->execute([$lecturerLoginId]);
$preferences = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch assigned districts for the lecturer
$stmt = $pdo->prepare('SELECT region, GROUP_CONCAT(district SEPARATOR ", ") AS districts FROM lecturer_assigned_districts WHERE lecturer_id = ? GROUP BY region');
$stmt->execute([$lecturerLoginId]);
$assignedRegions = $stmt->fetchAll(PDO::FETCH_ASSOC);

$preferredRegions = array_column($preferences, 'preferred_region');
?>

<div class="container d-flex justify-content-center align-items-center" style="min-height: 80vh;">
    <div class="card p-4" style="width: 100%; max-width: 600px;">
        <h2>Select Preferred Region</h2>

        <?php if ($statusMessage): ?>
            <div class="alert alert-info" id="statusMessage">
                <?php echo $statusMessage; ?>
            </div>
        <?php endif; ?>

        <form method="POST" action="region.php">
            <div class="mb-3">
                <label for="preferredRegion" class="form-label">Preferred Region</label>
                <select class="form-control" id="preferredRegion" name="preferredRegion" required>
                    <option value="">Select Region</option>
                    <?php foreach ($regions as $region): ?>
                        <option value="<?php echo htmlspecialchars($region['region']); ?>" <?php echo in_array($region['region'], $preferredRegions) ? 'disabled' : ''; ?>>
                            <?php echo htmlspecialchars($region['region']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
                <input type="hidden" name="centralRegion" value="Central Region">
            </div>
            <button type="submit" class="btn btn-outline-custom">Add Preference</button>
        </form>

        <?php if (!empty($preferredRegions)): ?>
            <h3 class="mt-4">Current Preferences</h3>
            <table class="table table-bordered">
                <thead>
                    <tr>
                        <th>Preferred Region</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($preferences as $preference): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($preference['preferred_region']); ?></td>
                            <td>
                                <?php 
                                    $assignedDistricts = array_filter($assignedRegions, function($ar) use ($preference) {
                                        return $ar['region'] === $preference['preferred_region'];
                                    });
                                    if (!empty($assignedDistricts)): 
                                ?>
                                    <button class="btn btn-outline-view-assigned btn-sm view-assigned" data-region="<?php echo htmlspecialchars($preference['preferred_region']); ?>" data-districts="<?php echo htmlspecialchars(current($assignedDistricts)['districts']); ?>">View Assigned Districts</button>
                                <?php else: ?>
                                    <?php if ($preference['preferred_region'] !== 'Central Region'): ?>
                                        <a href="delete_preference.php?id=<?php echo $preference['id']; ?>" class="btn btn-outline-danger btn-sm" onclick="return confirm('Are you sure you want to delete this preference?')">Delete</a>
                                    <?php endif; ?>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>
</div>

<!-- Assigned Districts Modal -->
<div class="modal fade" id="assignedDistrictsModal" tabindex="-1" aria-labelledby="assignedDistrictsModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="assignedDistrictsModalLabel">Assigned Districts</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <h3 id="modalRegionTitle"></h3>
                <ul id="districtsList"></ul>
            </div>
        </div>
    </div>
</div>

<?php include 'lect_footer.php'; ?>

<script>
document.addEventListener('DOMContentLoaded', function () {
    // Hide alert after 5 seconds
    const statusMessage = document.getElementById('statusMessage');
    if (statusMessage) {
        setTimeout(() => {
            statusMessage.style.display = 'none';
        }, 5000);
    }

    // View assigned districts
    document.querySelectorAll('.view-assigned').forEach(button => {
        button.addEventListener('click', function() {
            const region = this.getAttribute('data-region');
            const districts = this.getAttribute('data-districts').split(', ');

            document.getElementById('modalRegionTitle').textContent = `Assigned Districts for ${region}`;
            const districtsList = document.getElementById('districtsList');
            districtsList.innerHTML = '';
            districts.forEach(district => {
                const li = document.createElement('li');
                li.textContent = district;
                districtsList.appendChild(li);
            });

            const assignedDistrictsModal = new bootstrap.Modal(document.getElementById('assignedDistrictsModal'));
            assignedDistrictsModal.show();
        });
    });
});
</script>
