<?php
// Start session and verify admin login
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

require '../config/database.php'; // <-- Adjust path if needed

// =====================================================
// Step 1: Process an unassignment (Return) action if requested
// =====================================================
if (isset($_GET['return_student_id'])) {
    // Sanitize/validate the student id as needed
    $studentId = $_GET['return_student_id'];

    // Unassign the lecturer by setting lecturer_id to NULL
    $stmt = $pdo->prepare("UPDATE student_details SET lecturer_id = NULL WHERE id = ?");
    $stmt->execute([$studentId]);

    // Build the redirect URL without return_student_id
    $queryParams = $_GET;
    unset($queryParams['return_student_id']);
    $redirectUrl = $_SERVER['PHP_SELF'] . (!empty($queryParams) ? '?' . http_build_query($queryParams) : '');
    header("Location: " . $redirectUrl);
    exit;
}

// =====================================================
// Function to fetch summary
// =====================================================
function fetchSummary($pdo) {
    // Total submitted
    $stmt = $pdo->query("SELECT COUNT(*) AS total FROM student_details WHERE status = 'submitted'");
    $total_submitted = $stmt->fetch(PDO::FETCH_ASSOC)['total'];

    // Total assigned
    $stmt = $pdo->query("SELECT COUNT(*) AS total FROM student_details WHERE status = 'submitted' AND lecturer_id IS NOT NULL");
    $total_assigned = $stmt->fetch(PDO::FETCH_ASSOC)['total'];

    return [
        'total_submitted'  => $total_submitted,
        'total_assigned'   => $total_assigned,
        'total_unassigned' => $total_submitted - $total_assigned,
    ];
}

// =====================================================
// Function to fetch lecturers data, now including TELEPHONE
// =====================================================
function fetchLecturersData($pdo) {
    $sql = "
        SELECT 
            ll.id AS lecturer_id,
            ll.name AS lecturer_name,
            lad.region AS assigned_region,
            lad.district AS assigned_district,
            sd.id AS student_id,
            sd.first_name,
            sd.surname,
            sd.organisation_name,
            sd.location,
            sd.telephone AS student_telephone
        FROM lecturer_login ll
        LEFT JOIN lecturer_assigned_districts lad 
            ON ll.id = lad.lecturer_id
        LEFT JOIN student_details sd 
            ON sd.lecturer_id = ll.id
           AND sd.district = lad.district
           AND sd.status = 'submitted'
        ORDER BY ll.name, lad.region, lad.district, sd.surname
    ";
    $stmt = $pdo->query($sql);
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $lecturerData = [];
    foreach ($rows as $r) {
        $lid    = $r['lecturer_id'];
        $name   = ucwords(strtolower($r['lecturer_name']));
        $region = ucwords(strtolower($r['assigned_region']));
        $dist   = ucwords(strtolower($r['assigned_district']));

        if (!isset($lecturerData[$lid])) {
            $lecturerData[$lid] = [
                'name'           => $name,
                'regions'        => [],
                'total_students' => 0,
            ];
        }
        if (!isset($lecturerData[$lid]['regions'][$region])) {
            $lecturerData[$lid]['regions'][$region] = [];
        }
        if (!isset($lecturerData[$lid]['regions'][$region][$dist])) {
            $lecturerData[$lid]['regions'][$region][$dist] = [];
        }
        if ($r['first_name']) {
            $lecturerData[$lid]['regions'][$region][$dist][] = [
                'id'           => $r['student_id'],
                'name'         => ucwords(strtolower($r['first_name'] . ' ' . $r['surname'])),
                'organisation' => $r['organisation_name'],
                'location'     => $r['location'],
                'telephone'    => $r['student_telephone'],
            ];
            $lecturerData[$lid]['total_students']++;
        }
    }
    return $lecturerData;
}

// Fetch summary and lecturers data
$summary         = fetchSummary($pdo);
$lecturerData    = fetchLecturersData($pdo);

// Get filters
$selectedLecturer = $_GET['lecturer'] ?? '';
$selectedRegion   = $_GET['region']   ?? '';

// Attempt to find the chosen lecturer in our data
$chosenLecturer = (!empty($selectedLecturer) && isset($lecturerData[$selectedLecturer]))
                 ? $lecturerData[$selectedLecturer]
                 : null;
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <title>View Assigned Students</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.1.1/css/all.min.css" rel="stylesheet">
  <style>
    @media print { .d-print-none { display: none !important; } }
  </style>
  <script>
    function printResults() {
      const div = document.getElementById('printableArea');
      const w = window.open('', '', 'width=800,height=600');
      w.document.write(`
        <html>
          <head>
            <title>Print Results</title>
            <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
            <style>table{width:100%;border-collapse:collapse}th,td{border:1px solid #000;padding:8px}</style>
          </head>
          <body onload="window.print();window.close()">
            ${div.innerHTML}
          </body>
        </html>
      `);
      w.document.close();
    }
  </script>
</head>
<body>

<?php include 'admin_header.php'; ?>

<div class="container mt-4">
  <h1 class="mb-4">Assigned Students Summary</h1>

  <div class="mb-4">
    <a href="assign_students.php" class="btn btn-secondary d-print-none">
      <i class="fas fa-arrow-left"></i> Back to Assign Students
    </a>
  </div>

  <div class="mb-4 d-print-none">
    <a href="export_assigned_students.php?type=summary" class="btn btn-info">
      <i class="fas fa-file-excel"></i> Export Summary Only
    </a>
    <a href="export_assigned_students.php?type=all" class="btn btn-success">
      <i class="fas fa-file-excel"></i> Export All Details
    </a>
    <form action="export_assigned_students.php" method="GET" class="d-inline-block">
      <input type="hidden" name="type" value="lecturer">
      <select name="lecturer" class="form-select d-inline-block w-auto" required>
        <option value="">Select Lecturer</option>
        <?php foreach ($lecturerData as $lid => $l): ?>
          <option value="<?= $lid ?>"><?= $l['name'] ?></option>
        <?php endforeach; ?>
      </select>
      <button class="btn btn-primary">Export Lecturer Sheet</button>
    </form>
  </div>

  <div class="card mb-4 d-print-none">
    <div class="card-body">
      <h4>Summary</h4>
      <p><strong>Total Submitted Students:</strong> <?= $summary['total_submitted'] ?></p>
      <p><strong>Total Assigned Students:</strong> <?= $summary['total_assigned'] ?></p>
      <p><strong>Total Unassigned Students:</strong> <?= $summary['total_unassigned'] ?></p>
    </div>
  </div>

  <div class="card mb-4 d-print-none">
    <div class="card-body">
      <h4>Filter</h4>
      <form method="GET">
        <div class="row">
          <div class="col-md-6 mb-3">
            <label class="form-label">Select Lecturer</label>
            <select name="lecturer" class="form-select">
              <option value="">-- Choose a Lecturer --</option>
              <?php foreach ($lecturerData as $lid => $l): ?>
                <option value="<?= $lid ?>" <?= $selectedLecturer == $lid ? 'selected' : '' ?>>
                  <?= $l['name'] ?>
                </option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-6 mb-3">
            <label class="form-label">Select Region</label>
            <select name="region" class="form-select">
              <option value="">-- All Regions --</option>
              <?php
                $regions = [];
                foreach ($lecturerData as $l) {
                  foreach (array_keys($l['regions']) as $r) {
                    $regions[$r] = $r;
                  }
                }
              ?>
              <?php foreach ($regions as $r): ?>
                <option value="<?= $r ?>" <?= $selectedRegion == $r ? 'selected' : '' ?>>
                  <?= $r ?>
                </option>
              <?php endforeach; ?>
            </select>
          </div>
        </div>
        <button class="btn btn-primary">Filter</button>
      </form>
    </div>
  </div>

  <div id="printableArea" class="card">
    <div class="card-body">
      <h4>Results</h4>

      <?php if (!$selectedLecturer): ?>
        <p class="text-muted">No lecturer selected. Please select a lecturer above.</p>

      <?php elseif (!$chosenLecturer): ?>
        <p class="text-danger">No data found for the selected lecturer.</p>

      <?php else: ?>
        <div class="d-print-none mb-3">
          <button onclick="printResults()" class="btn btn-outline-secondary">
            <i class="fas fa-print"></i> Print
          </button>
        </div>

        <!-- Download Filtered Students -->
        <form action="export_filtered_students.php" method="POST" class="d-print-none mb-3">
          <input type="hidden" name="lecturer_id" value="<?= htmlspecialchars($selectedLecturer) ?>">
          <input type="hidden" name="region"       value="<?= htmlspecialchars($selectedRegion) ?>">
          <button class="btn btn-success">
            <i class="fas fa-download"></i> Download Filtered Students
          </button>
        </form>

        <h5 class="mt-4">
          <?= $chosenLecturer['name'] ?> (<?= $chosenLecturer['total_students'] ?> students assigned)
        </h5>

        <table class="table table-striped table-bordered mt-3">
          <thead>
            <tr>
              <th>Region</th>
              <th>District</th>
              <th>Student Name</th>
              <th>Organisation</th>
              <th>Location</th>
              <th>Telephone</th>
              <th class="d-print-none">Action</th>
            </tr>
          </thead>
          <tbody>
            <?php foreach ($chosenLecturer['regions'] as $reg => $dists): ?>
              <?php if ($selectedRegion && $selectedRegion != $reg) continue; ?>
              <?php foreach ($dists as $dist => $students): ?>
                <?php foreach ($students as $st): ?>
                  <tr>
                    <td><?= $reg ?></td>
                    <td><?= $dist ?></td>
                    <td><?= $st['name'] ?></td>
                    <td><?= $st['organisation'] ?></td>
                    <td><?= $st['location'] ?></td>
                    <td><?= $st['telephone'] ?></td>
                    <td class="d-print-none">
                      <?php
                        $p = $_GET;
                        $p['return_student_id'] = $st['id'];
                        $url = $_SERVER['PHP_SELF'] . '?' . http_build_query($p);
                      ?>
                      <a href="<?= $url ?>" class="btn btn-warning btn-sm"
                         onclick="return confirm('Unassign this student?');">
                        Return
                      </a>
                    </td>
                  </tr>
                <?php endforeach; ?>
              <?php endforeach; ?>
            <?php endforeach; ?>
          </tbody>
        </table>
      <?php endif; ?>
    </div>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
