<?php
// Include your database connection
include '../config/database.php';

// Include the header
include('pi_header.php');

/**
 * Safely decrypts data if it is valid base64-encoded and contains an IV.
 * Otherwise, returns the original data (plaintext).
 *
 * @param string $data The data from the database (possibly encrypted).
 * @return string Decrypted data if validly encrypted, or original data if not.
 */
function decryptData($data) {
    if (empty($data)) {
        return $data;
    }
    $cipher = 'AES-256-CBC';
    $iv_length = openssl_cipher_iv_length($cipher);
    $encryption_key = 'your-secret-key'; // Replace with your actual key

    $decoded = base64_decode($data, true);
    if ($decoded === false) {
        return $data;
    }
    if (strlen($decoded) <= $iv_length) {
        return $data;
    }
    $iv = substr($decoded, 0, $iv_length);
    $ciphertext = substr($decoded, $iv_length);
    $decrypted = openssl_decrypt($ciphertext, $cipher, $encryption_key, 0, $iv);
    return ($decrypted !== false) ? $decrypted : $data;
}

// Fetch data from the database
try {
    $sql = "SELECT 
                ir.id,
                ir.start_date,
                ir.end_date,
                ir.department,
                ir.executive_summary,
                ir.introduction,
                ir.institution_profile,
                ir.goals_objectives,
                ir.core_duties,
                ir.organizational_challenges,
                ir.student_challenges,
                ir.achievements,
                ir.lessons_learned,
                ir.recommendations,
                ir.conclusion,
                ir.references,
                ir.appendix,
                ir.submission_date,
                ir.status,
                u.index_number,
                sd.surname,
                sd.first_name,
                sd.other_names,
                sd.organisation_name AS company,
                sd.course AS program,
                u.group AS user_group
            FROM internship_reports ir
            INNER JOIN users u ON ir.user_id = u.id
            INNER JOIN student_details sd ON ir.user_id = sd.user_id";
    $stmt = $pdo->query($sql);
    $reports = $stmt->fetchAll();
} catch (PDOException $e) {
    die("Error fetching data: " . htmlspecialchars($e->getMessage(), ENT_QUOTES, 'UTF-8'));
}

// Get unique groups from reports
$reportGroups = [];
foreach ($reports as $report) {
    $group = $report['user_group'];
    if (!in_array($group, $reportGroups)) {
        $reportGroups[] = $group;
    }
}

// Count total students in both groups
$stmt = $pdo->prepare("SELECT COUNT(*) FROM users WHERE `group` IN (?, ?)");
$stmt->execute(['Industrial Internship', 'Teaching Practice']);
$totalStudentsBothGroups = $stmt->fetchColumn();

// Count students in each group
$stmt = $pdo->prepare("SELECT COUNT(*) FROM users WHERE `group` = ?");
$stmt->execute(['Industrial Internship']);
$totalStudentsInternship = $stmt->fetchColumn();
$stmt->execute(['Teaching Practice']);
$totalStudentsTeaching = $stmt->fetchColumn();

// Get unique programs for filter
$programs = [];
foreach ($reports as $report) {
    $program = decryptData($report['program']);
    if (!in_array($program, $programs)) {
        $programs[] = $program;
    }
}
sort($programs);

// Count reports by program
$programCounts = [];
foreach ($programs as $program) {
    $programCounts[$program] = 0;
}
foreach ($reports as $report) {
    $program = decryptData($report['program']);
    $programCounts[$program]++;
}

// Fetch teaching reports
try {
    $sqlTeaching = "SELECT tr.*, u.index_number, sd.surname, sd.first_name, sd.other_names
                    FROM teaching_reports tr
                    INNER JOIN users u ON tr.user_id = u.id
                    INNER JOIN student_details sd ON tr.user_id = sd.user_id";
    $stmtTeaching = $pdo->query($sqlTeaching);
    $teachingReports = $stmtTeaching->fetchAll();
} catch (PDOException $e) {
    die("Error fetching teaching reports: " . htmlspecialchars($e->getMessage(), ENT_QUOTES, 'UTF-8'));
}

// Combine both report types into a single array with a 'type' field
$allReports = [];
foreach ($reports as $report) {
    $allReports[] = [
        'type' => 'internship',
        'data' => $report
    ];
}
foreach ($teachingReports as $treport) {
    $allReports[] = [
        'type' => 'teaching',
        'data' => $treport
    ];
}
?>

<div class="container mt-4">
  <h1 class="text-center mb-4">All Reports</h1>
  
  <!-- Report Counts Summary -->
  <div class="row mb-4">
    <div class="col-md-4">
      <div class="card text-center bg-primary text-white">
        <div class="card-body">
          <h5 class="card-title">Total Reports</h5>
          <h3 class="card-text" id="totalCount"><?php echo count($allReports) . ' / ' . $totalStudentsBothGroups; ?></h3>
        </div>
      </div>
    </div>
    <div class="col-md-4">
      <div class="card text-center bg-success text-white">
        <div class="card-body">
          <h5 class="card-title">Internship Reports</h5>
          <h3 class="card-text" id="internshipCount"><?php echo count($reports) . ' / ' . $totalStudentsInternship; ?></h3>
        </div>
      </div>
    </div>
    <div class="col-md-4">
      <div class="card text-center bg-info text-white">
        <div class="card-body">
          <h5 class="card-title">Teaching Reports</h5>
          <h3 class="card-text" id="teachingCount"><?php echo count($teachingReports) . ' / ' . $totalStudentsTeaching; ?></h3>
        </div>
      </div>
    </div>
  </div>
  
  <!-- Program Filter -->
  <div class="row mb-3">
    <div class="col-md-6">
      <label for="programFilter" class="form-label"><strong>Filter by Program:</strong></label>
      <select class="form-select" id="programFilter">
        <option value="">All Programs</option>
        <?php foreach ($programs as $program): ?>
          <option value="<?php echo htmlspecialchars($program); ?>">
            <?php echo htmlspecialchars($program); ?> (<?php echo $programCounts[$program]; ?>)
          </option>
        <?php endforeach; ?>
      </select>
    </div>
  </div>
  
  <ul class="nav nav-tabs mb-3" id="reportTabs" role="tablist">
    <li class="nav-item" role="presentation">
      <button class="nav-link active" id="all-tab" data-type="all" type="button" role="tab">All (<?php echo count($allReports); ?>)</button>
    </li>
    <li class="nav-item" role="presentation">
      <button class="nav-link" id="internship-tab" data-type="internship" type="button" role="tab">Post Internship Reports (<?php echo count($reports); ?>)</button>
    </li>
    <li class="nav-item" role="presentation">
      <button class="nav-link" id="teaching-tab" data-type="teaching" type="button" role="tab">Teaching Reports (<?php echo count($teachingReports); ?>)</button>
    </li>
  </ul>
  <div class="table-responsive">
    <table class="table table-striped table-bordered" id="reportsTable">
      <thead class="table-dark">
        <tr>
          <th>Type</th>
          <th>Student Name &amp; Index Number</th>
          <th>Organization/Status</th>
          <th>Department/Submitted At</th>
          <th>Actions</th>
        </tr>
      </thead>
      <tbody>
        <?php foreach ($allReports as $item):
          if ($item['type'] === 'internship') {
            $report = $item['data'];
            $surname          = decryptData($report['surname']);
            $firstName        = decryptData($report['first_name']);
            $otherNames       = decryptData($report['other_names']);
            $indexNumber      = decryptData($report['index_number']);
            $organization     = decryptData($report['company']);
            $department       = decryptData($report['department']);
            $program          = decryptData($report['program']);
            $fullName = trim($surname . ' ' . $firstName . ' ' . $otherNames);
            $startDate        = $report['start_date'];
            $endDate          = $report['end_date'];
            $submissionDate   = $report['submission_date'];
            $status           = decryptData($report['status']);
        ?>
        <tr data-type="internship" data-program="<?php echo htmlspecialchars($program); ?>">
          <td>Internship</td>
          <td><?php echo htmlspecialchars($fullName . " (" . $indexNumber . ")"); ?></td>
          <td><?php echo htmlspecialchars($organization); ?></td>
          <td><?php echo htmlspecialchars($department . " - " . $program); ?></td>
          <td class="text-center">
            <button 
              type="button"
              class="btn btn-primary btn-sm d-flex align-items-center justify-content-center"
              style="width: 30px; height: 30px; padding: 0; border-radius: 5px;"
              data-bs-toggle="modal"
              data-bs-target="#detailsModal"
              data-type="internship"
              data-fullname="<?php echo htmlspecialchars($fullName); ?>"
              data-index_number="<?php echo htmlspecialchars($indexNumber); ?>"
              data-company="<?php echo htmlspecialchars($organization); ?>"
              data-department="<?php echo htmlspecialchars($department); ?>"
              data-start_date="<?php echo htmlspecialchars($startDate); ?>"
              data-end_date="<?php echo htmlspecialchars($endDate); ?>"
              data-executive_summary="<?php echo htmlspecialchars(decryptData($report['executive_summary']), ENT_QUOTES, 'UTF-8', false); ?>"
              data-introduction="<?php echo htmlspecialchars(decryptData($report['introduction']), ENT_QUOTES, 'UTF-8', false); ?>"
              data-institution_profile="<?php echo htmlspecialchars(decryptData($report['institution_profile']), ENT_QUOTES, 'UTF-8', false); ?>"
              data-goals_objectives="<?php echo htmlspecialchars(decryptData($report['goals_objectives']), ENT_QUOTES, 'UTF-8', false); ?>"
              data-core_duties="<?php echo htmlspecialchars(decryptData($report['core_duties']), ENT_QUOTES, 'UTF-8', false); ?>"
              data-organizational_challenges="<?php echo htmlspecialchars(decryptData($report['organizational_challenges']), ENT_QUOTES, 'UTF-8', false); ?>"
              data-student_challenges="<?php echo htmlspecialchars(decryptData($report['student_challenges']), ENT_QUOTES, 'UTF-8', false); ?>"
              data-achievements="<?php echo htmlspecialchars(decryptData($report['achievements']), ENT_QUOTES, 'UTF-8', false); ?>"
              data-lessons_learned="<?php echo htmlspecialchars(decryptData($report['lessons_learned']), ENT_QUOTES, 'UTF-8', false); ?>"
              data-recommendations="<?php echo htmlspecialchars(decryptData($report['recommendations']), ENT_QUOTES, 'UTF-8', false); ?>"
              data-conclusion="<?php echo htmlspecialchars(decryptData($report['conclusion']), ENT_QUOTES, 'UTF-8', false); ?>"
              data-references="<?php echo htmlspecialchars(decryptData($report['references']), ENT_QUOTES, 'UTF-8', false); ?>"
              data-appendix="<?php echo htmlspecialchars(decryptData($report['appendix']), ENT_QUOTES, 'UTF-8', false); ?>"
              data-submission_date="<?php echo htmlspecialchars($submissionDate); ?>"
              data-status="<?php echo htmlspecialchars($status); ?>"
              title="View Details"
            >
              <img 
                src="../public/img/view.svg"
                alt="View"
                style="width: 16px; height: 16px; filter: invert(1);"
              >
            </button>
          </td>
        </tr>
        <?php } else if ($item['type'] === 'teaching') {
            $treport = $item['data'];
            $surname = htmlspecialchars($treport['surname']);
            $firstName = htmlspecialchars($treport['first_name']);
            $otherNames = htmlspecialchars($treport['other_names']);
            $indexNumber = htmlspecialchars($treport['index_number']);
            $status = htmlspecialchars($treport['status']);
            $submittedAt = htmlspecialchars($treport['submitted_at']);
            $fullName = trim($surname . ' ' . $firstName . ' ' . $otherNames);
        ?>
        <tr data-type="teaching">
          <td>Teaching</td>
          <td><?php echo $fullName . " (" . $indexNumber . ")"; ?></td>
          <td><?php echo $status; ?></td>
          <td><?php echo $submittedAt; ?></td>
          <td class="text-center">
            <button 
              type="button"
              class="btn btn-primary btn-sm d-flex align-items-center justify-content-center"
              style="width: 30px; height: 30px; padding: 0; border-radius: 5px;"
              data-bs-toggle="modal"
              data-bs-target="#detailsModal"
              data-type="teaching"
              data-fullname="<?php echo $fullName; ?>"
              data-index_number="<?php echo $indexNumber; ?>"
              data-reflective_practice="<?php echo htmlspecialchars($treport['reflective_practice']); ?>"
              data-teaching_philosophy="<?php echo htmlspecialchars($treport['teaching_philosophy']); ?>"
              data-cv_path="<?php echo htmlspecialchars($treport['cv_path']); ?>"
              data-photo_album_entries='<?php echo htmlspecialchars($treport['photo_album_entries']); ?>'
              data-status="<?php echo $status; ?>"
              data-submitted_at="<?php echo $submittedAt; ?>"
              title="View Details"
            >
              <img 
                src="../public/img/view.svg"
                alt="View"
                style="width: 16px; height: 16px; filter: invert(1);"
              >
            </button>
          </td>
        </tr>
        <?php } endforeach; ?>
      </tbody>
    </table>
  </div>
</div>

<!-- Modal for viewing details -->
<div 
  class="modal fade" 
  id="detailsModal" 
  tabindex="-1" 
  aria-labelledby="detailsModalLabel" 
  aria-hidden="true"
>
  <div class="modal-dialog modal-dialog-scrollable modal-lg">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="detailsModalLabel">Internship Report Details</h5>
        <button 
          type="button" 
          class="btn-close" 
          data-bs-dismiss="modal" 
          aria-label="Close"
        ></button>
      </div>
      <div class="modal-body dark-mode-text">
        <div id="modalContent"><!-- JS will fill this --></div>
      </div>
      <div class="modal-footer">
        <button 
          type="button" 
          class="btn btn-secondary" 
          data-bs-dismiss="modal"
        >
          Close
        </button>
      </div>
    </div>
  </div>
</div>

<!-- Script to populate the modal content when triggered -->
<script>
function decodeHtmlEntities(str) {
  var txt = document.createElement('textarea');
  txt.innerHTML = str;
  var once = txt.value;
  txt.innerHTML = once;
  return txt.value;
}

  var detailsModal = document.getElementById('detailsModal');
  detailsModal.addEventListener('show.bs.modal', function (event) {
    var button = event.relatedTarget;
    var type = button.getAttribute('data-type');
    var contentHtml = '';
    if (type === 'internship') {
      var report = {
        fullname: button.getAttribute('data-fullname'),
        index_number: button.getAttribute('data-index_number'),
        company: button.getAttribute('data-company'),
        department: button.getAttribute('data-department'),
        start_date: button.getAttribute('data-start_date'),
        end_date: button.getAttribute('data-end_date'),
        executive_summary: button.getAttribute('data-executive_summary'),
        introduction: button.getAttribute('data-introduction'),
        institution_profile: button.getAttribute('data-institution_profile'),
        goals_objectives: button.getAttribute('data-goals_objectives'),
        core_duties: button.getAttribute('data-core_duties'),
        organizational_challenges: button.getAttribute('data-organizational_challenges'),
        student_challenges: button.getAttribute('data-student_challenges'),
        achievements: button.getAttribute('data-achievements'),
        lessons_learned: button.getAttribute('data-lessons_learned'),
        recommendations: button.getAttribute('data-recommendations'),
        conclusion: button.getAttribute('data-conclusion'),
        references: button.getAttribute('data-references'),
        appendix: button.getAttribute('data-appendix'),
        submission_date: button.getAttribute('data-submission_date'),
        status: button.getAttribute('data-status')
      };
      contentHtml = `
      <p><strong>Student Name:</strong> ${decodeHtmlEntities(report.fullname)}</p>
      <p><strong>Index Number:</strong> ${decodeHtmlEntities(report.index_number)}</p>
      <p><strong>Organization:</strong> ${decodeHtmlEntities(report.company)}</p>
      <p><strong>Department:</strong> ${decodeHtmlEntities(report.department)}</p>
      <p><strong>Start Date:</strong> ${decodeHtmlEntities(report.start_date)}</p>
      <p><strong>End Date:</strong> ${decodeHtmlEntities(report.end_date)}</p>
        <hr>
      <p><strong>Executive Summary:</strong> ${decodeHtmlEntities(report.executive_summary)}</p>
      <p><strong>Introduction:</strong> ${decodeHtmlEntities(report.introduction)}</p>
      <p><strong>Institution Profile:</strong> ${decodeHtmlEntities(report.institution_profile)}</p>
      <p><strong>Goals &amp; Objectives:</strong> ${decodeHtmlEntities(report.goals_objectives)}</p>
      <p><strong>Core Duties:</strong> ${decodeHtmlEntities(report.core_duties)}</p>
      <p><strong>Organizational Challenges:</strong> ${decodeHtmlEntities(report.organizational_challenges)}</p>
      <p><strong>Student Challenges:</strong> ${decodeHtmlEntities(report.student_challenges)}</p>
      <p><strong>Achievements:</strong> ${decodeHtmlEntities(report.achievements)}</p>
      <p><strong>Lessons Learned:</strong> ${decodeHtmlEntities(report.lessons_learned)}</p>
      <p><strong>Recommendations:</strong> ${decodeHtmlEntities(report.recommendations)}</p>
      <p><strong>Conclusion:</strong> ${decodeHtmlEntities(report.conclusion)}</p>
      <p><strong>References:</strong> ${decodeHtmlEntities(report.references)}</p>
      <p><strong>Appendix:</strong> ${decodeHtmlEntities(report.appendix)}</p>
      <p><strong>Submission Date:</strong> ${decodeHtmlEntities(report.submission_date)}</p>
      <p><strong>Status:</strong> ${decodeHtmlEntities(report.status)}</p>
      `;
    } else if (type === 'teaching') {
      var report = {
        fullname: button.getAttribute('data-fullname'),
        index_number: button.getAttribute('data-index_number'),
        reflective_practice: button.getAttribute('data-reflective_practice'),
        teaching_philosophy: button.getAttribute('data-teaching_philosophy'),
        cv_path: button.getAttribute('data-cv_path'),
        photo_album_entries: button.getAttribute('data-photo_album_entries'),
        status: button.getAttribute('data-status'),
        submitted_at: button.getAttribute('data-submitted_at')
      };
      var photoHtml = '';
      try {
        var photos = JSON.parse(report.photo_album_entries);
        if (Array.isArray(photos) && photos.length > 0) {
          photoHtml = '<div class="photo-grid">';
          photos.forEach(function(p) {
            photoHtml += '<div class="thumb-card">';
            photoHtml += '<img src="../uploads/photo_album/' + encodeURIComponent(p.filename) + '" alt="Photo" style="width:100px;height:100px;object-fit:cover;">';
          photoHtml += '<div class="caption-display">' + (p.caption ? decodeHtmlEntities(p.caption) : '—') + '</div>';
            photoHtml += '</div>';
          });
          photoHtml += '</div>';
        } else {
          photoHtml = '<p>No photos submitted.</p>';
        }
      } catch (e) {
        photoHtml = '<p>Error loading photos.</p>';
      }
      var cvLink = '';
      if (report.cv_path) {
        var ext = report.cv_path.split('.').pop().toLowerCase();
        if (ext === 'pdf') {
          cvLink = '<a href="../' + report.cv_path + '" target="_blank">View CV</a>';
        } else {
          cvLink = '<a href="../' + report.cv_path + '" target="_blank">Download CV</a>';
        }
      }
      contentHtml = `
      <p><strong>Student Name:</strong> ${decodeHtmlEntities(report.fullname)}</p>
      <p><strong>Index Number:</strong> ${decodeHtmlEntities(report.index_number)}</p>
      <p><strong>Status:</strong> ${decodeHtmlEntities(report.status)}</p>
      <p><strong>Submitted At:</strong> ${decodeHtmlEntities(report.submitted_at)}</p>
        <hr>
      <p><strong>Reflective Practice:</strong> ${decodeHtmlEntities(report.reflective_practice)}</p>
      <p><strong>Teaching Philosophy:</strong> ${decodeHtmlEntities(report.teaching_philosophy)}</p>
        <p><strong>CV:</strong> ${cvLink}</p>
        <h6>Photos & Captions</h6>
        ${photoHtml}
      `;
    }
    document.getElementById('modalContent').innerHTML = contentHtml;
  });
</script>

<script>
// Tab-style filtering with count updates
const tabs = document.querySelectorAll('#reportTabs .nav-link');
const rows = document.querySelectorAll('#reportsTable tbody tr');
const totalCount = document.getElementById('totalCount');
const internshipCount = document.getElementById('internshipCount');
const teachingCount = document.getElementById('teachingCount');
const programFilter = document.getElementById('programFilter');

let currentTabType = 'all';
let currentProgramFilter = '';

function updateVisibleCount() {
  let visibleCount = 0;
  let visibleInternshipCount = 0;
  let visibleTeachingCount = 0;
  
  rows.forEach(row => {
    if (row.style.display !== 'none') {
      visibleCount++;
      if (row.getAttribute('data-type') === 'internship') {
        visibleInternshipCount++;
      } else if (row.getAttribute('data-type') === 'teaching') {
        visibleTeachingCount++;
      }
    }
  });
  
  // Update the count display in the active tab
  const activeTab = document.querySelector('#reportTabs .nav-link.active');
  const activeType = activeTab.getAttribute('data-type');
  
  if (activeType === 'all') {
    activeTab.textContent = `All (${visibleCount})`;
  } else if (activeType === 'internship') {
    activeTab.textContent = `Post Internship Reports (${visibleInternshipCount})`;
  } else if (activeType === 'teaching') {
    activeTab.textContent = `Teaching Reports (${visibleTeachingCount})`;
  }
}

function filterRows() {
  rows.forEach(row => {
    const rowType = row.getAttribute('data-type');
    const rowProgram = row.getAttribute('data-program');
    
    let showByType = currentTabType === 'all' || rowType === currentTabType;
    let showByProgram = currentProgramFilter === '' || rowProgram === currentProgramFilter;
    
    if (showByType && showByProgram) {
      row.style.display = '';
    } else {
      row.style.display = 'none';
    }
  });
  
  updateVisibleCount();
}

tabs.forEach(tab => {
  tab.addEventListener('click', function() {
    tabs.forEach(t => t.classList.remove('active'));
    this.classList.add('active');
    currentTabType = this.getAttribute('data-type');
    filterRows();
  });
});

// Program filter event listener
programFilter.addEventListener('change', function() {
  currentProgramFilter = this.value;
  filterRows();
});

// Initialize counts on page load
updateVisibleCount();
</script>

<style>
.card {
  transition: transform 0.2s ease-in-out;
  border: none;
  box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.card:hover {
  transform: translateY(-2px);
  box-shadow: 0 4px 8px rgba(0,0,0,0.15);
}

.card-title {
  font-size: 0.9rem;
  font-weight: 600;
  margin-bottom: 0.5rem;
}

.card-text {
  font-size: 2rem;
  font-weight: bold;
  margin-bottom: 0;
}

.nav-tabs .nav-link {
  font-weight: 500;
}

.nav-tabs .nav-link.active {
  font-weight: 600;
}

.form-label {
  color: #495057;
  font-weight: 500;
}

#programFilter {
  border: 2px solid #dee2e6;
  transition: border-color 0.15s ease-in-out;
}

#programFilter:focus {
  border-color: #0d6efd;
  box-shadow: 0 0 0 0.2rem rgba(13, 110, 253, 0.25);
}
</style>

<?php
// Include the footer (closes the HTML tags and loads theme script)
include('pi_footer.php');
?>

