<?php
/**
 * pi.php  |  Programme-In-Charge dashboard
 * -------------------------------------------------
 * Session guard – identical pattern to your other protected pages.
 * Users who are not logged in (i.e., no `admin_id` in the session)
 * are redirected to login.php before any HTML is sent.
 */
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

if (!isset($_SESSION['admin_id'])) {   //  <-- Same key used across the admin/PI area
    header('Location: login');     //  <-- Same login file in this folder
    exit();
}

// Include database connection
include '../config/database.php';

// Handle teaching settings form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_teaching_settings'])) {
    $pptRequirement = isset($_POST['teaching_ppt_requirement']) ? '1' : '0';
    
    try {
        // Update the setting
        $stmt = $pdo->prepare("
            INSERT INTO settings (setting_key, setting_value, description, updated_at) 
            VALUES ('teaching_ppt_requirement', ?, 'Require PowerPoint submission for teaching internship grading (1=enabled, 0=disabled)', NOW())
            ON DUPLICATE KEY UPDATE setting_value = ?, updated_at = NOW()
        ");
        $stmt->execute([$pptRequirement, $pptRequirement]);
        
        $successMessage = "Teaching internship settings updated successfully!";
    } catch (PDOException $e) {
        $errorMessage = "Error updating settings: " . $e->getMessage();
    }
}

// Get current teaching settings
try {
    $stmt = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = 'teaching_ppt_requirement'");
    $stmt->execute();
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    $currentPptRequirement = $result ? $result['setting_value'] : '1'; // Default to enabled
} catch (PDOException $e) {
    $currentPptRequirement = '1'; // Default to enabled
}

include 'pi_header.php';               // Common header (local folder)
?>

<!-- Main Content -->
<div class="container mt-4">
    <!-- Back to Admin Dashboard Button -->
    <div class="mb-3">
        <a href="index.php" class="btn" style="background-color: #1f2a38; color: white; border-color: #1f2a38;">
            <i class="fas fa-tachometer-alt"></i> Back to Admin Dashboard
        </a>
    </div>
    
    <?php if (isset($successMessage)): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="fas fa-check-circle"></i> <?php echo htmlspecialchars($successMessage); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    <?php endif; ?>
    
    <?php if (isset($errorMessage)): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fas fa-exclamation-circle"></i> <?php echo htmlspecialchars($errorMessage); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    <?php endif; ?>

    <div class="row g-4">
        <!-- Card: Create & Manage Groups -->
        <div class="col-md-4">
            <div class="card card-custom p-3">
                <h5 class="card-title">Create and Manage Groups</h5>
                <p class="card-text">See all created lecturer groups.</p>
                <a href="view_groups" class="btn btn-custom">View Groups</a>
            </div>
        </div>

        <!-- Card: Assign Lecturers -->
        <div class="col-md-4">
            <div class="card card-custom p-3">
                <h5 class="card-title">Assign Lecturers to Groups</h5>
                <p class="card-text">Allocate lecturers to student groups.</p>
                <a href="assign_lecturer" class="btn btn-custom">Assign Lecturer</a>
            </div>
        </div>

        <!-- Card: Assign Students -->
        <div class="col-md-4">
            <div class="card card-custom p-3">
                <h5 class="card-title">Assign Students to Groups</h5>
                <p class="card-text">Allocate students to their respective groups.</p>
                <a href="assign_student" class="btn btn-custom">Assign Students</a>
            </div>
        </div>

        <!-- Card: Group Supervision Summary -->
        <div class="col-md-4">
            <div class="card card-custom p-3">
                <h5 class="card-title">Group Supervision Summary</h5>
                <p class="card-text">View the total number of students and lecturers for each group.</p>
                <a href="group_student_counts.php" class="btn btn-custom">View Group Summary</a>
            </div>
        </div>

        <!-- Card: Results -->
        <div class="col-md-4">
            <div class="card card-custom p-3">
                <h5 class="card-title">Results</h5>
                <p class="card-text">View student scores and results.</p>
                <a href="pi_student_scores" class="btn btn-custom">View Results</a>
            </div>
        </div>

        <!-- Card: Teaching Settings -->
        <div class="col-md-4">
            <div class="card card-custom p-3">
                <h5 class="card-title">Teaching Settings</h5>
                <p class="card-text">Manage teaching internship grading requirements.</p>
                <button type="button" class="btn btn-custom" data-bs-toggle="modal" data-bs-target="#teachingSettingsModal">
                    Manage Settings
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Teaching Settings Modal -->
<div class="modal fade" id="teachingSettingsModal" tabindex="-1" aria-labelledby="teachingSettingsModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header" style="background-color: #1f2a38; color: white;">
                <h5 class="modal-title" id="teachingSettingsModalLabel">
                    <i class="fas fa-cog"></i> Teaching Internship Settings
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <form method="POST">
                    <div class="mb-4">
                        <div class="form-check form-switch">
                            <input class="form-check-input" type="checkbox" id="teaching_ppt_requirement" 
                                   name="teaching_ppt_requirement" 
                                   <?php echo $currentPptRequirement === '1' ? 'checked' : ''; ?>>
                            <label class="form-check-label" for="teaching_ppt_requirement">
                                <strong>Require PowerPoint Submission for Teaching Internship Grading</strong>
                            </label>
                        </div>
                        <div class="form-text">
                            <i class="fas fa-info-circle"></i> 
                            When enabled, lecturers can only grade teaching internship students who have submitted their PowerPoint presentations. 
                            When disabled, lecturers can grade students regardless of PowerPoint submission status.
                        </div>
                    </div>
                    
                    <!-- Current Status -->
                    <div class="alert alert-info">
                        <h6><i class="fas fa-info-circle"></i> Current Status</h6>
                        <div class="row">
                            <div class="col-md-6">
                                <strong>PowerPoint Requirement:</strong>
                                <span class="badge <?php echo $currentPptRequirement === '1' ? 'bg-success' : 'bg-warning'; ?>">
                                    <?php echo $currentPptRequirement === '1' ? 'Enabled' : 'Disabled'; ?>
                                </span>
                            </div>
                            <div class="col-md-6">
                                <strong>Effect:</strong>
                                <?php if ($currentPptRequirement === '1'): ?>
                                    <span class="text-success">Lecturers must wait for PowerPoint submission</span>
                                <?php else: ?>
                                    <span class="text-warning">Lecturers can grade without PowerPoint</span>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    
                    <div class="d-grid gap-2 d-md-flex justify-content-md-end">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" name="update_teaching_settings" class="btn btn-primary">
                            <i class="fas fa-save"></i> Save Settings
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<?php include 'pi_footer.php'; ?>
