<?php
session_start();

// If you only want logged-in admins, uncomment:
// if (!isset($_SESSION['admin_id'])) {
//     header('Location: login.php');
//     exit;
// }

include '../config/database.php'; // Adjust this path if needed

// 1. Query database for (district, count) from student_details
$query = "
    SELECT district, COUNT(*) AS total_students
    FROM student_details
    GROUP BY district
    ORDER BY district
";
$stmt = $pdo->prepare($query);
$stmt->execute();
$results = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Build an associative array: districtName => count
$districtCounts = [];
foreach ($results as $row) {
    $districtName = $row['district'];
    $count        = (int)$row['total_students'];
    // e.g. "Cape Coast Metropolitan" => 12
    $districtCounts[$districtName] = $count;
}

// Calculate min & max for a color scale
$minCount = $results ? min(array_column($results, 'total_students')) : 0;
$maxCount = $results ? max(array_column($results, 'total_students')) : 1;

// If min == max (e.g. all are zero or all the same), avoid division by zero
if ($minCount === $maxCount) {
    $maxCount = $minCount + 1; // so we don't divide by zero later
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Ghana District Map - Student Counts</title>
    <!-- Leaflet CSS -->
    <link rel="stylesheet" 
          href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css"
          integrity="sha256-1yNQ+NAf2AEN3B14u0I8y+9EKA+7AJAM7r3AWqKepWY="
          crossorigin=""/>

    <style>
      body {
        margin: 0;
        padding: 0;
      }

      /* Map container */
      #map {
        width: 100%;
        height: 90vh; /* or any desired height */
      }

      /* Simple optional legend styling (if you add a legend) */
      .legend {
        background: white;
        padding: 8px;
        line-height: 1.5em;
        font-size: 14px;
      }
      .legend i {
        width: 15px;
        height: 15px;
        float: left;
        margin-right: 5px;
        opacity: 0.7;
      }
    </style>
</head>
<body>

<h2 style="margin:1rem;">Ghana District Map - Student Counts</h2>
<div id="map"></div>

<!-- Leaflet JS -->
<script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"
        integrity="sha256-kxO2W3g1tUP2+rhu/0Pp7PFk3slf0p4HTa3f2FjU3=E="
        crossorigin="">
</script>

<script>
// Pass PHP array to JS
const districtCounts = <?php echo json_encode($districtCounts, JSON_HEX_QUOT|JSON_HEX_TAG|JSON_HEX_AMP|JSON_HEX_APOS); ?>;
const minCount = <?php echo json_encode($minCount); ?>;
const maxCount = <?php echo json_encode($maxCount); ?>;

// Initialize the map, center on Ghana approximately
const map = L.map('map').setView([7.95, -1.03], 7);
// Add a base tile (OpenStreetMap)
L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
    maxZoom: 19
}).addTo(map);

/**
 * A function to get a color for each district.
 * 
 * We can do linear interpolation from green to red based on the ratio:
 *   ratio = (count - minCount) / (maxCount - minCount)
 * Or define manual threshold steps. Here we do a simple linear.
 */
function getDistrictColor(districtName) {
    const count = districtCounts[districtName] || 0;
    const ratio = (count - minCount) / (maxCount - minCount);
    // clamp ratio between 0 and 1
    const r = Math.round(255 * ratio);         // 0 => green(0), 1 => red(255)
    const g = Math.round(255 * (1 - ratio));   // 1 => green(255), 0 => red(0)
    return `rgb(${r},${g},0)`; 
}

// Fetch the Ghana district GeoJSON (adjust path to your file)
fetch('ghana_districts.geojson')
  .then(response => response.json())
  .then(geoData => {
    // Create a Leaflet GeoJSON layer
    const geoLayer = L.geoJSON(geoData, {
      style: (feature) => {
        const distName = feature.properties.district_name; // match your GeoJSON property
        return {
          color: '#555',           // boundary line color
          weight: 1,               // boundary thickness
          fillColor: getDistrictColor(distName),
          fillOpacity: 0.6
        };
      },
      onEachFeature: (feature, layer) => {
        const distName = feature.properties.district_name;
        const count = districtCounts[distName] || 0;
        // Bind a popup
        layer.bindPopup(`
          <strong>${distName}</strong><br>
          Students: ${count}
        `);
      }
    }).addTo(map);

    // Optionally fit to the GeoJSON bounds
    map.fitBounds(geoLayer.getBounds());
  })
  .catch(err => {
    console.error("Error loading GeoJSON:", err);
  });
</script>
</body>
</html>
