<?php
session_start();
require '../config/database.php';
require '../vendor/autoload.php';

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

// Admin session check
if (!isset($_SESSION['admin_id'])) {
    header("Location: login");
    exit;
}

// Helper function to compute letter grade based on average score
function getLetterGrade($score)
{
    if ($score >= 80) {
        return 'A';
    } elseif ($score >= 75) {
        return 'B+';
    } elseif ($score >= 70) {
        return 'B';
    } elseif ($score >= 65) {
        return 'C+';
    } elseif ($score >= 60) {
        return 'C';
    } elseif ($score >= 55) {
        return 'D+';
    } elseif ($score >= 50) {
        return 'D';
    } else {
        return 'E';
    }
}

// Get selected group ID from URL parameter
$selectedGroupId = isset($_GET['group_id']) && $_GET['group_id'] !== '' ? $_GET['group_id'] : null;

// Retrieve groups based on filter
if ($selectedGroupId) {
    $stmtGroups = $pdo->prepare("SELECT * FROM pi_lecturer_groups WHERE id = ? ORDER BY group_name");
    $stmtGroups->execute([$selectedGroupId]);
    $groups = $stmtGroups->fetchAll(PDO::FETCH_ASSOC);
} else {
    $stmtGroups = $pdo->query("SELECT * FROM pi_lecturer_groups ORDER BY group_name");
    $groups = $stmtGroups->fetchAll(PDO::FETCH_ASSOC);
}

// Create new Spreadsheet object
$spreadsheet = new Spreadsheet();

// Set document properties
$spreadsheet->getProperties()
    ->setCreator('IMS System')
    ->setLastModifiedBy('IMS System')
    ->setTitle('Student Scores Report')
    ->setSubject('Student Scores by Group')
    ->setDescription('Student scores report exported from IMS system');

// Set headers
$headers = [
    'Index Number',
    'Student Name',
    'Program',
    'Report Score (50 pts)',
    'Presentation Score (50 pts)',
    'Final Score (100 pts)',
    'Grade'
];

$worksheetIndex = 0;

foreach ($groups as $group) {
    // Create or get worksheet for this group
    if ($worksheetIndex === 0) {
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle($group['group_name']);
    } else {
        $sheet = $spreadsheet->createSheet();
        $sheet->setTitle($group['group_name']);
    }
    
    // Get lecturers for this group
    $stmtLecturers = $pdo->prepare("SELECT id, name FROM lecturer_login WHERE group_id = ?");
    $stmtLecturers->execute([$group['id']]);
    $lecturers = $stmtLecturers->fetchAll(PDO::FETCH_ASSOC);
    
    $lecturerNames = array_map(function($lect) {
        return $lect['name'];
    }, $lecturers);
    $lecturerNamesStr = implode(', ', $lecturerNames);
    
    // Add group header with supervisors
    $sheet->setCellValue('A1', 'Group: ' . $group['group_name']);
    $sheet->setCellValue('A2', 'Supervisors: ' . $lecturerNamesStr);
    $sheet->setCellValue('A3', 'Category: ' . ($group['category'] ?? 'Not Set'));
    $sheet->getStyle('A1:A3')->getFont()->setBold(true);
    $sheet->getStyle('A1:A3')->getFont()->setSize(12);
    $sheet->getStyle('A1:A3')->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID);
    $sheet->getStyle('A1:A3')->getFill()->getStartColor()->setRGB('E8F4FD');
    
    // Set headers starting from row 5
    $col = 'A';
    foreach ($headers as $header) {
        $sheet->setCellValue($col . '5', $header);
        if ($col === 'A') {
            // Set fixed width for Index Number column
            $sheet->getColumnDimension($col)->setWidth(15);
        } else {
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }
        $col++;
    }
    
    // Style the header row
    $sheet->getStyle('A5:G5')->getFont()->setBold(true);
    $sheet->getStyle('A5:G5')->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID);
    $sheet->getStyle('A5:G5')->getFill()->getStartColor()->setRGB('1f2a38');
    $sheet->getStyle('A5:G5')->getFont()->getColor()->setRGB('FFFFFF');
    
    $row = 6;
    
    // Get students for this group with their supervising lecturers
    $query = "
        SELECT
            s.id AS student_id,
            CONCAT(s.surname, ' ', s.first_name, ' ', COALESCE(s.other_names, '')) AS student_name,
            u.index_number,
            s.course AS program,
            u.group AS user_group,
            ROUND(AVG(psg.final_score), 2) AS average_score,
            ll.name AS supervising_lecturer
        FROM student_details s
        JOIN users u ON s.user_id = u.id
        LEFT JOIN pi_student_grades psg ON s.id = psg.student_id
        LEFT JOIN lecturer_login ll ON s.lecturer_id = ll.id
        WHERE s.lecturer_group_id = ?
        GROUP BY s.id
        ORDER BY s.surname, s.first_name
    ";
    $stmtStudents = $pdo->prepare($query);
    $stmtStudents->execute([$group['id']]);
    $students = $stmtStudents->fetchAll(PDO::FETCH_ASSOC);
    
    foreach ($students as $student) {
        $sheet->setCellValue('A' . $row, $student['index_number']);
        $sheet->setCellValue('B' . $row, $student['student_name']);
        $sheet->setCellValue('C' . $row, $student['program']);
        
        if ($student['user_group'] === 'Teaching Practice') {
            // Get teaching practice score
            $stmtTeach = $pdo->prepare("SELECT AVG(total_score) AS avg_score FROM teaching_grades WHERE student_id = ?");
            $stmtTeach->execute([$student['student_id']]);
            $teachScore = $stmtTeach->fetch(PDO::FETCH_ASSOC);
            
            $sheet->setCellValue('D' . $row, 'N/A');
            $sheet->setCellValue('E' . $row, 'N/A');
            
            if ($teachScore && $teachScore['avg_score'] !== null) {
                $finalScore = round($teachScore['avg_score'], 2);
                $sheet->setCellValue('F' . $row, $finalScore);
                $sheet->setCellValue('G' . $row, getLetterGrade($finalScore));
            } else {
                $sheet->setCellValue('F' . $row, 'Not Graded');
                $sheet->setCellValue('G' . $row, 'Not Graded');
            }
        } else {
            // Get industrial internship scores
            $stmtReport = $pdo->prepare("
                SELECT report_score FROM pi_student_grades 
                WHERE student_id = ? 
                ORDER BY graded_at DESC LIMIT 1
            ");
            $stmtReport->execute([$student['student_id']]);
            $reportScore = $stmtReport->fetch(PDO::FETCH_ASSOC);
            
            $stmtPres = $pdo->prepare("
                SELECT presentation_score FROM pi_student_grades 
                WHERE student_id = ? 
                ORDER BY graded_at DESC LIMIT 1
            ");
            $stmtPres->execute([$student['student_id']]);
            $presScore = $stmtPres->fetch(PDO::FETCH_ASSOC);
            
            $stmtFinal = $pdo->prepare("
                SELECT final_score FROM pi_student_grades 
                WHERE student_id = ? 
                ORDER BY graded_at DESC LIMIT 1
            ");
            $stmtFinal->execute([$student['student_id']]);
            $finalScore = $stmtFinal->fetch(PDO::FETCH_ASSOC);
            
            $sheet->setCellValue('D' . $row, $reportScore && $reportScore['report_score'] !== null ? $reportScore['report_score'] : 'Not Graded');
            $sheet->setCellValue('E' . $row, $presScore && $presScore['presentation_score'] !== null ? $presScore['presentation_score'] : 'Not Graded');
            $sheet->setCellValue('F' . $row, $finalScore && $finalScore['final_score'] !== null ? $finalScore['final_score'] : 'Not Graded');
            $sheet->setCellValue('G' . $row, $finalScore && $finalScore['final_score'] !== null ? getLetterGrade($finalScore['final_score']) : 'Not Graded');
        }
        
        // Add alternating row colors for better readability
        if ($row % 2 == 0) {
            $sheet->getStyle('A' . $row . ':G' . $row)->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID);
            $sheet->getStyle('A' . $row . ':G' . $row)->getFill()->getStartColor()->setRGB('F8F9FA');
        }
        
        $row++;
    }
    
    // Add summary statistics at the bottom
    $summaryRow = $row + 2;
    $sheet->setCellValue('A' . $summaryRow, 'Summary Statistics');
    $sheet->getStyle('A' . $summaryRow)->getFont()->setBold(true);
    $sheet->getStyle('A' . $summaryRow)->getFont()->setSize(14);
    
    $summaryRow++;
    $sheet->setCellValue('A' . $summaryRow, 'Total Students: ' . count($students));
    $sheet->setCellValue('D' . $summaryRow, 'Teaching Practice: ' . count(array_filter($students, function($s) { return $s['user_group'] === 'Teaching Practice'; })));
    $sheet->setCellValue('G' . $summaryRow, 'Industrial Internship: ' . count(array_filter($students, function($s) { return $s['user_group'] !== 'Teaching Practice'; })));
    
    // Add borders to the entire data range
    $lastRow = $row - 1;
    if ($lastRow >= 6) {
        $sheet->getStyle('A5:G' . $lastRow)->getBorders()->getAllBorders()->setBorderStyle(\PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN);
    }
    
    $worksheetIndex++;
}

// Generate filename with timestamp
$timestamp = date('Y-m-d_H-i-s');
$groupName = $selectedGroupId ? $groups[0]['group_name'] : 'All_Groups';
$filename = "Student_Scores_{$groupName}_{$timestamp}.xlsx";

// Set headers for download
header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
header('Content-Disposition: attachment;filename="' . $filename . '"');
header('Cache-Control: max-age=0');

// Create Excel file
$writer = new Xlsx($spreadsheet);
$writer->save('php://output');
exit;
?> 