<?php
// export.php

// Include Composer's autoloader for PhpSpreadsheet.
require '../vendor/autoload.php';

// Include your project's PDO database connection.
include '../config/database.php';

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Alignment;

// ---------------------------------------------------------------------
// PRG Pattern Implementation
// ---------------------------------------------------------------------
// If the form is submitted via POST, redirect to a GET request with export=1.
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['export'])) {
    header("Location: ?export=1");
    exit();
}

// ---------------------------------------------------------------------
// Export functionality: When export=1 is set in the GET parameters.
// ---------------------------------------------------------------------
if (isset($_GET['export'])) {

    // ======================================================================
    // 1. Get the Lecturer Summary Data
    // ======================================================================
    $summarySql = "
        SELECT 
            ll.name AS lecturer_name,
            COUNT(lad.district) AS district_count,
            COUNT(DISTINCT lad.region) AS region_count
        FROM lecturer_assigned_districts lad
        JOIN lecturer_login ll ON lad.lecturer_id = ll.id
        GROUP BY lad.lecturer_id, ll.name
        ORDER BY ll.name
    ";

    try {
        $stmtSummary = $pdo->query($summarySql);
        $summaryData = $stmtSummary->fetchAll();
    } catch (PDOException $e) {
        die("Error executing summary query: " . htmlspecialchars($e->getMessage(), ENT_QUOTES, 'UTF-8'));
    }

    // ======================================================================
    // 2. Get the Detailed Region Data for each Lecturer Assignment
    // ======================================================================
    $detailSql = "SELECT 
                    lad.region, 
                    lad.district, 
                    ll.name AS lecturer_name,
                    (SELECT COUNT(*) 
                     FROM student_details sd 
                     WHERE sd.region = lad.region AND sd.district = lad.district
                    ) AS student_count
                FROM lecturer_assigned_districts lad
                JOIN lecturer_login ll ON lad.lecturer_id = ll.id
                ORDER BY lad.region, lad.district";

    try {
        $stmtDetail = $pdo->query($detailSql);
        $detailRows = $stmtDetail->fetchAll();
    } catch (PDOException $e) {
        die("Error executing detail query: " . htmlspecialchars($e->getMessage(), ENT_QUOTES, 'UTF-8'));
    }

    // Group detailed records by region.
    $regions = [];
    foreach ($detailRows as $row) {
        $region = $row['region'];
        if (!isset($regions[$region])) {
            $regions[$region] = [];
        }
        $regions[$region][] = $row;
    }

    // ======================================================================
    // 3. Create the Excel Spreadsheet
    // ======================================================================
    $spreadsheet = new Spreadsheet();

    // ---------------------
    // Create the Summary Sheet (First Sheet)
    // ---------------------
    $summarySheet = $spreadsheet->getActiveSheet();
    $summarySheet->setTitle('Summary');

    // Set header row for summary sheet.
    $summarySheet->setCellValue('A1', 'Lecturer Name');
    $summarySheet->setCellValue('B1', 'Districts Assigned');
    $summarySheet->setCellValue('C1', 'Regions Assigned');

    // Style the header row.
    $summarySheet->getStyle('A1:C1')->getFont()->setBold(true);
    $summarySheet->getStyle('A1:C1')->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);

    // Populate summary data (starting at row 2).
    $rowIndex = 2;
    foreach ($summaryData as $record) {
        $summarySheet->setCellValue("A{$rowIndex}", $record['lecturer_name']);
        $summarySheet->setCellValue("B{$rowIndex}", (int)$record['district_count']);
        $summarySheet->setCellValue("C{$rowIndex}", (int)$record['region_count']);
        $rowIndex++;
    }

    // Auto size columns A to C.
    foreach (range('A', 'C') as $colID) {
        $summarySheet->getColumnDimension($colID)->setAutoSize(true);
    }

    // ---------------------
    // Create a Worksheet for Each Region
    // ---------------------
    foreach ($regions as $regionName => $records) {
        // Ensure the sheet name is at most 31 characters.
        $sheetName = substr($regionName, 0, 31);
        $sheet = new \PhpOffice\PhpSpreadsheet\Worksheet\Worksheet($spreadsheet, $sheetName);
        $spreadsheet->addSheet($sheet);

        // Add the region name as a main header (merged across columns A to C).
        $sheet->mergeCells('A1:C1');
        $sheet->setCellValue('A1', $regionName);
        $sheet->getStyle('A1')->getFont()->setBold(true);
        $sheet->getStyle('A1')->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);

        // Set table header row (starting on row 2).
        $sheet->setCellValue('A2', 'District');
        $sheet->setCellValue('B2', 'Lecturer Name');
        $sheet->setCellValue('C2', 'Number of Students');

        // Bold the header row.
        $sheet->getStyle('A2:C2')->getFont()->setBold(true);

        // Populate data rows (starting from row 3).
        $rowIndex = 3;
        foreach ($records as $record) {
            $sheet->setCellValue("A{$rowIndex}", $record['district']);
            $sheet->setCellValue("B{$rowIndex}", $record['lecturer_name']);
            $sheet->setCellValue("C{$rowIndex}", (int)$record['student_count']);
            $rowIndex++;
        }

        // Auto size columns A to C.
        foreach (range('A', 'C') as $columnID) {
            $sheet->getColumnDimension($columnID)->setAutoSize(true);
        }
    }

    // Set the Summary sheet as the active sheet.
    $spreadsheet->setActiveSheetIndex(0);

    // ======================================================================
    // 4. Output the Excel File for Download
    // ======================================================================
    header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    header('Content-Disposition: attachment;filename="lecturer_summary_and_details.xlsx"');
    header('Cache-Control: max-age=0');

    $writer = new Xlsx($spreadsheet);
    $writer->save('php://output');
    exit();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Export Lecturer Summary and Region Details</title>
  <!-- Bootstrap 5 CSS -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.0.2/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    body { margin-top: 20px; }
  </style>
</head>
<body>
  <div class="container">
    <h1 class="mb-4">Export Lecturer Summary and Region Details</h1>
    <p>
      Click the button below to download an Excel file with:
    </p>
    <ul>
      <li>A summary sheet showing Lecturer Name, Number of Districts Assigned, and Number of Regions Assigned.</li>
      <li>Separate sheets for each region with detailed assignments and number of students.</li>
    </ul>
    <!-- Use a form with POST method to implement the PRG pattern -->
    <form method="post" action="">
      <button type="submit" name="export" class="btn btn-success">Export to Excel</button>
    </form>
  </div>
  
  <!-- Bootstrap 5 JS (optional) -->
  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.0.2/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
