<?php
/**
 * assign_student.php  |  Manually / automatically allocate students to PI groups
 * -------------------------------------------------------------------------------
 * SESSION GATE – identical pattern to your other secured pages.
 * If the user is not logged in (no `admin_id` in the session), redirect to login.php
 * before any output is sent.
 */
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

if (!isset($_SESSION['admin_id'])) {        // same session key used throughout
    header('Location: login.php');          // same login target
    exit();
}

/* ------------------------------------------------------------------ */
/*  Error display (turn down in production)                            */
/* ------------------------------------------------------------------ */
error_reporting(E_ALL);
ini_set('display_errors', 1);

/* ------------------------------------------------------------------ */
/*  Database connection                                                */
/* ------------------------------------------------------------------ */
require '../config/database.php';

include 'pi_header.php';   // after the session guard
/* ================================================================== */
/*  Course abbreviation lookup table                                   */
/* ================================================================== */
$courseAbbreviations = [
    'Accounting'                              => 'ACCT',
    'Procurement and Supply Chain Management' => 'PSCM',
    'Marketing and Entrepreneurship'          => 'Mkt&E',
    'Business Information Systems'            => 'BIS',
    'Human Resource Management'               => 'HRM',
    'Banking and Finance'                     => 'B&F',
    'Risk and Insurance'                      => 'R&I',
];

/* ------------------------------------------------------------------ */
/* 1. Students eligible for manual allocation                          */
/* ------------------------------------------------------------------ */
try {
    $students = $pdo->query("
        SELECT s.id, s.first_name, s.surname, u.group, s.course, s.lecturer_group_id
        FROM student_details s
        JOIN users u ON s.user_id = u.id
        WHERE s.lecturer_group_id IS NULL
        ORDER BY u.group DESC, s.first_name ASC
    ")->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    die('Error fetching students: ' . $e->getMessage());
}

/* ------------------------------------------------------------------ */
/* 2. Students eligible for auto-allocation                            */
/* ------------------------------------------------------------------ */
try {
    $auto_students = $pdo->query("
        SELECT s.id, s.lecturer_id
        FROM student_details s
        JOIN users u ON s.user_id = u.id
        WHERE s.lecturer_group_id IS NULL
          AND s.lecturer_id IS NOT NULL
          AND u.group != 'Teaching Practice'
    ")->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    die('Error fetching auto-allocated students: ' . $e->getMessage());
}

/* ------------------------------------------------------------------ */
/* 3. Groups that have ≥1 lecturer                                     */
/* ------------------------------------------------------------------ */
try {
    $groups = $pdo->query("
        SELECT g.id, g.group_name
        FROM pi_lecturer_groups g
        JOIN lecturer_login l ON g.id = l.group_id
        GROUP BY g.id
        ORDER BY g.group_name ASC
    ")->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    die('Error fetching groups: ' . $e->getMessage());
}

/* ------------------------------------------------------------------ */
/* 4. Current student allocations                                      */
/* ------------------------------------------------------------------ */
try {
    $allocations = $pdo->query("
        SELECT s.id AS student_id,
               CONCAT(s.first_name, ' ', s.surname) AS student_name,
               s.course,
               g.id AS group_id,
               g.group_name
        FROM student_details s
        JOIN pi_lecturer_groups g ON s.lecturer_group_id = g.id
        ORDER BY g.group_name ASC
    ")->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $allocations = [];
}

/* ------------------------------------------------------------------ */
/*  Helper: redirect to self to avoid resubmission                     */
/* ------------------------------------------------------------------ */
function redirectToSelf() {
    header('Location: ' . $_SERVER['PHP_SELF']);
    exit();
}

/* ------------------------------------------------------------------ */
/*  Manual assignment handler                                          */
/* ------------------------------------------------------------------ */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['assign'])) {
    if (!empty($_POST['student_id']) && !empty($_POST['group_id'])) {
        $student_id = $_POST['student_id'];
        $group_id   = $_POST['group_id'];

        try {
            $stmt = $pdo->prepare("
                UPDATE student_details
                SET lecturer_group_id = :group_id
                WHERE id = :student_id
            ");
            $stmt->execute(['group_id' => $group_id, 'student_id' => $student_id]);
            redirectToSelf();
        } catch (PDOException $e) {
            echo "<div class='alert alert-danger'>Error assigning student: " .
                 $e->getMessage() . "</div>";
        }
    } else {
        echo "<div class='alert alert-warning'>Please select both a student and a group.</div>";
    }
}

/* ------------------------------------------------------------------ */
/*  Remove individual student                                          */
/* ------------------------------------------------------------------ */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['remove_individual'])) {
    if (!empty($_POST['student_id'])) {
        try {
            $stmt = $pdo->prepare("
                UPDATE student_details
                SET lecturer_group_id = NULL
                WHERE id = :student_id
            ");
            $stmt->execute(['student_id' => $_POST['student_id']]);
            redirectToSelf();
        } catch (PDOException $e) {
            echo "<div class='alert alert-danger'>Error removing student: " .
                 $e->getMessage() . "</div>";
        }
    }
}

/* ------------------------------------------------------------------ */
/*  Random allocation handler                                          */
/* ------------------------------------------------------------------ */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['random_assign'])) {
    if ($auto_students) {
        try {
            $stmtLookup = $pdo->prepare("
                SELECT group_id
                FROM lecturer_login
                WHERE id = :lecturer_id
                LIMIT 1
            ");
            $stmtUpdate = $pdo->prepare("
                UPDATE student_details
                SET lecturer_group_id = :group_id
                WHERE id = :student_id
            ");

            foreach ($auto_students as $student) {
                $stmtLookup->execute(['lecturer_id' => $student['lecturer_id']]);
                $lecturerGroup = $stmtLookup->fetchColumn();

                if ($lecturerGroup) {
                    $stmtUpdate->execute([
                        'group_id'   => $lecturerGroup,
                        'student_id' => $student['id'],
                    ]);
                }
            }
            redirectToSelf();
        } catch (PDOException $e) {
            echo "<div class='alert alert-danger'>Error: " . $e->getMessage() . "</div>";
        }
    } else {
        echo "<div class='alert alert-warning'>No students or available groups found!</div>";
    }
}

/* ------------------------------------------------------------------ */
/*  Remove all students from groups                                    */
/* ------------------------------------------------------------------ */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['remove_all'])) {
    try {
        $pdo->exec("
            UPDATE student_details
            SET lecturer_group_id = NULL
            WHERE lecturer_group_id IS NOT NULL
        ");
        redirectToSelf();
    } catch (PDOException $e) {
        echo "<div class='alert alert-danger'>Error: " . $e->getMessage() . "</div>";
    }
}
?>

<!-- ----------------------------------------------------------------- -->
<!--  HTML & JS                                                         -->
<!-- ----------------------------------------------------------------- -->
<div class="container mt-5">
    <!-- Manual assignment card -->
    <div class="row">
        <div class="col-md-8 mx-auto">
            <div class="card shadow-sm p-3">
                <h4 class="text-center mb-3">Manually Assign Student to a Group</h4>
                <form method="post">
                    <div class="mb-3">
                        <label class="form-label">Select Student</label>
                        <select name="student_id" class="form-control form-control-sm" required>
                            <option value="">-- Select Student --</option>
                            <?php foreach ($students as $s): ?>
                                <?php
                                    $studentName = htmlspecialchars($s['first_name'] . ' ' . $s['surname']);
                                    $courseAbbr  = $courseAbbreviations[$s['course']] ?? $s['course'];
                                ?>
                                <option value="<?= $s['id'] ?>">
                                    <?= $studentName ?> (<?= $courseAbbr ?>)
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Select Group</label>
                        <select name="group_id" class="form-control form-control-sm" required>
                            <option value="">-- Select Group --</option>
                            <?php foreach ($groups as $g): ?>
                                <option value="<?= $g['id'] ?>">
                                    <?= htmlspecialchars($g['group_name']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <button type="submit" name="assign" class="btn btn-primary btn-sm w-100">
                        Manually Assign Student
                    </button>
                </form>
            </div>
        </div>
    </div>

    <!-- Action buttons -->
    <div class="text-center mt-4">
        <form method="post" class="d-inline">
            <button type="submit" name="random_assign" class="btn btn-success">
                Randomly Assign All Students
            </button>
        </form>
        <form method="post" class="d-inline">
            <button type="submit" name="remove_all" class="btn btn-danger"
                    onclick="return confirm('Are you sure you want to remove all students?');">
                Remove All Students
            </button>
        </form>
    </div>

    <!-- Group filter -->
    <div class="mt-4">
        <label for="groupFilter">Filter by Group:</label>
        <select id="groupFilter" class="form-control form-control-sm w-25" onchange="filterGroup()">
            <option value="">-- Show All Groups --</option>
            <?php foreach ($groups as $g): ?>
                <option value="<?= $g['id'] ?>">
                    <?= htmlspecialchars($g['group_name']) ?>
                </option>
            <?php endforeach; ?>
        </select>
    </div>

    <!-- Selected group name -->
    <h4 class="text-center mt-4" id="selectedGroupName">All Groups</h4>

    <!-- Allocations table -->
    <table id="studentTable" class="table table-striped table-sm">
        <thead>
            <tr>
                <th>Student (Course)</th>
                <th>Action</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($allocations as $a): ?>
                <tr data-group="<?= $a['group_id'] ?>">
                    <td>
                        <?php
                            echo htmlspecialchars(
                                $a['student_name'] . ' (' .
                                ($courseAbbreviations[$a['course']] ?? $a['course']) .
                                ')'
                            );
                        ?>
                    </td>
                    <td>
                        <form method="post" style="display:inline;">
                            <input type="hidden" name="student_id" value="<?= $a['student_id'] ?>">
                            <button type="submit" name="remove_individual"
                                    class="btn btn-danger btn-sm"
                                    onclick="return confirm('Are you sure you want to remove this student?');">
                                Remove
                            </button>
                        </form>
                    </td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
</div>

<script>
    function filterGroup() {
        const selected = document.getElementById('groupFilter').value;
        const rows     = document.querySelectorAll('#studentTable tbody tr');
        const heading  = document.getElementById('selectedGroupName');

        heading.textContent = selected === ''
            ? 'All Groups'
            : document.querySelector(`#groupFilter option[value='${selected}']`).textContent;

        rows.forEach(row => {
            row.style.display = (selected === '' || row.dataset.group === selected) ? '' : 'none';
        });
    }
</script>

<?php include 'pi_footer.php'; ?>
