<?php
session_start();
require '../config/database.php';

// Admin session check
if (!isset($_SESSION['admin_id'])) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);
$group_id = $input['group_id'] ?? '';

try {
    // Build the query based on whether a specific group is selected
    if (!empty($group_id)) {
        $stmtStudents = $pdo->prepare("
            SELECT s.id AS student_id, s.surname, s.first_name, s.other_names, s.course AS program, u.group AS user_group
            FROM student_details s
            JOIN users u ON s.user_id = u.id
            WHERE s.lecturer_group_id = ?
            ORDER BY s.surname, s.first_name
        ");
        $stmtStudents->execute([$group_id]);
    } else {
        $stmtStudents = $pdo->prepare("
            SELECT s.id AS student_id, s.surname, s.first_name, s.other_names, s.course AS program, u.group AS user_group
            FROM student_details s
            JOIN users u ON s.user_id = u.id
            ORDER BY s.surname, s.first_name
        ");
        $stmtStudents->execute();
    }
    
    $students = $stmtStudents->fetchAll(PDO::FETCH_ASSOC);
    
    if (empty($students)) {
        echo json_encode(['success' => false, 'message' => 'No students found']);
        exit;
    }
    
    $studentResults = [];
    $reportScores = [];
    $presentationScores = [];
    $finalScores = [];
    $gradeDistribution = ['A' => 0, 'B_plus' => 0, 'B' => 0, 'C_plus' => 0, 'C' => 0, 'D_E' => 0];
    
    foreach ($students as $student) {
        $studentName = trim($student['surname'] . ' ' . $student['first_name'] . ' ' . ($student['other_names'] ?? ''));
        
        if ($student['user_group'] === 'Teaching Practice') {
            $stmtTeach = $pdo->prepare("SELECT AVG(total_score) AS average_score FROM teaching_grades WHERE student_id = ?");
            $stmtTeach->execute([$student['student_id']]);
            $teachingData = $stmtTeach->fetch(PDO::FETCH_ASSOC);
            
            $finalScore = round($teachingData['average_score'] ?? 0, 2);
            $letterGrade = getLetterGrade($finalScore);
            
            if ($finalScore > 0) {
                $finalScores[] = $finalScore;
                updateGradeDistribution($gradeDistribution, $letterGrade);
            }
            
            $studentResults[] = [
                'student_id' => $student['student_id'],
                'student_name' => $studentName,
                'program' => $student['program'],
                'user_group' => $student['user_group'],
                'report_score' => null,
                'presentation_score' => $finalScore,
                'final_score' => $finalScore,
                'letter_grade' => $letterGrade
            ];
        } else {
            // Get all grades for this student (both individual and 'both' type)
            $stmtGrades = $pdo->prepare("
                SELECT psg.*, l.name AS lecturer_name
                FROM pi_student_grades psg
                JOIN lecturer_login l ON psg.lecturer_id = l.id
                WHERE psg.student_id = ? AND psg.grading_type IN ('report', 'presentation', 'both') AND psg.lecturer_id != 999999
                ORDER BY psg.graded_at DESC
            ");
            $stmtGrades->execute([$student['student_id']]);
            $allGrades = $stmtGrades->fetchAll(PDO::FETCH_ASSOC);
            
            // Prioritize 'both' type grades if they exist, as they contain complete data
            $bothTypeGrades = array_filter($allGrades, function($grade) {
                return $grade['grading_type'] === 'both';
            });
            
            if (!empty($bothTypeGrades)) {
                // Use the most recent 'both' type grade
                $grades = [reset($bothTypeGrades)];
            } else {
                // Fall back to individual grades
                $grades = array_filter($allGrades, function($grade) {
                    return in_array($grade['grading_type'], ['report', 'presentation']);
                });
            }
            
            if (!empty($grades)) {
                // More robust score extraction and calculation
                $reportScoresForStudent = [];
                $presentationScoresForStudent = [];
                
                foreach ($grades as $grade) {
                    // Handle report scores
                    if (isset($grade['report_score']) && $grade['report_score'] !== null && $grade['report_score'] !== '') {
                        $score = floatval($grade['report_score']);
                        if ($score > 0) {
                            $reportScoresForStudent[] = $score;
                        }
                    }
                    
                    // Handle presentation scores
                    if (isset($grade['presentation_score']) && $grade['presentation_score'] !== null && $grade['presentation_score'] !== '') {
                        $score = floatval($grade['presentation_score']);
                        if ($score > 0) {
                            $presentationScoresForStudent[] = $score;
                        }
                    }
                }
                
                $reportScore = count($reportScoresForStudent) > 0 ? 
                    round(array_sum($reportScoresForStudent) / count($reportScoresForStudent), 2) : 0;
                
                $presentationScore = count($presentationScoresForStudent) > 0 ? 
                    round(array_sum($presentationScoresForStudent) / count($presentationScoresForStudent), 2) : 0;
                
                // Skip students who don't have BOTH report AND presentation scores
                // This ensures we only process students with complete grading data
                // Check if we actually have valid scores (not just 0 values)
                $hasValidReportScore = !empty($reportScoresForStudent) && $reportScore > 0;
                $hasValidPresentationScore = !empty($presentationScoresForStudent) && $presentationScore > 0;
                
                if (!$hasValidReportScore || !$hasValidPresentationScore) {
                    continue; // Skip this student - leave them completely untouched
                }
                
                $finalScore = round(($reportScore + $presentationScore), 2);
                $letterGrade = getLetterGrade($finalScore);
                
                if ($reportScore > 0) $reportScores[] = $reportScore;
                if ($presentationScore > 0) $presentationScores[] = $presentationScore;
                if ($finalScore > 0) {
                    $finalScores[] = $finalScore;
                    updateGradeDistribution($gradeDistribution, $letterGrade);
                }
                
                $studentResults[] = [
                    'student_id' => $student['student_id'],
                    'student_name' => $studentName,
                    'program' => $student['program'],
                    'user_group' => $student['user_group'],
                    'report_score' => $reportScore,
                    'presentation_score' => $presentationScore,
                    'final_score' => $finalScore,
                    'letter_grade' => $letterGrade
                ];
            } else {
                // Skip students without grades - don't include them in results
                // This leaves them completely untouched in the database
                continue;
            }
        }
    }
    
    $reportSummary = [
        'count' => count($reportScores),
        'total' => array_sum($reportScores),
        'average' => count($reportScores) > 0 ? round(array_sum($reportScores) / count($reportScores), 2) : 0
    ];
    
    $presentationSummary = [
        'count' => count($presentationScores),
        'total' => array_sum($presentationScores),
        'average' => count($presentationScores) > 0 ? round(array_sum($presentationScores) / count($presentationScores), 2) : 0
    ];
    
    $finalSummary = [
        'count' => count($finalScores),
        'total' => array_sum($finalScores),
        'average' => count($finalScores) > 0 ? round(array_sum($finalScores) / count($finalScores), 2) : 0
    ];
    
    echo json_encode([
        'success' => true,
        'total_students' => count($students),
        'students_with_grades' => count($studentResults),
        'students_without_grades' => count($students) - count($studentResults),
        'report_summary' => $reportSummary,
        'presentation_summary' => $presentationSummary,
        'final_summary' => $finalSummary,
        'grade_distribution' => $gradeDistribution,
        'student_results' => $studentResults
    ]);
    
} catch (Exception $e) {
    echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
}

function getLetterGrade($score) {
    if ($score >= 80) return 'A';
    elseif ($score >= 75) return 'B+';
    elseif ($score >= 70) return 'B';
    elseif ($score >= 65) return 'C+';
    elseif ($score >= 60) return 'C';
    elseif ($score >= 55) return 'D+';
    elseif ($score >= 50) return 'D';
    else return 'E';
}

function updateGradeDistribution(&$distribution, $grade) {
    switch ($grade) {
        case 'A': $distribution['A']++; break;
        case 'B+': $distribution['B_plus']++; break;
        case 'B': $distribution['B']++; break;
        case 'C+': $distribution['C_plus']++; break;
        case 'C': $distribution['C']++; break;
        case 'D':
        case 'E': $distribution['D_E']++; break;
    }
}
?>
