<?php
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
if (!isset($_SESSION['admin_id'])) {
    header('Location: login');
    exit;
}
include 'admin_header.php';
include '../config/database.php';

// Fetch all lecturers with their preferences and assigned districts
$query = '
    SELECT ll.id, ll.name, lp.preferred_region, 
           (SELECT GROUP_CONCAT(district SEPARATOR ", ") 
            FROM lecturer_assigned_districts lad 
            WHERE lad.lecturer_id = ll.id AND lad.region = lp.preferred_region) AS assigned_districts
    FROM lecturer_login ll
    JOIN lecturer_preferences lp ON ll.id = lp.lecturer_id
';
$stmt = $pdo->prepare($query);
$stmt->execute();
$lecturers = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch distinct regions for the filter
$query = 'SELECT DISTINCT preferred_region FROM lecturer_preferences';
$stmt = $pdo->prepare($query);
$stmt->execute();
$regions = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch distinct districts for the filter
$query = 'SELECT DISTINCT district FROM lecturer_assigned_districts';
$stmt = $pdo->prepare($query);
$stmt->execute();
$districts = $stmt->fetchAll(PDO::FETCH_ASSOC);

$regionFilter = isset($_GET['region_filter']) ? htmlspecialchars($_GET['region_filter'], ENT_QUOTES, 'UTF-8') : '';
$districtFilter = isset($_GET['district_filter']) ? htmlspecialchars($_GET['district_filter'], ENT_QUOTES, 'UTF-8') : '';
if ($regionFilter || $districtFilter) {
    $query = '
        SELECT ll.id, ll.name, lp.preferred_region, 
               (SELECT GROUP_CONCAT(district SEPARATOR ", ") 
                FROM lecturer_assigned_districts lad 
                WHERE lad.lecturer_id = ll.id AND lad.region = lp.preferred_region) AS assigned_districts
        FROM lecturer_login ll
        JOIN lecturer_preferences lp ON ll.id = lp.lecturer_id
        WHERE 1=1
    ';
    if ($regionFilter) {
        $query .= ' AND lp.preferred_region = :region_filter';
    }
    if ($districtFilter) {
        $query .= ' AND EXISTS (SELECT 1 FROM lecturer_assigned_districts lad WHERE lad.lecturer_id = ll.id AND lad.district = :district_filter)';
    }
    $stmt = $pdo->prepare($query);
    if ($regionFilter) {
        $stmt->bindParam(':region_filter', $regionFilter);
    }
    if ($districtFilter) {
        $stmt->bindParam(':district_filter', $districtFilter);
    }
    $stmt->execute();
    $lecturers = $stmt->fetchAll(PDO::FETCH_ASSOC);
}
?>

<div class="container mt-4">
    <h2>Lecturer Preferences</h2>
    <form method="GET" action="manage_lecturers.php" class="mb-4">
        <div class="row">
            <div class="col-md-6">
                <select name="region_filter" class="form-select" onchange="this.form.submit()">
                    <option value="">All Regions</option>
                    <?php foreach ($regions as $region): ?>
                        <option value="<?php echo htmlspecialchars($region['preferred_region'], ENT_QUOTES, 'UTF-8'); ?>" <?php echo $regionFilter === $region['preferred_region'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($region['preferred_region'], ENT_QUOTES, 'UTF-8'); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-6">
                <select name="district_filter" class="form-select" onchange="this.form.submit()">
                    <option value="">All Districts</option>
                    <?php foreach ($districts as $district): ?>
                        <option value="<?php echo htmlspecialchars($district['district'], ENT_QUOTES, 'UTF-8'); ?>" <?php echo $districtFilter === $district['district'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($district['district'], ENT_QUOTES, 'UTF-8'); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-6 text-end mt-2">
                <button type="submit" class="btn btn-outline-custom">Filter</button>
            </div>
        </div>
    </form>
    <table class="table table-striped table-hover">
        <thead>
            <tr>
                <th>Name</th>
                <th>Preferred Region</th>
                <th>Assigned Districts</th>
                <th>Action</th>
            </tr>
        </thead>
        <tbody>
            <?php if (!empty($lecturers)): ?>
                <?php foreach ($lecturers as $lecturer): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($lecturer['name'], ENT_QUOTES, 'UTF-8'); ?></td>
                        <td><?php echo htmlspecialchars($lecturer['preferred_region'], ENT_QUOTES, 'UTF-8'); ?></td>
                        <td><?php echo htmlspecialchars($lecturer['assigned_districts'] ?: 'No districts assigned', ENT_QUOTES, 'UTF-8'); ?></td>
                        <td>
                            <button class="btn btn-outline-custom assign-districts" data-id="<?php echo htmlspecialchars($lecturer['id'], ENT_QUOTES, 'UTF-8'); ?>" data-name="<?php echo htmlspecialchars($lecturer['name'], ENT_QUOTES, 'UTF-8'); ?>" data-region="<?php echo htmlspecialchars($lecturer['preferred_region'], ENT_QUOTES, 'UTF-8'); ?>">Assign Districts</button>
                        </td>
                    </tr>
                <?php endforeach; ?>
            <?php else: ?>
                <tr>
                    <td colspan="4">No lecturers found.</td>
                </tr>
            <?php endif; ?>
        </tbody>
    </table>
</div>

<!-- Assign Districts Modal -->
<div class="modal fade" id="assignDistrictsModal" tabindex="-1" aria-labelledby="assignDistrictsModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="assignDistrictsModalLabel">Assign Districts</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <form id="assignDistrictsForm" method="POST" action="assign_districts.php">
                    <input type="hidden" name="lecturer_id" id="lecturerId">
                    <input type="hidden" name="region" id="region">
                    <div class="mb-3">
                        <label for="districts" class="form-label">Select Districts</label>
                        <select class="form-select" id="districts" name="districts">
                            <!-- District options will be populated dynamically -->
                        </select>
                    </div>
                    <button type="button" id="addDistrict" class="btn btn-outline-custom">Add District</button>

                    <h3 class="mt-4">Selected Districts</h3>
                    <table class="table table-bordered" id="selectedDistrictsTable">
                        <thead>
                            <tr>
                                <th>District</th>
                                <th>Action</th>
                            </tr>
                        </thead>
                        <tbody>
                            <!-- Selected districts will be added here dynamically -->
                        </tbody>
                    </table>

                    <button type="submit" class="btn btn-outline-custom">Save Changes</button>
                </form>
            </div>
        </div>
    </div>
</div>

<?php include 'admin_footer.php'; ?>

<script>
document.querySelectorAll('.assign-districts').forEach(button => {
    button.addEventListener('click', function() {
        const lecturerId = this.getAttribute('data-id');
        const lecturerName = this.getAttribute('data-name');
        const preferredRegion = this.getAttribute('data-region');

        // Set the values in the modal form
        document.getElementById('lecturerId').value = lecturerId;
        document.getElementById('region').value = preferredRegion;
        document.getElementById('assignDistrictsModalLabel').textContent = 'Assign Districts to ' + lecturerName;

        // Fetch districts for the preferred region
        fetch('fetch_districts.php?region=' + encodeURIComponent(preferredRegion))
            .then(response => response.json())
            .then(data => {
                const districtsSelect = document.getElementById('districts');
                districtsSelect.innerHTML = '';
                data.forEach(district => {
                    const option = document.createElement('option');
                    option.value = district.district;
                    option.textContent = district.district;
                    districtsSelect.appendChild(option);
                });

                // Fetch existing assigned districts
                fetch('fetch_assigned_districts.php?lecturer_id=' + encodeURIComponent(lecturerId) + '&region=' + encodeURIComponent(preferredRegion))
                    .then(response => response.json())
                    .then(data => {
                        const tableBody = document.querySelector('#selectedDistrictsTable tbody');
                        tableBody.innerHTML = '';
                        data.forEach(district => {
                            const row = document.createElement('tr');
                            row.innerHTML = `
                                <td>${district.district}</td>
                                <td><button type="button" class="btn btn-outline-danger btn-sm remove-district">Remove</button></td>
                                <input type="hidden" name="selected_districts[]" value="${district.district}">
                            `;
                            tableBody.appendChild(row);

                            // Add event listener to remove button
                            row.querySelector('.remove-district').addEventListener('click', function() {
                                row.remove();
                            });
                        });
                    });

                // Show the modal
                const assignDistrictsModal = new bootstrap.Modal(document.getElementById('assignDistrictsModal'));
                assignDistrictsModal.show();
            });
    });
});

document.getElementById('addDistrict').addEventListener('click', function() {
    const districtsSelect = document.getElementById('districts');
    const selectedDistrict = districtsSelect.value;

    if (selectedDistrict) {
        const tableBody = document.querySelector('#selectedDistrictsTable tbody');
        const row = document.createElement('tr');
        row.innerHTML = `
            <td>${selectedDistrict}</td>
            <td><button type="button" class="btn btn-outline-danger btn-sm remove-district">Remove</button></td>
            <input type="hidden" name="selected_districts[]" value="${selectedDistrict}">
        `;
        tableBody.appendChild(row);

        // Add event listener to remove button
        row.querySelector('.remove-district').addEventListener('click', function() {
            row.remove();
        });
    }
});
</script>
