<?php
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
if (!isset($_SESSION['admin_id'])) {
    header('Location: login');
    exit;
}
include 'admin_header.php';
include '../config/database.php';

// Fetch distinct values for filters
$stmt = $pdo->prepare('SELECT DISTINCT region FROM districts');
$stmt->execute();
$regions = $stmt->fetchAll(PDO::FETCH_ASSOC);

$stmt = $pdo->prepare('SELECT DISTINCT course FROM student_details');
$stmt->execute();
$courses = $stmt->fetchAll(PDO::FETCH_ASSOC);

$stmt = $pdo->prepare('SELECT DISTINCT organisation_name FROM student_details');
$stmt->execute();
$organisations = $stmt->fetchAll(PDO::FETCH_ASSOC);

$types = ['Industrial Internship', 'Teaching Practice'];

// Fetch students based on selected filters
$selectedRegion = isset($_GET['region']) ? htmlspecialchars($_GET['region'], ENT_QUOTES, 'UTF-8') : '';
$selectedCourse = isset($_GET['course']) ? htmlspecialchars($_GET['course'], ENT_QUOTES, 'UTF-8') : '';
$selectedOrganisation = isset($_GET['organisation']) ? htmlspecialchars($_GET['organisation'], ENT_QUOTES, 'UTF-8') : '';
$selectedType = isset($_GET['type']) ? htmlspecialchars($_GET['type'], ENT_QUOTES, 'UTF-8') : 'Industrial Internship';

$query = 'SELECT sd.*, u.index_number FROM student_details sd JOIN users u ON sd.user_id = u.id WHERE u.group = :type';
$conditions = [];

if ($selectedRegion) {
    $conditions[] = 'sd.region = :region';
}
if ($selectedCourse) {
    $conditions[] = 'sd.course = :course';
}
if ($selectedOrganisation) {
    $conditions[] = 'sd.organisation_name = :organisation';
}

if ($conditions) {
    $query .= ' AND ' . implode(' AND ', $conditions);
}

$stmt = $pdo->prepare($query);
$stmt->bindParam(':type', $selectedType);

if ($selectedRegion) {
    $stmt->bindParam(':region', $selectedRegion);
}
if ($selectedCourse) {
    $stmt->bindParam(':course', $selectedCourse);
}
if ($selectedOrganisation) {
    $stmt->bindParam(':organisation', $selectedOrganisation);
}

$stmt->execute();
$students = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Function to generate correct file path
function getFilePath($filename) {
    return '../uploads/' . htmlspecialchars($filename, ENT_QUOTES, 'UTF-8'); // Ensure the path is relative to the uploads directory
}
?>

<div class="container mt-4">
    <h2>Internships</h2>
    <form method="GET" action="index.php" class="mb-4">
        <div class="row">
            <div class="col-md-3">
                <select name="type" class="form-select">
                    <option value="">Select Type</option>
                    <?php foreach ($types as $type): ?>
                        <option value="<?php echo htmlspecialchars($type, ENT_QUOTES, 'UTF-8'); ?>" <?php echo $selectedType === $type ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($type, ENT_QUOTES, 'UTF-8'); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-3">
                <select name="region" id="region" class="form-select">
                    <option value="">Select Region</option>
                    <?php foreach ($regions as $region): ?>
                        <option value="<?php echo htmlspecialchars($region['region'], ENT_QUOTES, 'UTF-8'); ?>" <?php echo $selectedRegion === $region['region'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($region['region'], ENT_QUOTES, 'UTF-8'); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-3">
                <select name="course" class="form-select">
                    <option value="">Select Course</option>
                    <?php foreach ($courses as $course): ?>
                        <option value="<?php echo htmlspecialchars($course['course'], ENT_QUOTES, 'UTF-8'); ?>" <?php echo $selectedCourse === $course['course'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($course['course'], ENT_QUOTES, 'UTF-8'); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-3">
                <select name="organisation" class="form-select">
                    <option value="">Select Organisation</option>
                    <?php foreach ($organisations as $organisation): ?>
                        <option value="<?php echo htmlspecialchars($organisation['organisation_name'], ENT_QUOTES, 'UTF-8'); ?>" <?php echo $selectedOrganisation === $organisation['organisation_name'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($organisation['organisation_name'], ENT_QUOTES, 'UTF-8'); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
        </div>
        <div class="row mt-3">
            <div class="col-md-3">
                <button class="btn btn-outline-custom" type="submit">Filter</button>
            </div>
        </div>
    </form>

    <?php if (count($students) > 0): ?>
    <table class="table table-striped">
        <thead>
            <tr>
                <th><a href="?sort=name">Name <i class="fas fa-sort"></i></a></th>
                <th><a href="?sort=organisation">Organisation <i class="fas fa-sort"></i></a></th>
                <th><a href="?sort=district">District <i class="fas fa-sort"></i></a></th>
                <th><a href="?sort=region">Region <i class="fas fa-sort"></i></a></th>
                <th>Action</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($students as $student): ?>
                <tr>
                    <td><?php echo htmlspecialchars($student['first_name'] . ' ' . $student['surname'], ENT_QUOTES, 'UTF-8'); ?></td>
                    <td><?php echo htmlspecialchars($student['organisation_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                    <td><?php echo htmlspecialchars($student['district'], ENT_QUOTES, 'UTF-8'); ?></td>
                    <td><?php echo htmlspecialchars($student['region'], ENT_QUOTES, 'UTF-8'); ?></td>
                    <td>
                        <button class="btn btn-outline-info btn-sm view-student" data-id="<?php echo $student['id']; ?>" data-index="<?php echo htmlspecialchars($student['index_number'], ENT_QUOTES, 'UTF-8'); ?>" data-name="<?php echo htmlspecialchars($student['first_name'] . ' ' . $student['surname'], ENT_QUOTES, 'UTF-8'); ?>" data-organisation="<?php echo htmlspecialchars($student['organisation_name'], ENT_QUOTES, 'UTF-8'); ?>" data-district="<?php echo htmlspecialchars($student['district'], ENT_QUOTES, 'UTF-8'); ?>" data-region="<?php echo htmlspecialchars($student['region'], ENT_QUOTES, 'UTF-8'); ?>" data-telephone="<?php echo htmlspecialchars($student['telephone'], ENT_QUOTES, 'UTF-8'); ?>" data-email="<?php echo htmlspecialchars($student['email'], ENT_QUOTES, 'UTF-8'); ?>" data-course="<?php echo htmlspecialchars($student['course'], ENT_QUOTES, 'UTF-8'); ?>" data-months="<?php echo htmlspecialchars($student['months'], ENT_QUOTES, 'UTF-8'); ?>" data-location="<?php echo htmlspecialchars($student['location'], ENT_QUOTES, 'UTF-8'); ?>" data-gps="<?php echo htmlspecialchars($student['gps_location'], ENT_QUOTES, 'UTF-8'); ?>" data-letter="<?php echo getFilePath($student['acceptance_letter']); ?>">View</button>
                    </td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
    <?php else: ?>
        <p>No students found for the selected filters.</p>
    <?php endif; ?>
</div>

<!-- Student Details Modal -->
<div class="modal fade" id="studentModal" tabindex="-1" aria-labelledby="studentModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="studentModalLabel">Student Details</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <p><strong>Index Number:</strong> <span id="studentIndex"></span></p>
                <p><strong>Name:</strong> <span id="studentName"></span></p>
                <p><strong>Telephone:</strong> <span id="studentTelephone"></span></p>
                <p><strong>Email:</strong> <span id="studentEmail"></span></p>
                <p><strong>Course:</strong> <span id="studentCourse"></span></p>
                <p><strong>Organisation:</strong> <span id="studentOrganisation"></span></p>
                <p><strong>Months:</strong> <span id="studentMonths"></span></p>
                <p><strong>Location:</strong> <span id="studentLocation"></span></p>
                <p><strong>GPS Location:</strong> <span id="studentGps"></span></p>
                <p><strong>Region:</strong> <span id="studentRegion"></span></p>
                <p><strong>District:</strong> <span id="studentDistrict"></span></p>
                <p><strong>Acceptance Letter:</strong> <a class="btn btn-outline-info btn-sm" id="studentLetter" target="_blank">View</a></p>
            </div>
        </div>
    </div>
</div>

<script>
    document.addEventListener('DOMContentLoaded', function () {
        const studentModal = new bootstrap.Modal(document.getElementById('studentModal'));

        document.querySelectorAll('.view-student').forEach(item => {
            item.addEventListener('click', event => {
                document.getElementById('studentIndex').textContent = event.currentTarget.getAttribute('data-index');
                document.getElementById('studentName').textContent = event.currentTarget.getAttribute('data-name');
                document.getElementById('studentOrganisation').textContent = event.currentTarget.getAttribute('data-organisation');
                document.getElementById('studentDistrict').textContent = event.currentTarget.getAttribute('data-district');
                document.getElementById('studentRegion').textContent = event.currentTarget.getAttribute('data-region');
                document.getElementById('studentTelephone').textContent = event.currentTarget.getAttribute('data-telephone');
                document.getElementById('studentEmail').textContent = event.currentTarget.getAttribute('data-email');
                document.getElementById('studentCourse').textContent = event.currentTarget.getAttribute('data-course');
                document.getElementById('studentMonths').textContent = event.currentTarget.getAttribute('data-months');
                document.getElementById('studentLocation').textContent = event.currentTarget.getAttribute('data-location');
                document.getElementById('studentGps').textContent = event.currentTarget.getAttribute('data-gps');
                document.getElementById('studentLetter').setAttribute('href', event.currentTarget.getAttribute('data-letter'));
                studentModal.show();
            });
        });

        // Add sorting functionality
        const urlParams = new URLSearchParams(window.location.search);
        const sortParam = urlParams.get('sort');

        if (sortParam) {
            const sortElements = document.querySelectorAll('th a');
            sortElements.forEach(el => {
                if (el.href.includes(`sort=${sortParam}`)) {
                    el.querySelector('i').classList.add('fa-sort-up');
                }
            });

            const sortedData = [...document.querySelectorAll('tbody tr')].sort((a, b) => {
                const aValue = a.querySelector(`td:nth-child(${getColumnIndex(sortParam)})`).textContent.trim();
                const bValue = b.querySelector(`td:nth-child(${getColumnIndex(sortParam)})`).textContent.trim();
                return aValue.localeCompare(bValue);
            });

            const tbody = document.querySelector('tbody');
            tbody.innerHTML = '';
            sortedData.forEach(row => {
                tbody.appendChild(row);
            });
        }

        function getColumnIndex(sortParam) {
            switch (sortParam) {
                case 'name':
                    return 1;
                case 'organisation':
                    return 2;
                case 'district':
                    return 3;
                case 'region':
                    return 4;
                default:
                    return 1;
            }
        }
    });
</script>

<?php include 'admin_footer.php'; ?>
