<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

session_start();
require '../config/database.php';

// Check if lecturer is logged in
if (!isset($_SESSION['lecturer_id'])) {
    echo json_encode(['status' => 'error', 'message' => 'Not authenticated.']);
    exit;
}

// Retrieve the raw POST data or fallback to $_POST
$data = json_decode(file_get_contents('php://input'), true);
if (!$data) {
    $data = $_POST;
}

// Validate required data
if (!isset($data['student_id'])) {
    echo json_encode(['status' => 'error', 'message' => 'Missing student_id']);
    exit;
}

$student_id = $data['student_id'];
$lecturer_id = $_SESSION['lecturer_id'];

// Get the report score (0-50 points)
$report_score = isset($data['report_score']) ? (int)$data['report_score'] : 0;
$report_comments = isset($data['report_comments']) ? trim($data['report_comments']) : '';

// Validate score range
if ($report_score < 0 || $report_score > 50) {
    echo json_encode(['status' => 'error', 'message' => 'Report score must be between 0 and 50.']);
    exit;
}

try {
    // Check if a grade record already exists for this student & lecturer
    $stmtCheck = $pdo->prepare("SELECT id FROM pi_student_grades WHERE student_id = ? AND lecturer_id = ?");
    $stmtCheck->execute([$student_id, $lecturer_id]);
    $existingRecord = $stmtCheck->fetch(PDO::FETCH_ASSOC);

    if ($existingRecord) {
        // Record exists, so update it
        $sql = "UPDATE pi_student_grades SET report_score = ?, report_comments = ?, grading_type = 'report' WHERE student_id = ? AND lecturer_id = ?";
        $stmt = $pdo->prepare($sql);
        $result = $stmt->execute([$report_score, $report_comments, $student_id, $lecturer_id]);
        $operation = "updated";
    } else {
        // No record exists; insert a new one
        $sql = "INSERT INTO pi_student_grades (student_id, lecturer_id, report_score, report_comments, grading_type) VALUES (?, ?, ?, ?, 'report')";
        $stmt = $pdo->prepare($sql);
        $result = $stmt->execute([$student_id, $lecturer_id, $report_score, $report_comments]);
        $operation = "submitted";
    }

    if ($result) {
        // Get the lecturer's group_id
        $stmtGroup = $pdo->prepare("SELECT group_id FROM lecturer_login WHERE id = ?");
        $stmtGroup->execute([$lecturer_id]);
        $group = $stmtGroup->fetch(PDO::FETCH_ASSOC);
        $group_id = $group['group_id'] ?? null;
        
        // Calculate the average report score from all lecturers in the group
        $average_report_score = 0;
        if ($group_id) {
            $stmtAvgReport = $pdo->prepare("
                SELECT AVG(psg.report_score) AS avg_report_score 
                FROM pi_student_grades psg
                JOIN lecturer_login ll ON psg.lecturer_id = ll.id
                WHERE psg.student_id = :student_id
                  AND ll.group_id = :group_id
                  AND psg.report_score IS NOT NULL
                  AND psg.report_score > 0
            ");
            $stmtAvgReport->execute([
                'student_id' => $student_id,
                'group_id' => $group_id
            ]);
            $avgRowReport = $stmtAvgReport->fetch(PDO::FETCH_ASSOC);
            $average_report_score = $avgRowReport ? round($avgRowReport['avg_report_score'], 2) : 0;
        }
        
        // Update final score (for now, just the report score since we're focusing on report only)
        $final_score = $average_report_score;
        
        // Update the current record with the final calculated score
        $stmtUpdateFinal = $pdo->prepare("
            UPDATE pi_student_grades 
            SET final_score = :final_score 
            WHERE student_id = :student_id AND lecturer_id = :lecturer_id
        ");
        $stmtUpdateFinal->execute([
            'final_score' => $final_score,
            'student_id' => $student_id,
            'lecturer_id' => $lecturer_id
        ]);
        
        // --- Set grading_type based on which grades are present ---
        $stmtCheckBoth = $pdo->prepare("SELECT report_score, presentation_score FROM pi_student_grades WHERE student_id = ? AND lecturer_id = ?");
        $stmtCheckBoth->execute([$student_id, $lecturer_id]);
        $row = $stmtCheckBoth->fetch(PDO::FETCH_ASSOC);
        if ($row) {
            if (!empty($row['report_score']) && $row['report_score'] > 0 && !empty($row['presentation_score']) && $row['presentation_score'] > 0) {
                $stmtUpdateType = $pdo->prepare("UPDATE pi_student_grades SET grading_type = 'both' WHERE student_id = ? AND lecturer_id = ?");
                $stmtUpdateType->execute([$student_id, $lecturer_id]);
            } else if (!empty($row['report_score']) && $row['report_score'] > 0) {
                $stmtUpdateType = $pdo->prepare("UPDATE pi_student_grades SET grading_type = 'report' WHERE student_id = ? AND lecturer_id = ?");
                $stmtUpdateType->execute([$student_id, $lecturer_id]);
            } else if (!empty($row['presentation_score']) && $row['presentation_score'] > 0) {
                $stmtUpdateType = $pdo->prepare("UPDATE pi_student_grades SET grading_type = 'presentation' WHERE student_id = ? AND lecturer_id = ?");
                $stmtUpdateType->execute([$student_id, $lecturer_id]);
            }
        }
        
        echo json_encode([
            'status' => 'success', 
            'message' => "Report grade $operation successfully.",
            'report_score' => $report_score,
            'average_report_score' => $average_report_score,
            'final_score' => $final_score
        ]);
    } else {
        $errorInfo = $stmt->errorInfo();
        echo json_encode(['status' => 'error', 'message' => 'Error submitting report grade.', 'errorInfo' => $errorInfo]);
    }
} catch (Exception $e) {
    echo json_encode(['status' => 'error', 'message' => 'Exception: ' . $e->getMessage()]);
}
?> 