<?php
session_start();
require '../config/database.php';

// Check if lecturer is logged in
if (!isset($_SESSION['lecturer_id'])) {
    header('Location: login.php');
    exit;
}

// Retrieve the raw POST data or fallback to $_POST
$data = json_decode(file_get_contents('php://input'), true);
if (!$data) {
    $data = $_POST;
}

// Ensure required fields exist
if (!isset($data['student_id']) || !isset($_SESSION['lecturer_id'])) {
    echo json_encode(['status' => 'error', 'message' => 'Missing required fields.']);
    exit;
}

$student_id = $data['student_id'];
$lecturer_id = $_SESSION['lecturer_id'];

// Define the presentation grading fields
$presentationGradingFields = [
    'coherence',
    'contents',
    'presentation_etiquette',
    'ability_to_answer_questions',
    'time_awareness'
];

// Build parameters array and calculate the presentation score
$params = [
    'student_id' => $student_id,
    'lecturer_id' => $lecturer_id,
];

$presentation_score = 0;
foreach ($presentationGradingFields as $field) {
    $value = isset($data[$field]) ? (int)$data[$field] : 0;
    $params[$field] = $value;
    $presentation_score += $value;
}

$params['presentation_score'] = $presentation_score;
$params['presentation_comments'] = isset($data['presentation_comments']) ? trim($data['presentation_comments']) : '';

// Build the SET clause for presentation grading fields
$setClause = "";
foreach ($presentationGradingFields as $field) {
    $setClause .= "$field = :$field, ";
}
$setClause .= "presentation_comments = :presentation_comments, presentation_score = :presentation_score";

// Check if a grade record already exists for this student & lecturer
$stmtCheck = $pdo->prepare("SELECT id FROM pi_student_grades WHERE student_id = ? AND lecturer_id = ?");
$stmtCheck->execute([$student_id, $lecturer_id]);
$existingRecord = $stmtCheck->fetch(PDO::FETCH_ASSOC);

if ($existingRecord) {
    // Record exists, so update it
    $updateParams = [];
    foreach ($presentationGradingFields as $field) {
        $updateParams[$field] = $params[$field];
    }
    $updateParams['presentation_score'] = $presentation_score;
    $updateParams['presentation_comments'] = $params['presentation_comments'];
    $updateParams['id'] = $existingRecord['id'];
    
    $sql = "UPDATE pi_student_grades SET $setClause WHERE id = :id";
    $stmt = $pdo->prepare($sql);
    $operation = "updated";
} else {
    // No record exists; insert a new one
    $sql = "INSERT INTO pi_student_grades SET student_id = :student_id, lecturer_id = :lecturer_id, $setClause";
    $stmt = $pdo->prepare($sql);
    $operation = "submitted";
}

if ($existingRecord) {
    $result = $stmt->execute($updateParams);
} else {
    $result = $stmt->execute($params);
}

if ($result) {
    // Calculate the average report score and average presentation score for the student from all lecturers in the group who submitted each grade
    $average_report_score = 0;
    $average_presentation_score = 0;
    // Get the lecturer's group_id
    $stmtGroup = $pdo->prepare("SELECT group_id FROM lecturer_login WHERE id = ?");
    $stmtGroup->execute([$lecturer_id]);
    $group = $stmtGroup->fetch(PDO::FETCH_ASSOC);
    $group_id = $group['group_id'] ?? null;
    if ($group_id) {
        // Average report score (only non-null, >0)
        $stmtAvgReport = $pdo->prepare("
            SELECT AVG(psg.report_score) AS avg_report_score 
            FROM pi_student_grades psg
            JOIN lecturer_login ll ON psg.lecturer_id = ll.id
            WHERE psg.student_id = :student_id
              AND ll.group_id = :group_id
              AND psg.report_score IS NOT NULL
              AND psg.report_score > 0
        ");
        $stmtAvgReport->execute([
            'student_id' => $student_id,
            'group_id' => $group_id
        ]);
        $avgRowReport = $stmtAvgReport->fetch(PDO::FETCH_ASSOC);
        $average_report_score = $avgRowReport ? round($avgRowReport['avg_report_score'], 2) : 0;

        // Average presentation score (only non-null, >0)
        $stmtAvgPres = $pdo->prepare("
            SELECT AVG(psg.presentation_score) AS avg_presentation_score 
            FROM pi_student_grades psg
            JOIN lecturer_login ll ON psg.lecturer_id = ll.id
            WHERE psg.student_id = :student_id
              AND ll.group_id = :group_id
              AND psg.presentation_score IS NOT NULL
              AND psg.presentation_score > 0
        ");
        $stmtAvgPres->execute([
            'student_id' => $student_id,
            'group_id' => $group_id
        ]);
        $avgRowPres = $stmtAvgPres->fetch(PDO::FETCH_ASSOC);
        $average_presentation_score = $avgRowPres ? round($avgRowPres['avg_presentation_score'], 2) : 0;
    }
    // Calculate and update final score
    $final_score = $average_report_score + $average_presentation_score;
    $stmtUpdateFinal = $pdo->prepare("
        UPDATE pi_student_grades 
        SET final_score = :final_score 
        WHERE student_id = :student_id
    ");
    $stmtUpdateFinal->execute([
        'final_score' => $final_score,
        'student_id' => $student_id
    ]);
    // --- Set grading_type based on which grades are present ---
    $stmtCheckBoth = $pdo->prepare("SELECT report_score, presentation_score FROM pi_student_grades WHERE student_id = ? AND lecturer_id = ?");
    $stmtCheckBoth->execute([$student_id, $lecturer_id]);
    $row = $stmtCheckBoth->fetch(PDO::FETCH_ASSOC);
    if ($row) {
        if (!empty($row['report_score']) && $row['report_score'] > 0 && !empty($row['presentation_score']) && $row['presentation_score'] > 0) {
            $stmtUpdateType = $pdo->prepare("UPDATE pi_student_grades SET grading_type = 'both' WHERE student_id = ? AND lecturer_id = ?");
            $stmtUpdateType->execute([$student_id, $lecturer_id]);
        } else if (!empty($row['report_score']) && $row['report_score'] > 0) {
            $stmtUpdateType = $pdo->prepare("UPDATE pi_student_grades SET grading_type = 'report' WHERE student_id = ? AND lecturer_id = ?");
            $stmtUpdateType->execute([$student_id, $lecturer_id]);
        } else if (!empty($row['presentation_score']) && $row['presentation_score'] > 0) {
            $stmtUpdateType = $pdo->prepare("UPDATE pi_student_grades SET grading_type = 'presentation' WHERE student_id = ? AND lecturer_id = ?");
            $stmtUpdateType->execute([$student_id, $lecturer_id]);
        }
    }
    
    echo json_encode([
        'status' => 'success', 
        'message' => "Presentation grade $operation successfully.",
        'presentation_score' => $presentation_score,
        'final_score' => $final_score
    ]);
} else {
    echo json_encode(['status' => 'error', 'message' => 'Error submitting presentation grade.']);
}
?> 