<?php
// File: submit_grade.php
session_start();
require '../config/database.php'; // Adjust the path as needed

// Retrieve the raw POST data (assuming JSON via AJAX) or fallback to $_POST
$data = json_decode(file_get_contents('php://input'), true);
if (!$data) {
    $data = $_POST;
}

// Ensure required fields exist
if (!isset($data['student_id']) || !isset($_SESSION['lecturer_id'])) {
    echo json_encode(['status' => 'error', 'message' => 'Missing required fields.']);
    exit;
}

$student_id  = $data['student_id'];
$lecturer_id = $_SESSION['lecturer_id'];

// Define the grading fields as per the pi_student_grades table
$gradingFields = [
    'introduction_to_presentation',
    'logical_presentation',
    'activities_during_internship',
    'achievements_during_internship',
    'challenges_faced',
    'overcoming_challenges',
    'recommendations',
    'eye_contact',
    'audible_voice',
    'communication',
    'time_management'
];

// Build parameters array and calculate the total score
$params = [
    'student_id'  => $student_id,
    'lecturer_id' => $lecturer_id,
];
$total_score = 0;
foreach ($gradingFields as $field) {
    // Use a default value of 0 if not provided
    $value = isset($data[$field]) ? (int)$data[$field] : 0;
    $params[$field] = $value;
    $total_score += $value;
}
$params['total_score'] = $total_score;
$params['comments'] = isset($data['comments']) ? trim($data['comments']) : '';

// Build the SET clause for grading fields (used for both INSERT and UPDATE)
$setClause = "";
foreach ($gradingFields as $field) {
    $setClause .= "$field = :$field, ";
}
$setClause .= "comments = :comments, total_score = :total_score";

// First, check if a grade record already exists for this student & lecturer
$stmtCheck = $pdo->prepare("SELECT id FROM pi_student_grades WHERE student_id = ? AND lecturer_id = ?");
$stmtCheck->execute([$student_id, $lecturer_id]);
$existingRecord = $stmtCheck->fetch(PDO::FETCH_ASSOC);

if ($existingRecord) {
    // Record exists, so update it.
    // Build a new parameter array for the update (only include keys used in the SET clause plus :id)
    $updateParams = [];
    foreach ($gradingFields as $field) {
        $updateParams[$field] = $params[$field];
    }
    $updateParams['total_score'] = $total_score;
    $updateParams['comments'] = $params['comments'];
    $updateParams['id'] = $existingRecord['id'];
    
    $sql = "UPDATE pi_student_grades SET $setClause WHERE id = :id";
    $stmt = $pdo->prepare($sql);
    $operation = "updated";
} else {
    // No record exists; insert a new one.
    $sql = "INSERT INTO pi_student_grades SET student_id = :student_id, lecturer_id = :lecturer_id, $setClause";
    $stmt = $pdo->prepare($sql);
    $operation = "submitted";
}

if ($existingRecord) {
    $result = $stmt->execute($updateParams);
} else {
    $result = $stmt->execute($params);
}

if ($result) {
    // Retrieve current lecturer's group_id from lecturer_login table
    $stmtGroup = $pdo->prepare("SELECT group_id FROM lecturer_login WHERE id = ?");
    $stmtGroup->execute([$lecturer_id]);
    $group = $stmtGroup->fetch(PDO::FETCH_ASSOC);
    $group_id = (!empty($group['group_id'])) ? $group['group_id'] : null;
    
    $average_total = 0;
    if ($group_id) {
        // Compute the average total score for the student from all grade submissions by lecturers in this group.
        // Exclude records where all grading fields are zero (default/unsubmitted).
        $stmtAvg = $pdo->prepare("
            SELECT AVG(psg.total_score) AS avg_total 
            FROM pi_student_grades psg
            JOIN lecturer_login ll ON psg.lecturer_id = ll.id
            WHERE psg.student_id = :student_id
              AND ll.group_id = :group_id
              AND NOT (
                    psg.introduction_to_presentation = 0
                AND psg.logical_presentation = 0
                AND psg.activities_during_internship = 0
                AND psg.achievements_during_internship = 0
                AND psg.challenges_faced = 0
                AND psg.overcoming_challenges = 0
                AND psg.recommendations = 0
                AND psg.eye_contact = 0
                AND psg.audible_voice = 0
                AND psg.communication = 0
                AND psg.time_management = 0
              )
        ");
        $stmtAvg->execute([
            'student_id' => $student_id,
            'group_id'   => $group_id
        ]);
        $avgRow = $stmtAvg->fetch(PDO::FETCH_ASSOC);
        $average_total = $avgRow ? round($avgRow['avg_total'], 2) : 0;
    }
    
    echo json_encode([
        'status' => 'success', 
        'message' => "Grade $operation successfully.",
        'average_total' => $average_total
    ]);
} else {
    echo json_encode(['status' => 'error', 'message' => 'Error submitting grade.']);
}
?>
