<?php
// process_swap_decision.php
session_start();
header('Content-Type: application/json');

if (!isset($_SESSION['lecturer_id'])) {
    echo json_encode(['error' => 'User not logged in.']);
    exit;
}

require '../config/database.php';

$myLecturerId  = $_SESSION['lecturer_id'];
$swapRequestId = $_POST['swap_request_id'] ?? null;
$decision      = $_POST['decision'] ?? null;
$myStudentId   = $_POST['my_student_id'] ?? null;

if (!$swapRequestId || !$decision) {
    echo json_encode(['error' => 'Incomplete request.']);
    exit;
}

// Fetch the swap request
$stmt = $pdo->prepare("SELECT * FROM swap_requests WHERE id = ?");
$stmt->execute([$swapRequestId]);
$request = $stmt->fetch(PDO::FETCH_ASSOC);
if (!$request) {
    echo json_encode(['error' => 'Swap request not found.']);
    exit;
}

// Ensure the logged-in lecturer is the requested lecturer
if ($request['requested_lecturer_id'] != $myLecturerId) {
    echo json_encode(['error' => 'You are not authorized to respond to this request.']);
    exit;
}

if ($decision === 'decline') {
    $stmt = $pdo->prepare("UPDATE swap_requests SET status = 'declined' WHERE id = ?");
    $stmt->execute([$swapRequestId]);
    echo json_encode(['success' => 'Swap request declined.']);
    exit;
}

if ($decision === 'accept') {
    if (!$myStudentId) {
        echo json_encode(['error' => 'Please select your student to swap.']);
        exit;
    }
    // Verify that the selected student belongs to the logged-in lecturer
    $stmt = $pdo->prepare("SELECT lecturer_id FROM student_details WHERE id = ?");
    $stmt->execute([$myStudentId]);
    $myStudentLectId = $stmt->fetchColumn();
    if ($myStudentLectId != $myLecturerId) {
        echo json_encode(['error' => 'The selected student is not assigned to you.']);
        exit;
    }
    
    // Get IDs from the swap request:
    $requestingLectId    = $request['requesting_lecturer_id'];
    $requestingStudentId = $request['requesting_student_id'];
    
    try {
        $pdo->beginTransaction();
        
        // Update: Lecturer A's student becomes assigned to Lecturer B (logged in)
        $stmt = $pdo->prepare("UPDATE student_details SET lecturer_id = ? WHERE id = ?");
        $stmt->execute([$myLecturerId, $requestingStudentId]);
        
        // Update: Lecturer B's selected student becomes assigned to Lecturer A
        $stmt = $pdo->prepare("UPDATE student_details SET lecturer_id = ? WHERE id = ?");
        $stmt->execute([$requestingLectId, $myStudentId]);
        
        // Optionally: Update lecturer_assigned_districts for each lecturer
        // (Assumes that lecturer_assigned_districts table has a UNIQUE KEY on (lecturer_id, district, region))
        $stmtSel = $pdo->prepare("SELECT district, region FROM student_details WHERE id = ?");
        
        // For Lecturer B (logged in) now acquiring Lecturer A's student
        $stmtSel->execute([$requestingStudentId]);
        $studentA = $stmtSel->fetch(PDO::FETCH_ASSOC);
        if ($studentA) {
            $stmtDist = $pdo->prepare("
                INSERT INTO lecturer_assigned_districts (lecturer_id, district, region)
                VALUES (?, ?, ?)
                ON DUPLICATE KEY UPDATE district = VALUES(district), region = VALUES(region)
            ");
            $stmtDist->execute([$myLecturerId, $studentA['district'], $studentA['region']]);
        }
        
        // For Lecturer A (requesting lecturer) acquiring Lecturer B's selected student
        $stmtSel->execute([$myStudentId]);
        $studentB = $stmtSel->fetch(PDO::FETCH_ASSOC);
        if ($studentB) {
            $stmtDist->execute([$requestingLectId, $studentB['district'], $studentB['region']]);
        }
        
        // Mark the swap request as approved and record Lecturer B's selected student
        $stmt = $pdo->prepare("UPDATE swap_requests SET requested_student_id = ?, status = 'approved' WHERE id = ?");
        $stmt->execute([$myStudentId, $swapRequestId]);
        
        $pdo->commit();
        echo json_encode(['success' => 'Swap completed successfully and district assignments updated.']);
    } catch (Exception $e) {
        $pdo->rollBack();
        error_log("Swap failed: " . $e->getMessage());
        echo json_encode(['error' => 'Swap failed: ' . $e->getMessage()]);
    }
    exit;
}

echo json_encode(['error' => 'Invalid request decision.']);
exit;
?>
