<?php
session_start();
require '../config/database.php';

// Admin session check
if (!isset($_SESSION['admin_id'])) {
    header("Location: login");
    exit;
}

// Helper function to compute letter grade based on average score
function getLetterGrade($score)
{
    if ($score >= 80) {
        return 'A';
    } elseif ($score >= 75) {
        return 'B+';
    } elseif ($score >= 70) {
        return 'B';
    } elseif ($score >= 65) {
        return 'C+';
    } elseif ($score >= 60) {
        return 'C';
    } elseif ($score >= 55) {
        return 'D+';
    } elseif ($score >= 50) {
        return 'D';
    } else {
        return 'E';
    }
}

// Retrieve all lecturer groups
$stmtGroups = $pdo->query("SELECT * FROM pi_lecturer_groups ORDER BY group_name");
$groups = $stmtGroups->fetchAll(PDO::FETCH_ASSOC);

include('pi_header.php'); 
?>

<!-- Main Content -->
<div class="container mt-4">
    <h1>Student Scores by Group</h1>
    
    <?php if ($groups): ?>
        <?php foreach ($groups as $group): ?>
            <!-- Display Group Name -->
            <h2>Group: <?php echo htmlspecialchars($group['group_name']); ?></h2>
            
            <?php
            // Retrieve lecturers in this group
            $stmtLecturers = $pdo->prepare("SELECT id, name FROM lecturer_login WHERE group_id = ?");
            $stmtLecturers->execute([$group['id']]);
            $lecturers = $stmtLecturers->fetchAll(PDO::FETCH_ASSOC);
            
            if ($lecturers):
                // Create a comma-separated list of lecturer names
                $lecturerNames = array_map(function($lect) {
                    return $lect['name'];
                }, $lecturers);
                $lecturerNamesStr = implode(', ', $lecturerNames);
            ?>
                <div class="lecturers-list">
                    <strong>Lecturers:</strong> <?php echo htmlspecialchars($lecturerNamesStr); ?>
                </div>
            <?php else: ?>
                <p><em>No lecturers found for this group.</em></p>
            <?php endif; ?>
            
            <?php
            // Retrieve students assigned to any lecturer in this group.
            // Get lecturer IDs from this group.
            $lecturerIds = array_column($lecturers, 'id');
            if (!empty($lecturerIds)) {
                $inPlaceholder = rtrim(str_repeat('?,', count($lecturerIds)), ',');
                
                // Query to retrieve each student's details including their index number and average total score
                $query = "
                    SELECT
                        s.id AS student_id,
                        CONCAT(s.surname, ' ', s.first_name, ' ', COALESCE(s.other_names, '')) AS student_name,
                        u.index_number,
                        s.course AS program,
                        ROUND(AVG(psg.final_score), 2) AS average_score
                    FROM student_details s
                    JOIN users u ON s.user_id = u.id
                    LEFT JOIN pi_student_grades psg ON s.id = psg.student_id
                    WHERE s.lecturer_id IN ($inPlaceholder)
                    GROUP BY s.id
                    ORDER BY s.surname, s.first_name
                ";
                $stmtStudents = $pdo->prepare($query);
                $stmtStudents->execute($lecturerIds);
                $students = $stmtStudents->fetchAll(PDO::FETCH_ASSOC);
            } else {
                $students = [];
            }
            ?>
            
            <?php if ($students): ?>
                <table class="table table-bordered mt-3">
                    <thead>
                        <tr>
                            <th>Index Number</th>
                            <th>Student</th>
                            <th>Program</th>
                            <th>Average Score</th>
                            <th>Grade</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($students as $student): ?>
                            <tr id="student-row-<?php echo $student['student_id']; ?>">
                                <td><?php echo htmlspecialchars($student['index_number']); ?></td>
                                <td><?php echo htmlspecialchars($student['student_name']); ?></td>
                                <td><?php echo htmlspecialchars($student['program']); ?></td>
                                <td>
                                    <?php 
                                    echo ($student['average_score'] === null)
                                        ? '<span class="text-muted">Not Graded</span>'
                                        : htmlspecialchars($student['average_score']);
                                    ?>
                                </td>
                                <td>
                                    <?php
                                    if ($student['average_score'] === null) {
                                        echo '<span class="text-muted">Not Graded</span>';
                                    } else {
                                        echo getLetterGrade($student['average_score']);
                                    }
                                    ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php else: ?>
                <p><em>No students assigned or found in this group.</em></p>
            <?php endif; ?>
            
            <hr>
        <?php endforeach; ?>
    <?php else: ?>
        <p>No groups found.</p>
    <?php endif; ?>
</div>

<?php include('pi_footer.php'); ?>
