<?php
session_start();

if (!isset($_SESSION['lecturer_id'])) {
    header('Location: login.php');
    exit;
}

// Include database connection
require '../config/database.php';

// Retrieve persistent state from the settings table
$settingKey = 'post_internship_active';
$stmt = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = ?");
$stmt->execute([$settingKey]);
$setting = $stmt->fetch(PDO::FETCH_ASSOC);
$postInternshipActive = ($setting && $setting['setting_value'] === '1') ? true : false;

// Get teaching PowerPoint requirement setting
$pptRequirementKey = 'teaching_ppt_requirement';
$stmt = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = ?");
$stmt->execute([$pptRequirementKey]);
$pptSetting = $stmt->fetch(PDO::FETCH_ASSOC);
$pptRequirementEnabled = ($pptSetting && $pptSetting['setting_value'] === '1') ? true : false;

// Block access if post‑internship functionality is deactivated
if (!$postInternshipActive) {
    ?>
    <!DOCTYPE html>
    <html lang="en">
    <head>
       <meta charset="UTF-8">
       <title>Access Blocked</title>
       <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    </head>
    <body>
       <div class="container mt-5">
         <div class="alert alert-danger text-center">
            <h4>Access Blocked</h4>
            <p>The post‑internship functionality is currently deactivated.<br>
               Please contact your administrator or activate it from the management page.</p>
            <a href="upload_index_numbers.php" class="btn btn-warning">Go to Management</a>
         </div>
       </div>
    </body>
    </html>
    <?php
    exit;
}

// Helper functions available everywhere
function decryptData($data) {
    if (empty($data)) {
        return $data;
    }
    $cipher = 'AES-256-CBC';
    $iv_length = openssl_cipher_iv_length($cipher);
    $encryption_key = 'your-secret-key'; // Replace with your actual key

    $decoded = base64_decode($data, true);
    if ($decoded === false) {
        return $data;
    }
    if (strlen($decoded) <= $iv_length) {
        return $data;
    }
    $iv = substr($decoded, 0, $iv_length);
    $ciphertext = substr($decoded, $iv_length);
    $decrypted = openssl_decrypt($ciphertext, $cipher, $encryption_key, 0, $iv);
    return ($decrypted !== false) ? $decrypted : $data;
}
function getProgramInitials($programName) {
    $parts = explode(' ', $programName);
    $initials = '';
    foreach ($parts as $p) {
        if (!empty($p)) {
            $initials .= strtoupper($p[0]);
        }
    }
    return $initials;
}
?>

<?php include('lect_header.php'); ?>

<!-- Main Content -->
<div class="container mt-4">
    <h2 class="text-center mb-4">Lecturer Dashboard - Student Grading</h2>

    <?php
    // Database connection is already included above
    // Check lecturer ID from session
    $lecturer_id = $_SESSION['lecturer_id'];

    try {
        // Find the group_id for this lecturer
        $stmt = $pdo->prepare("
            SELECT group_id
            FROM lecturer_login
            WHERE id = :lecturer_id
        ");
        $stmt->execute(['lecturer_id' => $lecturer_id]);
        $lecturer = $stmt->fetch(PDO::FETCH_ASSOC);
        
        $group_id = $lecturer['group_id'] ?? null;

        // If no group, no students
        if (!$group_id) {
            $students = [];
        } else {
            // Query: get student details, plus internship report data if any, and group type
            $sql = "
                SELECT
                    s.id AS student_id,
                    s.first_name,
                    s.surname,
                    s.course,
                    g.group_name,
                    u.group AS user_group,
                    u.index_number,
                    tr.presentation_path,
    
                    -- Overall average
                    (SELECT ROUND(AVG(final_score), 2)
                     FROM pi_student_grades
                     WHERE student_id = s.id
                    ) AS final_score,
    
                    -- Lecturer's latest score
                    (SELECT final_score
                     FROM pi_student_grades
                     WHERE student_id = s.id
                       AND lecturer_id = :lecturer_id
                     ORDER BY graded_at DESC
                     LIMIT 1
                    ) AS lecturer_score,
    
                    -- Internship report fields
                    ir.id AS report_id,
                    ir.start_date,
                    ir.end_date,
                    ir.department,
                    ir.executive_summary,
                    ir.introduction,
                    ir.institution_profile,
                    ir.goals_objectives,
                    ir.core_duties,
                    ir.organizational_challenges,
                    ir.student_challenges,
                    ir.achievements,
                    ir.lessons_learned,
                    ir.recommendations,
                    ir.conclusion,
                    ir.references,
                    ir.appendix,
                    ir.submission_date,
                    ir.status
                FROM student_details s
                JOIN pi_lecturer_groups g ON s.lecturer_group_id = g.id
                JOIN users u ON s.user_id = u.id
                LEFT JOIN internship_reports ir ON s.user_id = ir.user_id
                LEFT JOIN teaching_reports tr ON s.user_id = tr.user_id
                WHERE s.lecturer_group_id = :group_id
            ";
    
            $stmt = $pdo->prepare($sql);
            $stmt->execute([
                'group_id'    => $group_id,
                'lecturer_id' => $lecturer_id
            ]);
            $students = $stmt->fetchAll(PDO::FETCH_ASSOC);
        }
    } catch (PDOException $e) {
        die("Error fetching students: " . htmlspecialchars($e->getMessage()));
    }
    ?>

    <?php
    // Determine group type (Industrial Internship or Teaching Practice)
    $groupType = null;
    foreach ($students as $row) {
        if (!empty($row['user_group'])) {
            $groupType = $row['user_group'];
            break;
        }
    }
    ?>

    <?php if (empty($students)): ?>
        <div class="alert alert-info">
            No students found for your assigned group.
        </div>
    <?php elseif ($groupType === 'Teaching Practice'): ?>
        <div class="alert alert-warning">
            <h4>Teaching Internship Group</h4>
            <p>Below are your assigned students. For teaching internship, students submit a PowerPoint file for assessment.</p>
        </div>
        <div class="table-responsive d-none d-md-block">
            <table class="table table-striped align-middle">
                <thead>
                    <tr>
                        <th>Student</th>
                        <th>Program</th>
                        <th>Index Number</th>
                        <th>PowerPoint Submission</th>
                    </tr>
                </thead>
                <tbody>
                <?php foreach ($students as $row): ?>
                    <?php
                        $firstName = decryptData($row['first_name']);
                        $surname   = decryptData($row['surname']);
                        $fullName  = trim($firstName . ' ' . $surname);
                        $program = decryptData($row['course']);
                        $indexNumber = $row['index_number'];
                        $pptPath = $row['presentation_path'];
                    ?>
                    <tr>
                        <td>
                            <div class="d-flex align-items-center">
                                <?php echo htmlspecialchars($fullName); ?>
                                
                                <!-- Teaching Grading Status Indicator for Desktop -->
                                <?php
                                // Check teaching grading status
                                $stmtTeachingGrade = $pdo->prepare("
                                    SELECT total_score 
                                    FROM teaching_grades 
                                    WHERE student_id = ? AND lecturer_id = ? 
                                    ORDER BY graded_at DESC 
                                    LIMIT 1
                                ");
                                $stmtTeachingGrade->execute([$row['student_id'], $lecturer_id]);
                                $teachingGrade = $stmtTeachingGrade->fetch(PDO::FETCH_ASSOC);
                                $isTeachingGraded = $teachingGrade && $teachingGrade['total_score'] !== null;
                                ?>
                                
                                <?php if ($isTeachingGraded): ?>
                                    <span class="badge bg-success ms-2" title="Teaching Internship Graded">
                                        <i class="fas fa-check"></i>
                                    </span>
                                <?php endif; ?>
                            </div>
                        </td>
                        <td><?php echo htmlspecialchars($program); ?></td>
                        <td><?php echo htmlspecialchars($indexNumber); ?></td>
                        <td>
                            <?php if (!empty($pptPath) && file_exists('../' . $pptPath)): ?>
                                <a href="<?php echo '../' . $pptPath; ?>" target="_blank" class="btn btn-primary btn-sm">View PowerPoint</a>
                                <button class="btn btn-success btn-sm ms-2" data-bs-toggle="modal" data-bs-target="#teachingGradeModal<?php echo $row['student_id']; ?>">Grade</button>
                            <?php elseif (!$pptRequirementEnabled): ?>
                                <span class="text-warning">No PowerPoint</span>
                                <button class="btn btn-success btn-sm ms-2" data-bs-toggle="modal" data-bs-target="#teachingGradeModal<?php echo $row['student_id']; ?>">Grade</button>
                            <?php else: ?>
                                <span class="text-danger">Not Submitted</span>
                            <?php endif; ?>
                        </td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <!-- Mobile View: Accordion for Teaching Internship -->
        <div class="accordion d-md-none" id="teachingStudentAccordion">
            <?php foreach ($students as $index => $row): ?>
                <?php
                    $firstName = decryptData($row['first_name']);
                    $surname   = decryptData($row['surname']);
                    $fullName  = trim($firstName . ' ' . $surname);
                    $program = decryptData($row['course']);
                    $indexNumber = $row['index_number'];
                    $pptPath = $row['presentation_path'];
                    
                    // Check teaching grading status
                    $stmtTeachingGrade = $pdo->prepare("
                        SELECT total_score 
                        FROM teaching_grades 
                        WHERE student_id = ? AND lecturer_id = ? 
                        ORDER BY graded_at DESC 
                        LIMIT 1
                    ");
                    $stmtTeachingGrade->execute([$row['student_id'], $lecturer_id]);
                    $teachingGrade = $stmtTeachingGrade->fetch(PDO::FETCH_ASSOC);
                    $isTeachingGraded = $teachingGrade && $teachingGrade['total_score'] !== null;
                ?>
                <div class="accordion-item">
                    <h2 class="accordion-header" id="teachingHeading<?php echo $index; ?>">
                        <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#teachingCollapse<?php echo $index; ?>" aria-expanded="false" aria-controls="teachingCollapse<?php echo $index; ?>">
                            <?php echo htmlspecialchars($fullName); ?> - <?php echo htmlspecialchars($program); ?>
                            <?php if ($isTeachingGraded): ?>
                                <span class="badge bg-success ms-2"><i class="fas fa-check"></i></span>
                            <?php endif; ?>
                        </button>
                    </h2>
                    <div id="teachingCollapse<?php echo $index; ?>" class="accordion-collapse collapse" aria-labelledby="teachingHeading<?php echo $index; ?>" data-bs-parent="#teachingStudentAccordion">
                        <div class="accordion-body">
                            <p><strong>Index Number:</strong> <?php echo htmlspecialchars($indexNumber); ?></p>
                            <p><strong>PowerPoint Submission:</strong>
                                <?php if (!empty($pptPath) && file_exists('../' . $pptPath)): ?>
                                    <span class="text-success">Submitted</span>
                                    <br><a href="<?php echo '../' . $pptPath; ?>" target="_blank" class="btn btn-primary btn-sm mt-1">View PowerPoint</a>
                                <?php elseif (!$pptRequirementEnabled): ?>
                                    <span class="text-warning">No PowerPoint</span>
                                <?php else: ?>
                                    <span class="text-danger">Not Submitted</span>
                                <?php endif; ?>
                            </p>
                            
                            <!-- Grade Button for Mobile -->
                            <div class="mt-3">
                                <?php if (!empty($pptPath) && file_exists('../' . $pptPath)): ?>
                                    <button class="btn btn-success btn-sm" data-bs-toggle="modal" data-bs-target="#teachingGradeModal<?php echo $row['student_id']; ?>">
                                        <i class="fas fa-star"></i> Grade Student
                                    </button>
                                <?php elseif (!$pptRequirementEnabled): ?>
                                    <button class="btn btn-success btn-sm" data-bs-toggle="modal" data-bs-target="#teachingGradeModal<?php echo $row['student_id']; ?>">
                                        <i class="fas fa-star"></i> Grade Student
                                    </button>
                                <?php else: ?>
                                    <button class="btn btn-secondary btn-sm" disabled>
                                        <i class="fas fa-lock"></i> PowerPoint Required
                                    </button>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
        
        <!-- Teaching Internship Grading Modals (Moved outside table loop for mobile access) -->
        <?php foreach ($students as $row): ?>
            <?php
                $firstName = decryptData($row['first_name']);
                $surname   = decryptData($row['surname']);
                $fullName  = trim($firstName . ' ' . $surname);
            ?>
                    <div class="modal fade" id="teachingGradeModal<?php echo $row['student_id']; ?>" tabindex="-1" aria-labelledby="teachingGradeModalLabel<?php echo $row['student_id']; ?>" aria-hidden="true">
                      <div class="modal-dialog modal-dialog-scrollable modal-lg">
                        <div class="modal-content">
                          <div class="modal-header">
                            <h5 class="modal-title" id="teachingGradeModalLabel<?php echo $row['student_id']; ?>">Teaching Internship Grading - <?php echo htmlspecialchars($fullName); ?></h5>
                          </div>
                          <div class="modal-body">
                            <form class="teaching-grading-form" method="POST" action="submit_teaching_grade.php">
                              <input type="hidden" name="student_id" value="<?php echo $row['student_id']; ?>">
                              <fieldset class="mb-3 border p-2">
                                <legend class="w-auto px-2">PowerPoint Assessment (Total 50)</legend>
                                <div class="row">
                                  <div class="col-md-4 mb-2">
                                    <label class="form-label">Appearance (10)</label>
                                    <input type="number" name="appearance" class="form-control ppt-section" min="0" max="10" required>
                                  </div>
                                  <div class="col-md-4 mb-2">
                                    <label class="form-label">Organisation of Presentation (10)</label>
                                    <input type="number" name="organisation" class="form-control ppt-section" min="0" max="10" required>
                                  </div>
                                  <div class="col-md-4 mb-2">
                                    <label class="form-label">CV (10)</label>
                                    <input type="number" name="cv" class="form-control ppt-section" min="0" max="10" required>
                                  </div>
                                  <div class="col-md-4 mb-2">
                                    <label class="form-label">Technical Practical Activity (10)</label>
                                    <input type="number" name="technical_practical" class="form-control ppt-section" min="0" max="10" required>
                                  </div>
                                  <div class="col-md-4 mb-2">
                                    <label class="form-label">Reflects Practical (10)</label>
                                    <input type="number" name="reflects_practical" class="form-control ppt-section" min="0" max="10" required>
                                  </div>
                                </div>
                                <div class="text-end"><small class="text-muted">Subtotal: <span class="ppt-total">0</span>/50</small></div>
                              </fieldset>
                              <fieldset class="mb-3 border p-2">
                                <legend class="w-auto px-2">Presentation Assessment (Total 50)</legend>
                                <div class="row">
                                  <div class="col-md-4 mb-2">
                                    <label class="form-label">Demeanor (10)</label>
                                    <input type="number" name="demeanor" class="form-control pres-section" min="0" max="10" required>
                                  </div>
                                  <div class="col-md-4 mb-2">
                                    <label class="form-label">Communication (10)</label>
                                    <input type="number" name="communication" class="form-control pres-section" min="0" max="10" required>
                                  </div>
                                  <div class="col-md-4 mb-2">
                                    <label class="form-label">Maintain Eye Contact (10)</label>
                                    <input type="number" name="eye_contact" class="form-control pres-section" min="0" max="10" required>
                                  </div>
                                  <div class="col-md-4 mb-2">
                                    <label class="form-label">Question and Answer (10)</label>
                                    <input type="number" name="question_answer" class="form-control pres-section" min="0" max="10" required>
                                  </div>
                                  <div class="col-md-4 mb-2">
                                    <label class="form-label">Understanding and Usage of Philosophy (10)</label>
                                    <input type="number" name="philosophy" class="form-control pres-section" min="0" max="10" required>
                                  </div>
                                </div>
                                <div class="text-end"><small class="text-muted">Subtotal: <span class="pres-total">0</span>/50</small></div>
                              </fieldset>
                              <div class="mb-3">
                                <label class="form-label"><strong>Comments (Optional)</strong></label>
                                <textarea name="comments" class="form-control" rows="3" placeholder="Add your comments here..."></textarea>
                              </div>
                              <div class="mt-3">
                                <button type="submit" class="btn btn-primary">Submit Grade</button>
                                <span class="ms-3 teaching-grade-status"></span>
                              </div>
                      <!-- Alert container for form submission feedback -->
                      <div class="teaching-grade-alert mt-3"></div>
                            </form>
                          </div>
                          <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                          </div>
                        </div>
                      </div>
                    </div>
                <?php endforeach; ?>
    <?php else: ?>
        <!-- Desktop View: Table -->
        <div class="table-responsive d-none d-md-block">
            <table class="table table-striped align-middle">
                <thead>
                    <tr>
                        <th>Student</th>
                        <th>Program</th>
                        <th>Group</th>
                        <th>Average Score</th>
                        <th>Action</th>
                    </tr>
                </thead>
                <tbody>
                <?php foreach ($students as $row): ?>
                    <?php
                        $firstName = decryptData($row['first_name']);
                        $surname   = decryptData($row['surname']);
                        $fullName  = trim($firstName . ' ' . $surname);
    
                        $program = decryptData($row['course']);
                        $programInitials = getProgramInitials($program);
    
                        // Internship report fields
                        $reportId       = $row['report_id'];
                        $startDate      = decryptData($row['start_date']);
                        $endDate        = decryptData($row['end_date']);
                        $department     = decryptData($row['department']);
                        $executiveSum   = decryptData($row['executive_summary']);
                        $introduction   = decryptData($row['introduction']);
                        $institution    = decryptData($row['institution_profile']);
                        $goalsObjectives= decryptData($row['goals_objectives']);
                        $coreDuties     = decryptData($row['core_duties']);
                        $orgChallenges  = decryptData($row['organizational_challenges']);
                        $studentChalls  = decryptData($row['student_challenges']);
                        $achievements   = decryptData($row['achievements']);
                        $lessonsLearned = decryptData($row['lessons_learned']);
                        $recommendations= decryptData($row['recommendations']);
                        $conclusion     = decryptData($row['conclusion']);
                        $references     = decryptData($row['references']);
                        $appendix       = decryptData($row['appendix']);
                        $submissionDate = decryptData($row['submission_date']);
                        $status         = decryptData($row['status']);
                    ?>
                    <tr>
                        <!-- Full Name -->
                        <td>
                            <input type="hidden" name="student_id" value="<?php echo $row['student_id']; ?>">
                            <div class="d-flex align-items-center">
                            <?php echo htmlspecialchars($fullName); ?>
                                
                                <!-- Grading Status Indicators for Desktop -->
                                <?php
                                // Check grading status for this student
                                $stmtGradeStatus = $pdo->prepare("
                                    SELECT report_score, presentation_score, grading_type
                                    FROM pi_student_grades 
                                    WHERE student_id = ? AND lecturer_id = ? 
                                    ORDER BY graded_at DESC 
                                    LIMIT 1
                                ");
                                $stmtGradeStatus->execute([$row['student_id'], $lecturer_id]);
                                $gradeStatus = $stmtGradeStatus->fetch(PDO::FETCH_ASSOC);
                                
                                $hasReportGrade = $gradeStatus && $gradeStatus['report_score'] !== null;
                                $hasPresentationGrade = $gradeStatus && $gradeStatus['presentation_score'] !== null;
                                ?>
                                
                                <?php if ($hasReportGrade && $hasPresentationGrade): ?>
                                    <span class="badge bg-success ms-2" title="Both Report and Presentation Graded">
                                        <i class="fas fa-check-double"></i>
                                    </span>
                                <?php elseif ($hasReportGrade): ?>
                                    <span class="badge bg-info ms-2" title="Report Graded">
                                        <i class="fas fa-check"></i>
                                    </span>
                                <?php endif; ?>
                            </div>
                        </td>
                        
                        <!-- Program -->
                        <td><?php echo htmlspecialchars($program); ?></td>
    
                        <!-- Group -->
                        <td><?php echo htmlspecialchars($row['group_name']); ?></td>
    
                        <!-- Average Score -->
                        <td>
                            <?php 
                            // Get the latest grading information for this student
                            $stmtFinalGrade = $pdo->prepare("
                                SELECT 
                                    report_score,
                                    presentation_score,
                                    final_score
                                FROM pi_student_grades 
                                WHERE student_id = ? 
                                ORDER BY graded_at DESC 
                                LIMIT 1
                            ");
                            $stmtFinalGrade->execute([$row['student_id']]);
                            $finalGrade = $stmtFinalGrade->fetch(PDO::FETCH_ASSOC);
                            
                            if ($finalGrade && $finalGrade['final_score'] !== null): ?>
                                <div class="small">
                                    <strong>Final: <?php echo htmlspecialchars($finalGrade['final_score']); ?></strong><br>
                                    <?php if ($finalGrade['report_score'] !== null): ?>
                                        Report: <?php echo htmlspecialchars($finalGrade['report_score']); ?><br>
                                    <?php endif; ?>
                                    <?php if ($finalGrade['presentation_score'] !== null): ?>
                                        Presentation: <?php echo htmlspecialchars($finalGrade['presentation_score']); ?>
                                    <?php endif; ?>
                                </div>
                            <?php else: ?>
                                <span class="text-muted">Not Graded</span>
                            <?php endif; ?>
                        </td>
    
                        <!-- Action Column -->
                        <td>
                            <!-- Report Button -->
                            <?php if ($reportId): ?>
                                <?php
                                // Check if lecturer has already graded the report for this student
                                $stmtReportGrade = $pdo->prepare("
                                    SELECT report_score FROM pi_student_grades 
                                    WHERE student_id = ? AND lecturer_id = ? AND report_score IS NOT NULL
                                    ORDER BY graded_at DESC LIMIT 1
                                ");
                                $stmtReportGrade->execute([$row['student_id'], $lecturer_id]);
                                $reportGrade = $stmtReportGrade->fetch(PDO::FETCH_ASSOC);
                                ?>
                                <button class="btn btn-info btn-sm me-1"
                                        data-bs-toggle="modal"
                                        data-bs-target="#reportModal"
                                        data-student-id="<?php echo $row['student_id']; ?>"
                                        data-fullname="<?php echo htmlspecialchars($fullName); ?>"
                                        data-start_date="<?php echo htmlspecialchars($startDate); ?>"
                                        data-end_date="<?php echo htmlspecialchars($endDate); ?>"
                                        data-department="<?php echo htmlspecialchars($department); ?>"
                                        data-executive_summary="<?php echo htmlspecialchars($executiveSum); ?>"
                                        data-introduction="<?php echo htmlspecialchars($introduction); ?>"
                                        data-institution_profile="<?php echo htmlspecialchars($institution); ?>"
                                        data-goals_objectives="<?php echo htmlspecialchars($goalsObjectives); ?>"
                                        data-core_duties="<?php echo htmlspecialchars($coreDuties); ?>"
                                        data-organizational_challenges="<?php echo htmlspecialchars($orgChallenges); ?>"
                                        data-student_challenges="<?php echo htmlspecialchars($studentChalls); ?>"
                                        data-achievements="<?php echo htmlspecialchars($achievements); ?>"
                                        data-lessons_learned="<?php echo htmlspecialchars($lessonsLearned); ?>"
                                        data-recommendations="<?php echo htmlspecialchars($recommendations); ?>"
                                        data-conclusion="<?php echo htmlspecialchars($conclusion); ?>"
                                        data-references="<?php echo htmlspecialchars($references); ?>"
                                        data-appendix="<?php echo htmlspecialchars($appendix); ?>"
                                        data-submission_date="<?php echo htmlspecialchars($submissionDate); ?>"
                                        data-status="<?php echo htmlspecialchars($status); ?>">
                                    <i class="fas fa-file-alt"></i> Report & Grade
                                    <?php if ($reportGrade): ?>
                                        <span class="badge bg-success ms-1"><i class="fas fa-check"></i> Graded</span>
                                    <?php endif; ?>
                                </button>
                            <?php else: ?>
                                <button class="btn btn-secondary btn-sm me-1" disabled>
                                    <i class="fas fa-file-alt"></i> No Report
                                </button>
                            <?php endif; ?>
    
                            <!-- Presentation Button (50%) -->
                            <?php if ($reportId): ?>
                                <?php 
                                // Check if lecturer has already graded the presentation
                                $stmtPresGrade = $pdo->prepare("
                                    SELECT presentation_score FROM pi_student_grades 
                                    WHERE student_id = ? AND lecturer_id = ? AND presentation_score IS NOT NULL
                                    ORDER BY graded_at DESC LIMIT 1
                                ");
                                $stmtPresGrade->execute([$row['student_id'], $lecturer_id]);
                                $presGrade = $stmtPresGrade->fetch(PDO::FETCH_ASSOC);
                                ?>
                                <?php if ($presGrade): ?>
                                    <button class="btn btn-warning btn-sm me-1"
                                            data-bs-toggle="modal"
                                            data-bs-target="#presentationGradeModal<?php echo $row['student_id']; ?>">
                                        Edit Presentation (<?php echo $presGrade['presentation_score']; ?>)
                                    </button>
                                <?php else: ?>
                                    <button class="btn btn-success btn-sm me-1"
                                            data-bs-toggle="modal"
                                            data-bs-target="#presentationGradeModal<?php echo $row['student_id']; ?>">
                                        Grade Presentation
                                    </button>
                                <?php endif; ?>
                            <?php endif; ?>
                        </td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    
        <!-- Mobile View: Accordion -->
        <div class="accordion d-md-none" id="studentAccordion">
            <?php foreach ($students as $index => $row): ?>
                <?php
                    $firstName = decryptData($row['first_name']);
                    $surname   = decryptData($row['surname']);
                    $fullName  = trim($firstName . ' ' . $surname);
                    $program = decryptData($row['course']);
                    // Decrypt all report fields for the modal
                    $reportId       = $row['report_id'];
                    $startDate      = decryptData($row['start_date']);
                    $endDate        = decryptData($row['end_date']);
                    $department     = decryptData($row['department']);
                    $executiveSum   = decryptData($row['executive_summary']);
                    $introduction   = decryptData($row['introduction']);
                    $institution    = decryptData($row['institution_profile']);
                    $goalsObjectives= decryptData($row['goals_objectives']);
                    $coreDuties     = decryptData($row['core_duties']);
                    $orgChallenges  = decryptData($row['organizational_challenges']);
                    $studentChalls  = decryptData($row['student_challenges']);
                    $achievements   = decryptData($row['achievements']);
                    $lessonsLearned = decryptData($row['lessons_learned']);
                    $recommendations= decryptData($row['recommendations']);
                    $conclusion     = decryptData($row['conclusion']);
                    $references     = decryptData($row['references']);
                    $appendix       = decryptData($row['appendix']);
                    $submissionDate = decryptData($row['submission_date']);
                    $status         = decryptData($row['status']);
                    
                    // Check grading status for this student
                    $stmtGradeStatus = $pdo->prepare("
                        SELECT report_score, presentation_score, grading_type
                        FROM pi_student_grades 
                        WHERE student_id = ? AND lecturer_id = ? 
                        ORDER BY graded_at DESC 
                        LIMIT 1
                    ");
                    $stmtGradeStatus->execute([$row['student_id'], $lecturer_id]);
                    $gradeStatus = $stmtGradeStatus->fetch(PDO::FETCH_ASSOC);
                    
                    $hasReportGrade = $gradeStatus && $gradeStatus['report_score'] !== null;
                    $hasPresentationGrade = $gradeStatus && $gradeStatus['presentation_score'] !== null;
                    $gradingType = $gradeStatus ? $gradeStatus['grading_type'] : null;
                ?>
                <div class="accordion-item">
                    <h2 class="accordion-header" id="heading<?php echo $index; ?>">
                        <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapse<?php echo $index; ?>" aria-expanded="false" aria-controls="collapse<?php echo $index; ?>">
                            <input type="hidden" name="student_id" value="<?php echo $row['student_id']; ?>">
                            <?php echo htmlspecialchars($fullName); ?> - <?php echo htmlspecialchars($program); ?>
                            
                            <!-- Grading Status Indicators -->
                            <div class="ms-auto d-flex align-items-center">
                                <?php if ($hasReportGrade && $hasPresentationGrade): ?>
                                    <span class="badge bg-success me-1" title="Both Report and Presentation Graded">
                                        <i class="fas fa-check-double"></i>
                                    </span>
                                <?php elseif ($hasReportGrade): ?>
                                    <span class="badge bg-info me-1" title="Report Graded">
                                        <i class="fas fa-check"></i>
                                    </span>
                                <?php endif; ?>
                            </div>
                        </button>
                    </h2>
                    <div id="collapse<?php echo $index; ?>" class="accordion-collapse collapse" aria-labelledby="heading<?php echo $index; ?>" data-bs-parent="#studentAccordion">
                        <div class="accordion-body">
                            <p><strong>Group:</strong> <?php echo htmlspecialchars($row['group_name']); ?></p>
                            <p><strong>Average Score:</strong>
                            <?php 
                            // Get the latest grading information for this student
                            $stmtFinalGrade = $pdo->prepare("
                                SELECT 
                                    report_score,
                                    presentation_score,
                                    final_score
                                FROM pi_student_grades 
                                WHERE student_id = ? 
                                ORDER BY graded_at DESC 
                                LIMIT 1
                            ");
                            $stmtFinalGrade->execute([$row['student_id']]);
                            $finalGrade = $stmtFinalGrade->fetch(PDO::FETCH_ASSOC);
                            
                            if ($finalGrade && $finalGrade['final_score'] !== null): ?>
                                <div class="small">
                                    <strong>Final: <?php echo htmlspecialchars($finalGrade['final_score']); ?></strong><br>
                                    <?php if ($finalGrade['report_score'] !== null): ?>
                                        Report: <?php echo htmlspecialchars($finalGrade['report_score']); ?><br>
                                    <?php endif; ?>
                                    <?php if ($finalGrade['presentation_score'] !== null): ?>
                                        Presentation: <?php echo htmlspecialchars($finalGrade['presentation_score']); ?>
                                    <?php endif; ?>
                                </div>
                            <?php else: ?>
                                <span class="text-muted">Not Graded</span>
                            <?php endif; ?>
                            </p>
                            <!-- Action Buttons -->
                            <div class="mb-2">
                                <?php if ($reportId): ?>
                                    <?php
                                    // Check if lecturer has already graded the report for this student
                                    $stmtReportGrade = $pdo->prepare("
                                        SELECT report_score FROM pi_student_grades 
                                        WHERE student_id = ? AND lecturer_id = ? AND report_score IS NOT NULL
                                        ORDER BY graded_at DESC LIMIT 1
                                    ");
                                    $stmtReportGrade->execute([$row['student_id'], $lecturer_id]);
                                    $reportGrade = $stmtReportGrade->fetch(PDO::FETCH_ASSOC);
                                    ?>
                                    <button class="btn btn-info btn-sm me-1"
                                            data-bs-toggle="modal"
                                            data-bs-target="#reportModal"
                                            data-student-id="<?php echo $row['student_id']; ?>"
                                            data-fullname="<?php echo htmlspecialchars($fullName); ?>"
                                            data-start_date="<?php echo htmlspecialchars($startDate); ?>"
                                            data-end_date="<?php echo htmlspecialchars($endDate); ?>"
                                            data-department="<?php echo htmlspecialchars($department); ?>"
                                            data-executive_summary="<?php echo htmlspecialchars($executiveSum); ?>"
                                            data-introduction="<?php echo htmlspecialchars($introduction); ?>"
                                            data-institution_profile="<?php echo htmlspecialchars($institution); ?>"
                                            data-goals_objectives="<?php echo htmlspecialchars($goalsObjectives); ?>"
                                            data-core_duties="<?php echo htmlspecialchars($coreDuties); ?>"
                                            data-organizational_challenges="<?php echo htmlspecialchars($orgChallenges); ?>"
                                            data-student_challenges="<?php echo htmlspecialchars($studentChalls); ?>"
                                            data-achievements="<?php echo htmlspecialchars($achievements); ?>"
                                            data-lessons_learned="<?php echo htmlspecialchars($lessonsLearned); ?>"
                                            data-recommendations="<?php echo htmlspecialchars($recommendations); ?>"
                                            data-conclusion="<?php echo htmlspecialchars($conclusion); ?>"
                                            data-references="<?php echo htmlspecialchars($references); ?>"
                                            data-appendix="<?php echo htmlspecialchars($appendix); ?>"
                                            data-submission_date="<?php echo htmlspecialchars($submissionDate); ?>"
                                            data-status="<?php echo htmlspecialchars($status); ?>">
                                        <i class="fas fa-file-alt"></i> Report & Grade
                                        <?php if ($reportGrade): ?>
                                            <span class="badge bg-success ms-1"><i class="fas fa-check"></i> Graded</span>
                                        <?php endif; ?>
                                    </button>
                                <?php else: ?>
                                    <button class="btn btn-secondary btn-sm me-1" disabled>
                                        <i class="fas fa-file-alt"></i> No Report
                                    </button>
                                <?php endif; ?>
    
                                <!-- Presentation Button (50%) -->
                                <?php if ($reportId): ?>
                                    <?php 
                                    // Check if lecturer has already graded the presentation
                                    $stmtPresGrade = $pdo->prepare("
                                        SELECT presentation_score FROM pi_student_grades 
                                        WHERE student_id = ? AND lecturer_id = ? AND presentation_score IS NOT NULL
                                        ORDER BY graded_at DESC LIMIT 1
                                    ");
                                    $stmtPresGrade->execute([$row['student_id'], $lecturer_id]);
                                    $presGrade = $stmtPresGrade->fetch(PDO::FETCH_ASSOC);
                                    ?>
                                    <?php if ($presGrade): ?>
                                        <button class="btn btn-warning btn-sm me-1"
                                                data-bs-toggle="modal"
                                                data-bs-target="#presentationGradeModal<?php echo $row['student_id']; ?>">
                                            Edit Presentation (<?php echo $presGrade['presentation_score']; ?>)
                                        </button>
                                    <?php else: ?>
                                        <button class="btn btn-success btn-sm me-1"
                                                data-bs-toggle="modal"
                                                data-bs-target="#presentationGradeModal<?php echo $row['student_id']; ?>">
                                            Grade Presentation
                                        </button>
                                    <?php endif; ?>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    
    <?php endif; ?>
</div>

<!-- Internship Report Modal -->
<div class="modal fade" id="reportModal" tabindex="-1" aria-labelledby="reportModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-dialog-scrollable modal-lg">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="reportModalLabel">Internship Report Details</h5>
      </div>
      <div class="modal-body">
        <div id="reportModalContent"></div>
        
        <!-- Report Grading Section -->
        <div id="reportGradingSection" class="mt-4" style="display: none;">
          <hr>
          <h6>Report Assessment (50% of total grade)</h6>
          <div class="alert alert-info">
            <strong>Grade this report out of 50 points</strong><br>
            This score will be averaged with other lecturers in your group.
          </div>
          
          <form id="reportGradingForm" method="POST" action="submit_report_grade.php">
            <input type="hidden" id="gradingStudentId" name="student_id" value="">
            
            <div class="row">
              <div class="col-md-6">
                <label for="reportScore" class="form-label"><strong>Report Score (0-50)</strong></label>
                <input type="number" 
                       id="reportScore" 
                       name="report_score" 
                       class="form-control" 
                       min="0" 
                       max="50" 
                       required
                       placeholder="Enter score (0-50)">
              </div>
              <div class="col-md-6">
                <label for="reportComments" class="form-label"><strong>Comments (Optional)</strong></label>
                <textarea id="reportComments" 
                          name="report_comments" 
                          class="form-control" 
                          rows="3" 
                          placeholder="Add your comments here..."></textarea>
              </div>
            </div>
            
            <div class="mt-3">
              <button type="submit" class="btn btn-primary" id="submitReportGrade">
                Submit Report Grade
              </button>
              <span id="reportGradeStatus" class="ms-3"></span>
            </div>
          </form>
        </div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
          Close
        </button>
      </div>
    </div>
  </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
  // Function to update grading indicators
  function updateGradingIndicators(studentId, gradingType, hasGrade) {
    // Update desktop table indicators
    const desktopRow = document.querySelector(`tr:has(input[value="${studentId}"])`);
    if (desktopRow) {
      const nameCell = desktopRow.querySelector('td:first-child .d-flex');
      if (nameCell) {
        // Remove existing badges
        nameCell.querySelectorAll('.badge').forEach(badge => badge.remove());
        
        // Add new badge based on grading type
        if (gradingType === 'teaching' && hasGrade) {
          const badge = document.createElement('span');
          badge.className = 'badge bg-success ms-2';
          badge.title = 'Teaching Internship Graded';
          badge.innerHTML = '<i class="fas fa-check"></i>';
          nameCell.appendChild(badge);
        } else if (gradingType === 'report' && hasGrade) {
          const badge = document.createElement('span');
          badge.className = 'badge bg-info ms-2';
          badge.title = 'Report Graded';
          badge.innerHTML = '<i class="fas fa-check"></i>';
          nameCell.appendChild(badge);
        } else if (gradingType === 'both' && hasGrade) {
          const badge = document.createElement('span');
          badge.className = 'badge bg-success ms-2';
          badge.title = 'Both Report and Presentation Graded';
          badge.innerHTML = '<i class="fas fa-check-double"></i>';
          nameCell.appendChild(badge);
        }
      }
    }
    
    // Update mobile accordion indicators
    const accordionButtons = document.querySelectorAll(`[data-bs-target*="collapse"], [data-bs-target*="teachingCollapse"]`);
    accordionButtons.forEach(button => {
      const accordionItem = button.closest('.accordion-item');
      const studentIdInItem = accordionItem.querySelector('input[name="student_id"]')?.value || 
                             accordionItem.querySelector('input[type="hidden"]')?.value;
      
      if (studentIdInItem == studentId) {
        // Remove existing badges
        button.querySelectorAll('.badge').forEach(badge => badge.remove());
        
        // Add new badge based on grading type
        if (gradingType === 'teaching' && hasGrade) {
          const badge = document.createElement('span');
          badge.className = 'badge bg-success ms-2';
          badge.innerHTML = '<i class="fas fa-check"></i>';
          button.appendChild(badge);
        } else if (gradingType === 'report' && hasGrade) {
          const badge = document.createElement('span');
          badge.className = 'badge bg-info me-1';
          badge.title = 'Report Graded';
          badge.innerHTML = '<i class="fas fa-check"></i>';
          const msAutoDiv = button.querySelector('.ms-auto');
          if (msAutoDiv) {
            msAutoDiv.appendChild(badge);
          } else {
            button.appendChild(badge);
          }
        } else if (gradingType === 'both' && hasGrade) {
          const badge = document.createElement('span');
          badge.className = 'badge bg-success me-1';
          badge.title = 'Both Report and Presentation Graded';
          badge.innerHTML = '<i class="fas fa-check-double"></i>';
          const msAutoDiv = button.querySelector('.ms-auto');
          if (msAutoDiv) {
            msAutoDiv.appendChild(badge);
          } else {
            button.appendChild(badge);
          }
        }
      }
    });
  }

  // Function to check and update grading status
  function checkAndUpdateGradingStatus(studentId) {
    fetch('check_grading_status.php', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/x-www-form-urlencoded',
      },
      body: 'student_id=' + studentId
    })
    .then(response => response.json())
    .then(data => {
      if (data.success) {
        // Update indicators based on the grading status
        if (data.grading_type === 'teaching') {
          updateGradingIndicators(studentId, 'teaching', data.has_grade);
        } else {
          if (data.has_report_grade && data.has_presentation_grade) {
            updateGradingIndicators(studentId, 'both', true);
          } else if (data.has_report_grade) {
            updateGradingIndicators(studentId, 'report', true);
          } else {
            updateGradingIndicators(studentId, 'report', false);
          }
        }
      }
    })
    .catch(error => {
      console.error('Error checking grading status:', error);
    });
  }

  // Live subtotal calculation for PowerPoint and Presentation sections
  function updateTeachingTotals(modal) {
    var pptTotal = 0;
    var presTotal = 0;
    modal.querySelectorAll('.ppt-section').forEach(function(input) {
      pptTotal += parseInt(input.value) || 0;
    });
    modal.querySelector('.ppt-total').textContent = pptTotal;
    modal.querySelectorAll('.pres-section').forEach(function(input) {
      presTotal += parseInt(input.value) || 0;
    });
    modal.querySelector('.pres-total').textContent = presTotal;
  }
  document.querySelectorAll('.modal').forEach(function(modal) {
    modal.addEventListener('input', function(e) {
      updateTeachingTotals(modal);
    });
  });

  // Handle Report & Grade button clicks
  document.querySelectorAll('[data-bs-target="#reportModal"]').forEach(function(button) {
    button.addEventListener('click', function(e) {
      e.preventDefault();
      
      var studentId = this.getAttribute('data-student-id');
      var modalContent = document.getElementById('reportModalContent');
      var gradingSection = document.getElementById('reportGradingSection');
      var gradingStudentId = document.getElementById('gradingStudentId');
      
      // Show loading state
      modalContent.innerHTML = '<div class="text-center"><i class="fas fa-spinner fa-spin"></i> Loading report details...</div>';
      gradingSection.style.display = 'none';
      
      // Fetch report details
      var formData = new FormData();
      formData.append('student_id', studentId);
      
      fetch('fetch_report_details.php', {
        method: 'POST',
        body: formData
      })
      .then(response => response.json())
      .then(data => {
        if (data.success) {
          modalContent.innerHTML = data.html;
          gradingSection.style.display = 'block';
          gradingStudentId.value = data.student_id;
        } else {
          modalContent.innerHTML = '<div class="alert alert-danger">' + data.message + '</div>';
        }
      })
      .catch(error => {
        console.error('Error:', error);
        modalContent.innerHTML = '<div class="alert alert-danger">An error occurred while loading the report.</div>';
      });
    });
  });

  // AJAX submission for teaching grading form
  document.querySelectorAll('.teaching-grading-form').forEach(function(form) {
    form.addEventListener('submit', function(e) {
      e.preventDefault();
      var formData = new FormData(form);
      var modalContent = form.closest('.modal-content');
      var alertBox = modalContent.querySelector('.teaching-grade-alert');
      var statusSpan = modalContent.querySelector('.teaching-grade-status');
      var studentId = formData.get('student_id');
      
      // Clear previous messages
      if (alertBox) alertBox.innerHTML = '';
      if (statusSpan) statusSpan.textContent = '';
      
      // Show loading state
      if (statusSpan) statusSpan.textContent = 'Submitting...';
      
      fetch('submit_teaching_grade.php', {
        method: 'POST',
        body: formData
      })
      .then(response => response.json())
      .then(data => {
        if (data.success) {
          if (alertBox) {
          alertBox.innerHTML = '<div class="alert alert-success">' + data.message + '</div>';
          }
          if (statusSpan) {
            statusSpan.textContent = 'Success!';
            statusSpan.className = 'ms-3 teaching-grade-status text-success';
          }
          
          // Update grading indicators
          if (studentId) {
            checkAndUpdateGradingStatus(studentId);
          }
          
          // Optional: Close modal after successful submission
          setTimeout(function() {
            var modal = form.closest('.modal');
            if (modal) {
              var modalInstance = bootstrap.Modal.getInstance(modal);
              if (modalInstance) modalInstance.hide();
            }
          }, 2000);
        } else {
          if (alertBox) {
          alertBox.innerHTML = '<div class="alert alert-danger">' + data.message + '</div>';
          }
          if (statusSpan) {
            statusSpan.textContent = 'Error!';
            statusSpan.className = 'ms-3 teaching-grade-status text-danger';
          }
        }
      })
      .catch(error => {
        console.error('Error:', error);
        if (alertBox) {
        alertBox.innerHTML = '<div class="alert alert-danger">An error occurred. Please try again.</div>';
        }
        if (statusSpan) {
          statusSpan.textContent = 'Error!';
          statusSpan.className = 'ms-3 teaching-grade-status text-danger';
        }
      });
    });
  });

  // Handle report grading form submission
  document.getElementById('reportGradingForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    var formData = new FormData(this);
    var statusSpan = document.getElementById('reportGradeStatus');
    var studentId = formData.get('student_id');
    
    // Show loading state
    statusSpan.textContent = 'Submitting grade...';
    statusSpan.className = 'ms-3 text-info';
    
    fetch('submit_report_grade.php', {
      method: 'POST',
      body: formData
    })
    .then(response => response.json())
    .then(data => {
      if (data.status === 'success') {
        statusSpan.textContent = 'Grade submitted successfully!';
        statusSpan.className = 'ms-3 text-success';
        // Reset form
        this.reset();
        
        // Show additional info if available
        if (data.report_score !== undefined) {
          statusSpan.textContent += ' (Score: ' + data.report_score + ')';
        }
        if (data.final_score !== undefined) {
          statusSpan.textContent += ' (Final: ' + data.final_score + ')';
        }
        
        // Update grading indicators
        if (studentId) {
          checkAndUpdateGradingStatus(studentId);
        }
      } else {
        statusSpan.textContent = 'Error: ' + data.message;
        statusSpan.className = 'ms-3 text-danger';
      }
    })
    .catch(error => {
      console.error('Error:', error);
      statusSpan.textContent = 'An error occurred. Please try again.';
      statusSpan.className = 'ms-3 text-danger';
    });
  });
});
</script>

<?php include 'grade_modal.php'; ?>
<?php include 'presentation_grade_modal.php'; ?>
<?php include 'lect_footer.php'; ?>
</body>
</html>
