<?php
session_start();
if (!isset($_SESSION['lecturer_id'])) {
    header('Location: login');
    exit;
}

include 'lect_header.php';  // Must load jQuery, Bootstrap, Bootstrap Icons, etc.
include '../config/database.php';

$lecturerId = $_SESSION['lecturer_id'];

// Fetch lecturer's name and SMS status
$stmt = $pdo->prepare('SELECT name, sms_sent FROM lecturer_login WHERE id = ?');
$stmt->execute([$lecturerId]);
$lecturer = $stmt->fetch();
$lecturerName = $lecturer ? htmlspecialchars($lecturer['name'], ENT_QUOTES, 'UTF-8') : '';
$smsSent = $lecturer ? $lecturer['sms_sent'] : 1;

// Fetch assigned students
$stmt = $pdo->prepare('
    SELECT sd.id, sd.first_name, sd.surname, u.index_number, sd.telephone, 
           sd.organisation_name, sd.location, sd.district, sd.region, sd.gps_location
    FROM student_details sd 
    JOIN users u ON sd.user_id = u.id
    WHERE sd.lecturer_id = ?
');
$stmt->execute([$lecturerId]);
$students = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!-- Main Container -->
<div class="container mt-4">
    <h2>Assigned Students</h2>
    <?php if (empty($students)): ?>
        <div class="alert alert-info" role="alert">
            You have not been assigned any students yet.
        </div>
    <?php else: ?>
        <!-- Hidden print header (shown only in print) -->
        <div id="printHeader" class="d-none">
            <h3>Students assigned to <?php echo $lecturerName; ?></h3>
        </div>

        <!-- Desktop Table -->
        <div class="table-responsive d-none d-md-block">
            <table class="table table-striped" id="studentsTable">
                <thead>
                    <tr>
                        <th>Name</th>
                        <th>Index Number</th>
                        <th>Telephone Number</th>
                        <th>Name of Organisation</th>
                        <th>Location</th>
                        <th>District/Region</th>
                        <th>GPS Location</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($students as $index => $student): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($student['first_name'] . ' ' . $student['surname'], ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><?php echo htmlspecialchars($student['index_number'], ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><?php echo htmlspecialchars($student['telephone'], ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><?php echo htmlspecialchars($student['organisation_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><?php echo htmlspecialchars($student['location'], ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><?php echo htmlspecialchars($student['district'] . '/' . substr($student['region'], 0, 2), ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><?php echo htmlspecialchars($student['gps_location'], ENT_QUOTES, 'UTF-8'); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <!-- Mobile Accordion -->
        <div class="accordion d-md-none" id="studentsAccordion">
            <?php foreach ($students as $index => $student): ?>
                <div class="accordion-item">
                    <h2 class="accordion-header" id="heading<?php echo $index; ?>">
                        <button class="accordion-button collapsed"
                                type="button" data-bs-toggle="collapse"
                                data-bs-target="#collapse<?php echo $index; ?>"
                                aria-expanded="false"
                                aria-controls="collapse<?php echo $index; ?>">
                            <?php echo htmlspecialchars($student['first_name'] . ' ' . $student['surname'], ENT_QUOTES, 'UTF-8'); ?>
                        </button>
                    </h2>
                    <div id="collapse<?php echo $index; ?>" 
                         class="accordion-collapse collapse"
                         aria-labelledby="heading<?php echo $index; ?>"
                         data-bs-parent="#studentsAccordion">
                        <div class="accordion-body">
                            <p><strong>Index Number:</strong>
                                <?php echo htmlspecialchars($student['index_number'], ENT_QUOTES, 'UTF-8'); ?>
                            </p>
                            <p><strong>Telephone Number:</strong>
                                <?php echo htmlspecialchars($student['telephone'], ENT_QUOTES, 'UTF-8'); ?>
                            </p>
                            <p><strong>Name of Organisation:</strong>
                                <?php echo htmlspecialchars($student['organisation_name'], ENT_QUOTES, 'UTF-8'); ?>
                            </p>
                            <p><strong>Location:</strong>
                                <?php echo htmlspecialchars($student['location'], ENT_QUOTES, 'UTF-8'); ?>
                            </p>
                            <p><strong>District/Region:</strong>
                                <?php echo htmlspecialchars($student['district'] . '/' . substr($student['region'], 0, 2), ENT_QUOTES, 'UTF-8'); ?>
                            </p>
                            <p><strong>GPS Location:</strong>
                                <?php echo htmlspecialchars($student['gps_location'], ENT_QUOTES, 'UTF-8'); ?>
                            </p>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>

        <!-- Action Buttons -->
        <button class="btn btn-outline-custom mt-3" 
                id="sendSmsButton" 
                <?php echo $smsSent ? 'disabled' : ''; ?>>
            Send Bulk SMS
        </button>
        <button class="btn btn-outline-custom mt-3 d-none d-md-inline-block" 
                id="printButton">
            Print
        </button>
    <?php endif; ?>
</div>

<!-- Grading Modal -->
<div class="modal fade" id="gradingModal" tabindex="-1" aria-labelledby="gradingModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <form id="gradingForm">
                <div class="modal-header">
                    <h5 class="modal-title" id="gradingModalLabel">Grade Student</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <!-- Hidden field to store student ID -->
                    <input type="hidden" id="gradingStudentId" name="student_id">
                    <div class="mb-3">
                        <label for="marksInput" class="form-label">Marks (0-100)</label>
                        <input type="number" class="form-control" id="marksInput" name="marks" min="0" max="100" required oninput="calculateGrade(this.value)">
                    </div>
                    <div class="mb-3">
                        <label for="gradeDisplay" class="form-label">Grade</label>
                        <input type="text" class="form-control" id="gradeDisplay" name="grade" readonly>
                    </div>
                    <div class="mb-3">
                        <label for="gpDisplay" class="form-label">GP</label>
                        <input type="text" class="form-control" id="gpDisplay" name="gp" readonly>
                    </div>
                    <div class="mb-3">
                        <label for="descriptionDisplay" class="form-label">Description</label>
                        <input type="text" class="form-control" id="descriptionDisplay" name="description" readonly>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="submit" class="btn btn-primary">Save Grade</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- SMS Modal -->
<div class="modal fade" id="smsModal" tabindex="-1" aria-labelledby="smsModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="smsModalLabel">Compose SMS</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <form id="smsForm">
                    <div class="mb-3">
                        <label for="smsMessage" class="form-label">Message</label>
                        <textarea class="form-control" id="smsMessage" rows="3" maxlength="160">
You have been assigned to <?php echo $lecturerName; ?>. You will be visited for supervision at your organisation soon.
                        </textarea>
                        <div id="charCount" class="form-text">0/160 characters used</div>
                    </div>
                    <button type="submit" class="btn btn-outline-custom">Send SMS</button>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- PI Team Modal (for loading colleague lecturers via AJAX) -->
<div class="modal fade" id="piTeamModal" tabindex="-1" aria-labelledby="piTeamModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered modal-fullscreen-sm-down">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="piTeamModalLabel">PI Team</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body" id="piTeamContent">
        <!-- AJAX-loaded content from colleague_lecturers.php will appear here -->
        <p class="text-center">Loading PI Team details...</p>
      </div>
    </div>
  </div>
</div>

<!-- Include the Speed Dial (FAB) -->
<?php include 'speed-dial.php'; ?>

<script>
// Your existing JavaScript (grading, SMS, print, etc.) remains here.
document.addEventListener('DOMContentLoaded', function() {
    if (typeof $ !== 'undefined' && $.fn.bootstrapToggle) {
        $('.supervision-toggle').bootstrapToggle();
    }
});

// ... (other existing scripts for grading, SMS, etc.) ...
</script>

<style>
@media print {
    body * {
        visibility: hidden;
    }
    #printHeader, #studentsTable, #studentsTable * {
        visibility: visible;
    }
    .table-responsive.d-none.d-md-block {
        display: block !important;
        visibility: visible;
    }
    #printHeader {
        position: absolute;
        top: 0;
        left: 0;
        width: 100%;
        text-align: center;
        margin-bottom: 20px;
    }
    #studentsTable {
        position: absolute;
        top: 50px;
        left: 0;
        width: 100%;
    }
    @page {
        size: landscape;
    }
}

/* Make PI Team modal close button white */
#piTeamModal .btn-close {
    filter: invert(1);
}
</style>

<?php include 'lect_footer.php'; ?>
