<?php
require '../vendor/autoload.php';

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Fill;

session_start();
if (!isset($_SESSION['lecturer_id'])) {
    header('Location: login');
    exit;
}

include '../config/database.php';

$lecturerId = htmlspecialchars($_SESSION['lecturer_id'], ENT_QUOTES, 'UTF-8');

// Fetch lecturer details
$stmt = $pdo->prepare('SELECT name, car_number, staff_id, department FROM lecturer_login WHERE id = ?');
$stmt->execute([$lecturerId]);
$lecturer = $stmt->fetch(PDO::FETCH_ASSOC);

// Fetch supervision data
$stmt = $pdo->prepare('
    SELECT sm.date_of_supervision, sm.student_name, sm.index_number, sm.from_location, sm.to_location, sm.distance, sm.nights
    FROM supervision_mapping sm
    WHERE sm.lecturer_id = ?
');
$stmt->execute([$lecturerId]);
$supervisionData = $stmt->fetchAll(PDO::FETCH_ASSOC);

if (!$supervisionData) {
    echo "No supervision data available to export.";
    exit;
}

$spreadsheet = new Spreadsheet();
$sheet = $spreadsheet->getActiveSheet();

// Set Lecturer Details
$sheet->setCellValue('A1', 'Lecturer Name: ' . htmlspecialchars($lecturer['name'], ENT_QUOTES, 'UTF-8'));
$sheet->setCellValue('A2', 'Staff ID: ' . htmlspecialchars($lecturer['staff_id'], ENT_QUOTES, 'UTF-8'));
$sheet->setCellValue('A3', 'Car Number: ' . htmlspecialchars($lecturer['car_number'], ENT_QUOTES, 'UTF-8'));
$sheet->setCellValue('A4', 'Department: ' . htmlspecialchars($lecturer['department'], ENT_QUOTES, 'UTF-8'));

// Set header row for supervision data
$headerRow = 6;
$sheet->setCellValue('A' . $headerRow, 'Date of Supervision')
      ->setCellValue('B' . $headerRow, 'Student Name')
      ->setCellValue('C' . $headerRow, 'Index Number')
      ->setCellValue('D' . $headerRow, 'From')
      ->setCellValue('E' . $headerRow, 'To')
      ->setCellValue('F' . $headerRow, 'Distance (KM)')
      ->setCellValue('G' . $headerRow, 'Nights');

// Apply styles to the header row
$headerStyleArray = [
    'font' => [
        'bold' => true,
    ],
    'alignment' => [
        'horizontal' => Alignment::HORIZONTAL_CENTER,
        'vertical' => Alignment::VERTICAL_CENTER,
    ],
    'borders' => [
        'allBorders' => [
            'borderStyle' => Border::BORDER_THIN,
        ],
    ],
    'fill' => [
        'fillType' => Fill::FILL_SOLID,
        'color' => ['argb' => 'FFCCCCCC'],
    ],
];
$sheet->getStyle('A' . $headerRow . ':G' . $headerRow)->applyFromArray($headerStyleArray);

// Populate spreadsheet with supervision data
$row = $headerRow + 1;
foreach ($supervisionData as $data) {
    $sheet->setCellValue('A' . $row, htmlspecialchars($data['date_of_supervision'], ENT_QUOTES, 'UTF-8'));
    $sheet->setCellValue('B' . $row, htmlspecialchars($data['student_name'], ENT_QUOTES, 'UTF-8'));
    $sheet->setCellValue('C' . $row, htmlspecialchars($data['index_number'], ENT_QUOTES, 'UTF-8'));
    $sheet->setCellValue('D' . $row, htmlspecialchars($data['from_location'], ENT_QUOTES, 'UTF-8'));
    $sheet->setCellValue('E' . $row, htmlspecialchars($data['to_location'], ENT_QUOTES, 'UTF-8'));
    $sheet->setCellValue('F' . $row, htmlspecialchars($data['distance'], ENT_QUOTES, 'UTF-8'));
    $sheet->setCellValue('G' . $row, htmlspecialchars($data['nights'], ENT_QUOTES, 'UTF-8'));
    $row++;
}

// Adjust column widths for better visibility
foreach (range('A', 'G') as $columnID) {
    $sheet->getColumnDimension($columnID)->setAutoSize(true);
}

// Save Excel file to a temporary path
$writer = new Xlsx($spreadsheet);
$fileName = 'Supervision_Mapping.xlsx';
$filePath = sys_get_temp_dir() . '/' . $fileName;

try {
    $writer->save($filePath);
} catch (Exception $e) {
    echo "Error creating Excel file: " . htmlspecialchars($e->getMessage(), ENT_QUOTES, 'UTF-8');
    exit;
}

// Set headers for download
header('Content-Description: File Transfer');
header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
header('Content-Disposition: attachment; filename="' . htmlspecialchars($fileName, ENT_QUOTES, 'UTF-8') . '"');
header('Content-Transfer-Encoding: binary');
header('Expires: 0');
header('Cache-Control: must-revalidate');
header('Pragma: public');
header('Content-Length: ' . filesize($filePath));

// Clean the output buffer
ob_clean();
flush();

// Send the file to the browser
readfile($filePath);

// Delete the temporary file after download
unlink($filePath);
exit;
?>
