<?php
// Database setup verification script for report grading
// Run this to check if everything is properly configured

require '../config/database.php';

echo "<h2>Database Setup Verification for Report Grading</h2>";

// Check 1: Verify required fields exist
echo "<h3>1. Checking Required Database Fields</h3>";
try {
    $stmt = $pdo->prepare("DESCRIBE pi_student_grades");
    $stmt->execute();
    $columns = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $requiredFields = [
        'report_score' => false,
        'report_comments' => false,
        'presentation_score' => false,
        'presentation_comments' => false,
        'final_score' => false,
        'grading_type' => false
    ];
    
    foreach ($columns as $column) {
        if (isset($requiredFields[$column['Field']])) {
            $requiredFields[$column['Field']] = true;
        }
    }
    
    $allFieldsPresent = true;
    foreach ($requiredFields as $field => $present) {
        $status = $present ? "✓" : "✗";
        $color = $present ? "green" : "red";
        echo "<span style='color: $color;'>$status $field</span><br>";
        if (!$present) $allFieldsPresent = false;
    }
    
    if (!$allFieldsPresent) {
        echo "<div style='background: #fff3cd; border: 1px solid #ffeaa7; padding: 10px; margin: 10px 0; border-radius: 5px;'>";
        echo "<strong>Missing fields detected!</strong><br>";
        echo "Please run the following SQL commands in your database:<br><br>";
        echo "<pre>";
        echo "-- Add missing fields to pi_student_grades table\n";
        if (!$requiredFields['report_comments']) {
            echo "ALTER TABLE pi_student_grades ADD COLUMN report_comments TEXT DEFAULT NULL COMMENT 'Comments for report assessment';\n";
        }
        if (!$requiredFields['presentation_comments']) {
            echo "ALTER TABLE pi_student_grades ADD COLUMN presentation_comments TEXT DEFAULT NULL COMMENT 'Comments for presentation assessment';\n";
        }
        if (!$requiredFields['report_score']) {
            echo "ALTER TABLE pi_student_grades ADD COLUMN report_score DECIMAL(5,2) DEFAULT NULL COMMENT 'Report assessment score (50% of total)';\n";
        }
        if (!$requiredFields['presentation_score']) {
            echo "ALTER TABLE pi_student_grades ADD COLUMN presentation_score DECIMAL(5,2) DEFAULT NULL COMMENT 'Individual presentation score (50% of total)';\n";
        }
        if (!$requiredFields['final_score']) {
            echo "ALTER TABLE pi_student_grades ADD COLUMN final_score DECIMAL(5,2) DEFAULT NULL COMMENT 'Final combined score (report + presentation)';\n";
        }
        if (!$requiredFields['grading_type']) {
            echo "ALTER TABLE pi_student_grades ADD COLUMN grading_type ENUM('report', 'presentation', 'both') DEFAULT 'both' COMMENT 'Type of grading being done';\n";
        }
        echo "</pre>";
        echo "</div>";
    } else {
        echo "<div style='background: #d4edda; border: 1px solid #c3e6cb; padding: 10px; margin: 10px 0; border-radius: 5px;'>";
        echo "<strong>✓ All required fields are present!</strong>";
        echo "</div>";
    }
    
} catch (Exception $e) {
    echo "<div style='background: #f8d7da; border: 1px solid #f5c6cb; padding: 10px; margin: 10px 0; border-radius: 5px;'>";
    echo "<strong>Error checking database structure:</strong> " . $e->getMessage();
    echo "</div>";
}

// Check 2: Verify sample data can be inserted
echo "<h3>2. Testing Data Insertion</h3>";
try {
    // Get sample student and lecturer
    $stmt = $pdo->prepare("SELECT id FROM student_details LIMIT 1");
    $stmt->execute();
    $student = $stmt->fetch(PDO::FETCH_ASSOC);
    
    $stmt = $pdo->prepare("SELECT id FROM lecturer_login LIMIT 1");
    $stmt->execute();
    $lecturer = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($student && $lecturer) {
        echo "Sample student ID: " . $student['id'] . "<br>";
        echo "Sample lecturer ID: " . $lecturer['id'] . "<br><br>";
        
        // Test insertion
        $stmt = $pdo->prepare("
            INSERT INTO pi_student_grades (student_id, lecturer_id, report_score, report_comments, final_score) 
            VALUES (?, ?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE 
            report_score = VALUES(report_score),
            report_comments = VALUES(report_comments),
            final_score = VALUES(final_score)
        ");
        
        $testScore = 45;
        $testComments = "Test comment for verification";
        $testFinalScore = 85;
        
        $result = $stmt->execute([
            $student['id'], 
            $lecturer['id'], 
            $testScore, 
            $testComments, 
            $testFinalScore
        ]);
        
        if ($result) {
            echo "<div style='background: #d4edda; border: 1px solid #c3e6cb; padding: 10px; margin: 10px 0; border-radius: 5px;'>";
            echo "<strong>✓ Test insertion successful!</strong><br>";
            echo "Report score: $testScore<br>";
            echo "Comments: $testComments<br>";
            echo "Final score: $testFinalScore";
            echo "</div>";
            
            // Clean up test data
            $stmt = $pdo->prepare("DELETE FROM pi_student_grades WHERE student_id = ? AND lecturer_id = ? AND report_comments = ?");
            $stmt->execute([$student['id'], $lecturer['id'], $testComments]);
            echo "<small style='color: #6c757d;'>Test data cleaned up.</small>";
        } else {
            echo "<div style='background: #f8d7da; border: 1px solid #f5c6cb; padding: 10px; margin: 10px 0; border-radius: 5px;'>";
            echo "<strong>✗ Test insertion failed!</strong>";
            echo "</div>";
        }
    } else {
        echo "<div style='background: #fff3cd; border: 1px solid #ffeaa7; padding: 10px; margin: 10px 0; border-radius: 5px;'>";
        echo "<strong>No sample data available for testing.</strong><br>";
        echo "Please ensure you have at least one student and one lecturer in the database.";
        echo "</div>";
    }
    
} catch (Exception $e) {
    echo "<div style='background: #f8d7da; border: 1px solid #f5c6cb; padding: 10px; margin: 10px 0; border-radius: 5px;'>";
    echo "<strong>Error testing data insertion:</strong> " . $e->getMessage();
    echo "</div>";
}

// Check 3: Verify file permissions
echo "<h3>3. Checking File Permissions</h3>";
$requiredFiles = [
    'submit_report_grade.php',
    'fetch_report_details.php',
    'check_report_grade.php'
];

foreach ($requiredFiles as $file) {
    if (file_exists($file)) {
        echo "✓ $file exists<br>";
    } else {
        echo "<span style='color: red;'>✗ $file missing</span><br>";
    }
}

echo "<h3>4. Summary</h3>";
echo "<p>If all checks pass above, your report grading system should be ready to use.</p>";
echo "<p>To test the full functionality:</p>";
echo "<ol>";
echo "<li>Log in as a lecturer</li>";
echo "<li>Go to the lecturer dashboard</li>";
echo "<li>Click 'Report & Grade' button for a student</li>";
echo "<li>Enter a score and comments</li>";
echo "<li>Submit the grade</li>";
echo "</ol>";
?> 