<?php
session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);
require '../config/database.php';
include('pi_header.php'); // Include header

// Define course abbreviations for Industrial Internship students
$courseAbbreviations = [
    'Accounting' => 'ACCT',
    'Procurement and Supply Chain Management' => 'PSCM',
    'Marketing and Entrepreneurship' => 'Mkt&E',
    'Business Information Systems' => 'BIS',
    'Human Resource Management' => 'HRM',
    'Banking and Finance' => 'B&F',
    'Risk and Insurance' => 'R&I'
];

// -----------------------------------------------------------
// 1. Fetch students eligible for manual allocation
// -----------------------------------------------------------
try {
    $stmt = $pdo->query("
        SELECT s.id, s.first_name, s.surname, u.group, s.course, s.lecturer_group_id
        FROM student_details s
        JOIN users u ON s.user_id = u.id
        WHERE s.lecturer_group_id IS NULL 
        ORDER BY u.group DESC, s.first_name ASC
    ");
    $students = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    die('Error fetching students: ' . $e->getMessage());
}

// -----------------------------------------------------------
// 2. Fetch students eligible for auto-allocation 
//    (excluding Teaching Practice & already assigned)
//    Note: these students already have a lecturer_id set.
// -----------------------------------------------------------
try {
    $stmt = $pdo->query("
        SELECT s.id, s.lecturer_id 
        FROM student_details s
        JOIN users u ON s.user_id = u.id
        WHERE s.lecturer_group_id IS NULL
          AND s.lecturer_id IS NOT NULL
          AND u.group != 'Teaching Practice'
    ");
    $auto_students = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    die('Error fetching auto-allocated students: ' . $e->getMessage());
}

// -----------------------------------------------------------
// 3. Fetch only groups that have at least one lecturer assigned
// -----------------------------------------------------------
try {
    $stmt = $pdo->query("
        SELECT g.id, g.group_name, g.category
        FROM pi_lecturer_groups g
        JOIN lecturer_login l ON g.id = l.group_id
        WHERE l.group_id IS NOT NULL
        GROUP BY g.id
        ORDER BY g.category ASC, g.group_name ASC
    ");
    $groups = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    die('Error fetching groups: ' . $e->getMessage());
}

// -----------------------------------------------------------
// 4. Fetch current student allocations
// -----------------------------------------------------------
try {
    $stmt = $pdo->query("
        SELECT 
            s.id AS student_id, 
            CONCAT(s.first_name, ' ', s.surname) AS student_name, 
            s.course, 
            g.id AS group_id, 
            g.group_name,
            g.category
        FROM student_details s
        JOIN pi_lecturer_groups g ON s.lecturer_group_id = g.id
        ORDER BY g.category ASC, g.group_name ASC
    ");
    $allocations = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $allocations = [];
}

// -----------------------------------------------------------
// Helper: Prevent duplicate form submissions
// -----------------------------------------------------------
function redirectToSelf() {
    header('Location: ' . $_SERVER['PHP_SELF']);
    exit();
}

// -----------------------------------------------------------
// Handle manual student assignment
// -----------------------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['assign'])) {
    if (!empty($_POST['student_id']) && !empty($_POST['group_id'])) {
        $student_id = $_POST['student_id'];
        $group_id   = $_POST['group_id'];
        try {
            $stmt = $pdo->prepare("
                UPDATE student_details 
                SET lecturer_group_id = :group_id 
                WHERE id = :student_id
            ");
            $stmt->execute(['group_id' => $group_id, 'student_id' => $student_id]);
            redirectToSelf();
        } catch (PDOException $e) {
            echo "<div class='alert alert-danger'>Error assigning student: " . $e->getMessage() . "</div>";
        }
    } else {
        echo "<div class='alert alert-warning'>Please select both a student and a group.</div>";
    }
}

// -----------------------------------------------------------
// Handle individual student removal
// -----------------------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['remove_individual'])) {
    if (!empty($_POST['student_id'])) {
        $student_id = $_POST['student_id'];
        try {
            $stmt = $pdo->prepare("
                UPDATE student_details 
                SET lecturer_group_id = NULL 
                WHERE id = :student_id
            ");
            $stmt->execute(['student_id' => $student_id]);
            redirectToSelf();
        } catch (PDOException $e) {
            echo "<div class='alert alert-danger'>Error removing student: " . $e->getMessage() . "</div>";
        }
    }
}

// -----------------------------------------------------------
// 5. Handle random student allocation
//    For each unallocated student who has a lecturer_id, 
//    find the group that lecturer belongs to (using lecturer_login.id),
//    and assign the student to that group ONLY if categories match.
// -----------------------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['random_assign'])) {
    if (!empty($auto_students)) {
        try {
            // Prepare a statement to look up the lecturer's group via lecturer_login.id
            $stmtLookup = $pdo->prepare("
                SELECT group_id 
                FROM lecturer_login 
                WHERE id = :lecturer_id
                LIMIT 1
            ");

            // Prepare a statement to check group category
            $stmtGroupCategory = $pdo->prepare("
                SELECT category 
                FROM pi_lecturer_groups 
                WHERE id = :group_id
                LIMIT 1
            ");

            // Prepare a statement to update the student's group
            $stmtUpdate = $pdo->prepare("
                UPDATE student_details 
                SET lecturer_group_id = :group_id 
                WHERE id = :student_id
            ");

            $assigned_count = 0;
            $skipped_count = 0;

            foreach ($auto_students as $student) {
                $lecturer_id = $student['lecturer_id'];

                // 1) Find the group that lecturer belongs to
                $stmtLookup->execute(['lecturer_id' => $lecturer_id]);
                $lecturerGroup = $stmtLookup->fetchColumn();

                // 2) If there's a valid group for that lecturer, check category match
                if ($lecturerGroup) {
                    $stmtGroupCategory->execute(['group_id' => $lecturerGroup]);
                    $group_category = $stmtGroupCategory->fetchColumn();
                    
                    // Get student's internship type
                    $stmtStudentGroup = $pdo->prepare("
                        SELECT `group` FROM users WHERE id = (
                            SELECT user_id FROM student_details WHERE id = :student_id
                        )
                    ");
                    $stmtStudentGroup->execute(['student_id' => $student['id']]);
                    $student_group = $stmtStudentGroup->fetchColumn();
                    
                    // Only assign if categories match
                    if ($group_category && $group_category === $student_group) {
                    $stmtUpdate->execute([
                        'group_id'   => $lecturerGroup,
                        'student_id' => $student['id']
                    ]);
                        $assigned_count++;
                    } else {
                        $skipped_count++;
                    }
                }
            }

            if ($assigned_count > 0) {
                $_SESSION['success'] = "Random assignment completed: {$assigned_count} students assigned.";
                if ($skipped_count > 0) {
                    $_SESSION['success'] .= " {$skipped_count} students skipped (category mismatch).";
                }
            } else {
                $_SESSION['warning'] = "No students could be assigned due to category mismatches.";
            }

            redirectToSelf();
        } catch (PDOException $e) {
            echo "<div class='alert alert-danger'>Error: " . $e->getMessage() . "</div>";
        }
    } else {
        echo "<div class='alert alert-warning'>No students or available groups found!</div>";
    }
}

// -----------------------------------------------------------
// Handle removing all students from allocated groups
// -----------------------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['remove_all'])) {
    try {
        $stmt = $pdo->prepare("
            UPDATE student_details 
            SET lecturer_group_id = NULL 
            WHERE lecturer_group_id IS NOT NULL
        ");
        $stmt->execute();
        redirectToSelf();
    } catch (PDOException $e) {
        echo "<div class='alert alert-danger'>Error: " . $e->getMessage() . "</div>";
    }
}

// -----------------------------------------------------------
// 6. Handle random Teaching Practice student allocation
//    For each unallocated Teaching Practice student who has a lecturer_id, 
//    find the group that lecturer belongs to and assign ONLY if it's a Teaching Practice group.
// -----------------------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['random_assign_teaching'])) {
    // Get Teaching Practice students eligible for assignment
    $teaching_students = $pdo->query("
        SELECT s.id, s.lecturer_id 
        FROM student_details s
        JOIN users u ON s.user_id = u.id
        WHERE s.lecturer_group_id IS NULL
          AND s.lecturer_id IS NOT NULL
          AND u.group = 'Teaching Practice'
    ")->fetchAll(PDO::FETCH_ASSOC);

    if (!empty($teaching_students)) {
        try {
            // Prepare statements
            $stmtLookup = $pdo->prepare("
                SELECT group_id 
                FROM lecturer_login 
                WHERE id = :lecturer_id
                LIMIT 1
            ");

            $stmtGroupCategory = $pdo->prepare("
                SELECT category 
                FROM pi_lecturer_groups 
                WHERE id = :group_id
                LIMIT 1
            ");

            $stmtUpdate = $pdo->prepare("
                UPDATE student_details 
                SET lecturer_group_id = :group_id 
                WHERE id = :student_id
            ");

            $assigned_count = 0;
            $skipped_count = 0;

            foreach ($teaching_students as $student) {
                $lecturer_id = $student['lecturer_id'];

                // Find the group that lecturer belongs to
                $stmtLookup->execute(['lecturer_id' => $lecturer_id]);
                $lecturerGroup = $stmtLookup->fetchColumn();

                // If there's a valid group for that lecturer, check if it's Teaching Practice
                if ($lecturerGroup) {
                    $stmtGroupCategory->execute(['group_id' => $lecturerGroup]);
                    $group_category = $stmtGroupCategory->fetchColumn();
                    
                    // Only assign if the group is Teaching Practice
                    if ($group_category && $group_category === 'Teaching Practice') {
                        $stmtUpdate->execute([
                            'group_id'   => $lecturerGroup,
                            'student_id' => $student['id']
                        ]);
                        $assigned_count++;
                    } else {
                        $skipped_count++;
                    }
                }
            }

            if ($assigned_count > 0) {
                $_SESSION['success'] = "Teaching Practice assignment completed: {$assigned_count} students assigned.";
                if ($skipped_count > 0) {
                    $_SESSION['success'] .= " {$skipped_count} students skipped (lecturer not in Teaching Practice group).";
                }
            } else {
                $_SESSION['warning'] = "No Teaching Practice students could be assigned. Lecturers may not be in Teaching Practice groups.";
            }

            redirectToSelf();
        } catch (PDOException $e) {
            echo "<div class='alert alert-danger'>Error: " . $e->getMessage() . "</div>";
        }
    } else {
        echo "<div class='alert alert-warning'>No Teaching Practice students available for assignment!</div>";
    }
}

// -----------------------------------------------------------
// 7. Handle random Industrial Internship student allocation
//    For each unallocated Industrial Internship student who has a lecturer_id, 
//    find the group that lecturer belongs to and assign ONLY if it's an Industrial Internship group.
// -----------------------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['random_assign_industrial'])) {
    // Get Industrial Internship students eligible for assignment
    $industrial_students = $pdo->query("
        SELECT s.id, s.lecturer_id 
        FROM student_details s
        JOIN users u ON s.user_id = u.id
        WHERE s.lecturer_group_id IS NULL
          AND s.lecturer_id IS NOT NULL
          AND u.group = 'Industrial Internship'
    ")->fetchAll(PDO::FETCH_ASSOC);

    if (!empty($industrial_students)) {
        try {
            // Prepare statements
            $stmtLookup = $pdo->prepare("
                SELECT group_id 
                FROM lecturer_login 
                WHERE id = :lecturer_id
                LIMIT 1
            ");

            $stmtGroupCategory = $pdo->prepare("
                SELECT category 
                FROM pi_lecturer_groups 
                WHERE id = :group_id
                LIMIT 1
            ");

            $stmtUpdate = $pdo->prepare("
                UPDATE student_details 
                SET lecturer_group_id = :group_id 
                WHERE id = :student_id
            ");

            $assigned_count = 0;
            $skipped_count = 0;

            foreach ($industrial_students as $student) {
                $lecturer_id = $student['lecturer_id'];

                // Find the group that lecturer belongs to
                $stmtLookup->execute(['lecturer_id' => $lecturer_id]);
                $lecturerGroup = $stmtLookup->fetchColumn();

                // If there's a valid group for that lecturer, check if it's Industrial Internship
                if ($lecturerGroup) {
                    $stmtGroupCategory->execute(['group_id' => $lecturerGroup]);
                    $group_category = $stmtGroupCategory->fetchColumn();
                    
                    // Only assign if the group is Industrial Internship
                    if ($group_category && $group_category === 'Industrial Internship') {
                        $stmtUpdate->execute([
                            'group_id'   => $lecturerGroup,
                            'student_id' => $student['id']
                        ]);
                        $assigned_count++;
                    } else {
                        $skipped_count++;
                    }
                }
            }

            if ($assigned_count > 0) {
                $_SESSION['success'] = "Industrial Internship assignment completed: {$assigned_count} students assigned.";
                if ($skipped_count > 0) {
                    $_SESSION['success'] .= " {$skipped_count} students skipped (lecturer not in Industrial Internship group).";
                }
            } else {
                $_SESSION['warning'] = "No Industrial Internship students could be assigned. Lecturers may not be in Industrial Internship groups.";
            }

            redirectToSelf();
        } catch (PDOException $e) {
            echo "<div class='alert alert-danger'>Error: " . $e->getMessage() . "</div>";
        }
    } else {
        echo "<div class='alert alert-warning'>No Industrial Internship students available for assignment!</div>";
    }
}
?>

<div class="container mt-5">
    <!-- Manually Assign Student to a Group Card -->
    <div class="row">
        <div class="col-md-8 mx-auto">
            <div class="card shadow-sm p-3">
                <h4 class="text-center mb-3">Manually Assign Student to a Group</h4>
                <form method="post">
                    <div class="mb-3">
                        <label class="form-label">Select Student</label>
                        <select name="student_id" class="form-control form-control-sm" required>
                            <option value="">-- Select Student --</option>
                            <?php foreach ($students as $student): ?>
                                <?php
                                    $studentName = htmlspecialchars($student['first_name'] . ' ' . $student['surname']);
                                    $courseAbbr  = $courseAbbreviations[$student['course']] ?? $student['course'];
                                ?>
                                <option value="<?php echo $student['id']; ?>">
                                    <?php echo $studentName; ?> (<?php echo $courseAbbr; ?>)
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Select Group</label>
                        <select name="group_id" class="form-control form-control-sm" required>
                            <option value="">-- Select Group --</option>
                            <?php foreach ($groups as $group): ?>
                                <option value="<?php echo $group['id']; ?>">
                                    <?php echo htmlspecialchars($group['group_name']); ?> (<?php echo htmlspecialchars($group['category']); ?>)
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <button type="submit" name="assign" class="btn btn-primary btn-sm w-100">
                        Manually Assign Student
                    </button>
                </form>
            </div>
        </div>
    </div>

    <!-- Buttons Section -->
    <div class="text-center mt-4">
        <div class="row">
            <div class="col-md-3">
                <form method="post" class="d-inline">
                    <button type="submit" name="random_assign" class="btn btn-success w-100">
                        <i class="fas fa-random"></i> Assign All Students
                    </button>
                </form>
            </div>
            <div class="col-md-3">
                <form method="post" class="d-inline">
                    <button type="submit" name="random_assign_teaching" class="btn btn-primary w-100">
                        <i class="fas fa-chalkboard-teacher"></i> Assign Teaching Practice
                    </button>
                </form>
            </div>
            <div class="col-md-3">
        <form method="post" class="d-inline">
                    <button type="submit" name="random_assign_industrial" class="btn btn-secondary w-100">
                        <i class="fas fa-industry"></i> Assign Industrial Internship
            </button>
        </form>
            </div>
            <div class="col-md-3">
        <form method="post" class="d-inline">
                    <button type="submit" name="remove_all" class="btn btn-danger w-100"
                    onclick="return confirm('Are you sure you want to remove all students?');">
                        <i class="fas fa-trash"></i> Remove All Students
            </button>
        </form>
            </div>
        </div>
    </div>

    <!-- Group Filter -->
    <div class="mt-4">
        <label for="groupFilter">Filter by Group:</label>
        <select id="groupFilter" class="form-control form-control-sm w-25" onchange="filterGroup()">
            <option value="">-- Show All Groups --</option>
            <?php foreach ($groups as $group): ?>
                <option value="<?php echo $group['id']; ?>">
                    <?php echo htmlspecialchars($group['group_name']); ?>
                </option>
            <?php endforeach; ?>
        </select>
    </div>

    <!-- Selected Group Name -->
    <h4 class="text-center mt-4" id="selectedGroupName">All Groups</h4>

    <!-- Current Allocations Table -->
    <div class="row mt-4">
        <div class="col-md-12">
            <div class="card shadow-sm">
                <div class="card-header">
                    <h5 class="mb-0">Current Student Allocations</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-striped table-sm">
        <thead>
            <tr>
                                    <th>Student Name</th>
                                    <th>Course</th>
                                    <th>Group</th>
                                    <th>Category</th>
                <th>Action</th>
            </tr>
        </thead>
        <tbody>
                                <?php if ($allocations): ?>
            <?php foreach ($allocations as $allocation): ?>
                                        <tr>
                                            <td><?= htmlspecialchars($allocation['student_name']) ?></td>
                                            <td><?= htmlspecialchars($allocation['course']) ?></td>
                                            <td><?= htmlspecialchars($allocation['group_name']) ?></td>
                                            <td>
                                                <span class="badge <?= $allocation['category'] === 'Teaching Practice' ? 'bg-primary' : 'bg-success' ?>">
                                                    <?= htmlspecialchars($allocation['category']) ?>
                                                </span>
                    </td>
                    <td>
                        <form method="post" style="display:inline;">
                                                    <input type="hidden" name="student_id" value="<?= $allocation['student_id'] ?>">
                            <button type="submit" name="remove_individual" class="btn btn-danger btn-sm"
                                                            onclick="return confirm('Remove this student from group?');">
                                Remove
                            </button>
                        </form>
                    </td>
                </tr>
            <?php endforeach; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="5" class="text-center">No students currently allocated.</td>
                                    </tr>
                                <?php endif; ?>
        </tbody>
    </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
    function filterGroup() {
        let selectedGroup = document.getElementById('groupFilter').value;
        let rows = document.querySelectorAll('#studentTable tbody tr');
        let groupName = document.getElementById('selectedGroupName');

        if (selectedGroup === "") {
            groupName.textContent = "All Groups";
        } else {
            let selectedOption = document.querySelector("#groupFilter option[value='" + selectedGroup + "']").textContent;
            groupName.textContent = selectedOption;
        }

        rows.forEach(row => {
            if (selectedGroup === '' || String(row.dataset.group) === selectedGroup) {
                row.style.display = '';
            } else {
                row.style.display = 'none';
            }
        });
    }
</script>

<?php include('pi_footer.php'); ?>
