<?php
session_start();
if (!isset($_SESSION['admin_id'])) {
    header('Location: login');
    exit;
}

include '../config/database.php';

// --------------------------------------------------------
// 1. HANDLE DELETE ACTION
// --------------------------------------------------------
$message = '';
if (isset($_GET['delete_id'])) {
    // Sanitize the incoming ID
    $deleteId = (int)$_GET['delete_id'];
    if ($deleteId > 0) {
        // Perform the deletion
        $stmtDel = $pdo->prepare('DELETE FROM lecturer_preferences WHERE id = ? LIMIT 1');
        $stmtDel->execute([$deleteId]);

        // Provide feedback (WITHOUT showing the numeric ID)
        $message = "Successfully deleted the region assignment.";
    } else {
        $message = "Invalid assignment ID provided.";
    }
}

// --------------------------------------------------------
// 2. HANDLE FORM SUBMISSION (INSERT NEW ROW)
// --------------------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Sanitize inputs
    $lecturerId = isset($_POST['lecturer_id']) ? (int)$_POST['lecturer_id'] : 0;
    $newRegion  = isset($_POST['new_region'])
        ? htmlspecialchars($_POST['new_region'], ENT_QUOTES, 'UTF-8')
        : '';

    if ($lecturerId > 0 && !empty($newRegion)) {
        // OPTIONAL: Check if the exact (lecturer_id, preferred_region) already exists
        $checkStmt = $pdo->prepare('
            SELECT COUNT(*) 
              FROM lecturer_preferences
             WHERE lecturer_id = :lecturerId
               AND preferred_region = :region
        ');
        $checkStmt->execute([
            ':lecturerId' => $lecturerId,
            ':region'     => $newRegion
        ]);
        $existingCount = $checkStmt->fetchColumn();

        if ($existingCount > 0) {
            // Already assigned
            $message = "This region is already assigned to the selected lecturer.";
        } else {
            // Insert a new row
            $insertStmt = $pdo->prepare('
                INSERT INTO lecturer_preferences (lecturer_id, preferred_region)
                VALUES (:lecturerId, :region)
            ');
            $insertStmt->execute([
                ':lecturerId' => $lecturerId,
                ':region'     => $newRegion
            ]);

            // Provide feedback (WITHOUT showing the numeric ID)
            $message = "Region '{$newRegion}' assigned successfully.";
        }
    } else {
        $message = "Invalid lecturer or region. Please select valid options.";
    }
}

// --------------------------------------------------------
// 3. FETCH ALL LECTURERS (FOR THE DROPDOWN)
// --------------------------------------------------------
$stmt = $pdo->query('SELECT id, name FROM lecturer_login ORDER BY name');
$allLecturers = $stmt->fetchAll(PDO::FETCH_ASSOC);

// --------------------------------------------------------
// 4. FETCH DISTINCT REGIONS FROM districts TABLE
// --------------------------------------------------------
$stmt = $pdo->query('SELECT DISTINCT region FROM districts ORDER BY region');
$possibleRegions = $stmt->fetchAll(PDO::FETCH_COLUMN);

// --------------------------------------------------------
// 5. FETCH ALL ASSIGNMENTS (INCL. THE pref_id FOR DELETION)
// --------------------------------------------------------
$stmt = $pdo->query('
    SELECT lp.id AS pref_id,
           ll.name AS lecturer_name,
           lp.preferred_region
      FROM lecturer_preferences lp
      JOIN lecturer_login ll ON ll.id = lp.lecturer_id
     ORDER BY lp.id DESC
');
$currentAssignments = $stmt->fetchAll(PDO::FETCH_ASSOC);

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Assign Region (One Row per Assignment)</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" 
          href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
</head>
<body>

<?php include 'admin_header.php'; ?>

<div class="container mt-4">
    <h2>Assign a Region (One Row per Assignment)</h2>

    <!-- Display any message -->
    <?php if (!empty($message)): ?>
        <div class="alert alert-info" id="messageAlert">
            <?php echo htmlspecialchars($message, ENT_QUOTES, 'UTF-8'); ?>
        </div>
    <?php endif; ?>

    <!-- FORM to Insert a New Row (lecturer_preferences) -->
    <div class="card mb-4">
        <div class="card-body">
            <h4 class="card-title">Add a New Region Assignment</h4>
            <form method="POST" action="assign_region.php">
                <div class="mb-3">
                    <label for="lecturer_id" class="form-label">Select Lecturer:</label>
                    <select name="lecturer_id" id="lecturer_id" class="form-select" required>
                        <option value="">-- Choose a Lecturer --</option>
                        <?php foreach ($allLecturers as $lecturer): ?>
                            <option value="<?php echo (int)$lecturer['id']; ?>">
                                <?php echo htmlspecialchars($lecturer['name'], ENT_QUOTES, 'UTF-8'); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="mb-3">
                    <label for="new_region" class="form-label">Select Region:</label>
                    <select name="new_region" id="new_region" class="form-select" required>
                        <option value="">-- Choose a Region --</option>
                        <?php foreach ($possibleRegions as $reg): ?>
                            <option value="<?php echo htmlspecialchars($reg, ENT_QUOTES, 'UTF-8'); ?>">
                                <?php echo htmlspecialchars($reg, ENT_QUOTES, 'UTF-8'); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <button type="submit" class="btn btn-primary">
                    Assign Region
                </button>
            </form>
        </div>
    </div>

    <!-- TABLE of current region assignments -->
    <div class="card">
        <div class="card-body">
            <h4 class="card-title">Current Region Assignments</h4>
            <?php if (!empty($currentAssignments)): ?>
                <table class="table table-striped">
                    <thead>
                        <tr>
                            <th>Lecturer Name</th>
                            <th>Preferred Region</th>
                            <th>Delete</th> <!-- New column for deletion -->
                        </tr>
                    </thead>
                    <tbody>
                    <?php foreach ($currentAssignments as $assignment): ?>
                        <tr>
                            <td>
                                <?php echo htmlspecialchars($assignment['lecturer_name'], ENT_QUOTES, 'UTF-8'); ?>
                            </td>
                            <td>
                                <?php echo htmlspecialchars($assignment['preferred_region'], ENT_QUOTES, 'UTF-8'); ?>
                            </td>
                            <td>
                                <!-- "Delete" button/link with a confirmation prompt -->
                                <a href="assign_region.php?delete_id=<?php echo (int)$assignment['pref_id']; ?>"
                                   class="btn btn-danger btn-sm"
                                   onclick="return confirm('Are you sure you want to delete this assignment?');">
                                    <i class="fas fa-trash-alt"></i> Delete
                                </a>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            <?php else: ?>
                <p>No region assignments found yet.</p>
            <?php endif; ?>
        </div>
    </div>

</div>

<script src="https://cdn.jsdelivr.net/npm/@popperjs/core@2.10.2/dist/umd/popper.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.min.js"></script>

<!-- Hide the success message after 3 seconds -->
<script>
  window.addEventListener('DOMContentLoaded', () => {
    const alertBox = document.getElementById('messageAlert');
    if (alertBox) {
      setTimeout(() => {
        alertBox.style.transition = 'opacity 0.5s ease';
        alertBox.style.opacity = '0';
        setTimeout(() => {
          if (alertBox.parentNode) {
            alertBox.parentNode.removeChild(alertBox);
          }
        }, 500);
      }, 3000); // 3 seconds
    }
  });
</script>
</body>
</html>
