<?php
/**
 * view_groups.php  |  Manage PI-Lecturer Groups
 * -------------------------------------------------
 * Session gate—identical pattern to your other secured pages.
 * If the user is not logged in, redirect to login.php before any output.
 */
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

if (!isset($_SESSION['admin_id'])) {       // same key used elsewhere
    header('Location: login');         // same login file in this folder
    exit();
}

/* ------------------------------------------------------------------ */
/*  Error reporting (remove or lower in production)                    */
/* ------------------------------------------------------------------ */
error_reporting(E_ALL);
ini_set('display_errors', 1);

/* ------------------------------------------------------------------ */
/*  Database connection                                                */
/* ------------------------------------------------------------------ */
require '../config/database.php';          // PDO connection

/* ------------------------------------------------------------------ */
/*  Initialise feedback variables                                      */
/* ------------------------------------------------------------------ */
$msg      = "";
$msg_type = "";

/* ------------------------------------------------------------------ */
/*  CREATE group                                                       */
/* ------------------------------------------------------------------ */
if ($_SERVER["REQUEST_METHOD"] === "POST" && isset($_POST['create_group'])) {
    $group_name = trim($_POST['group_name']);
    $category = $_POST['category'] ?? 'Industrial Internship';

    if ($group_name !== '') {
        try {
            $stmt = $pdo->prepare(
                "INSERT INTO pi_lecturer_groups (group_name, category) VALUES (:group_name, :category)"
            );
            $stmt->bindParam(':group_name', $group_name);
            $stmt->bindParam(':category', $category);
            $stmt->execute();

            $msg      = "Group created successfully!";
            $msg_type = "success";
        } catch (PDOException $e) {
            $msg      = "Database error: " . $e->getMessage();
            $msg_type = "danger";
        }
    } else {
        $msg      = "Group name cannot be empty.";
        $msg_type = "warning";
    }
}

/* ------------------------------------------------------------------ */
/*  UPDATE group                                                       */
/* ------------------------------------------------------------------ */
if ($_SERVER["REQUEST_METHOD"] === "POST" && isset($_POST['update_group'])) {
    $group_id = (int) $_POST['group_id'];
    $group_name = trim($_POST['edit_group_name']);
    $category = $_POST['edit_category'] ?? 'Industrial Internship';

    if ($group_name !== '') {
        try {
            $stmt = $pdo->prepare(
                "UPDATE pi_lecturer_groups SET group_name = :group_name, category = :category WHERE id = :id"
            );
            $stmt->bindParam(':group_name', $group_name);
            $stmt->bindParam(':category', $category);
            $stmt->bindParam(':id', $group_id, PDO::PARAM_INT);
            $stmt->execute();

            $msg      = "Group updated successfully!";
            $msg_type = "success";
        } catch (PDOException $e) {
            $msg      = "Database error: " . $e->getMessage();
            $msg_type = "danger";
        }
    } else {
        $msg      = "Group name cannot be empty.";
        $msg_type = "warning";
    }
}

/* ------------------------------------------------------------------ */
/*  DELETE group                                                       */
/* ------------------------------------------------------------------ */
if (isset($_GET['delete_id'])) {
    $delete_id = (int) $_GET['delete_id'];

    try {
        $stmt = $pdo->prepare(
            "DELETE FROM pi_lecturer_groups WHERE id = :id"
        );
        $stmt->bindParam(':id', $delete_id, PDO::PARAM_INT);
        $stmt->execute();

        $msg      = "Group deleted successfully!";
        $msg_type = "success";
    } catch (PDOException $e) {
        $msg      = "Error deleting group: " . $e->getMessage();
        $msg_type = "danger";
    }

    /* redirect to prevent accidental resubmission */
    header("Location: view_groups.php?msg={$msg}&msg_type={$msg_type}");
    exit();
}

/* ------------------------------------------------------------------ */
/*  Handle redirected feedback                                         */
/* ------------------------------------------------------------------ */
if (isset($_GET['msg'], $_GET['msg_type'])) {
    $msg      = $_GET['msg'];
    $msg_type = $_GET['msg_type'];
}

/* ------------------------------------------------------------------ */
/*  Fetch groups                                                       */
/* ------------------------------------------------------------------ */
try {
    $groups = $pdo->query(
        "SELECT * FROM pi_lecturer_groups ORDER BY id DESC"
    )->fetchAll();
} catch (PDOException $e) {
    $msg      = "Error fetching groups: " . $e->getMessage();
    $msg_type = "danger";
}

include 'pi_header.php';
?>

<!-- ----------------------------------------------------------------- -->
<!--  Page Content                                                     -->
<!-- ----------------------------------------------------------------- -->
<div class="container mt-5">
    <?php if ($msg !== ''): ?>
        <div id="alert-message"
             class="alert alert-<?= htmlspecialchars($msg_type) ?> alert-dismissible fade show"
             role="alert">
            <?= htmlspecialchars($msg) ?>
        </div>
    <?php endif; ?>

    <div class="row justify-content-center">
        <div class="col-md-6">
            <div class="card shadow-lg p-4">
                <h2 class="mb-4">Manage Lecturer Groups</h2>
                <form method="post" class="mb-4">
                    <div class="mb-3">
                        <label for="group_name" class="form-label">Group Name</label>
                        <input type="text"
                               name="group_name"
                               id="group_name"
                               class="form-control"
                               placeholder="Enter Group Name"
                               required>
                    </div>
                    <div class="mb-3">
                        <label for="category" class="form-label">Category</label>
                        <select name="category" id="category" class="form-select">
                            <option value="Industrial Internship">Industrial Internship</option>
                            <option value="Teaching Practice">Teaching Practice</option>
                        </select>
                    </div>
                    <button type="submit" name="create_group" class="btn btn-primary">
                        Create Group
                    </button>
                </form>
            </div>
        </div>
    </div>

    <!-- Groups Table -->
    <div class="row mt-4">
        <div class="col-md-12">
            <div class="card shadow-lg">
                <div class="card-header">
                    <h4 class="mb-0">Existing Groups</h4>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                <thead>
                    <tr>
                                    <th>ID</th>
                        <th>Group Name</th>
                                    <th>Category</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                                <?php if (!empty($groups)): ?>
                    <?php foreach ($groups as $group): ?>
                        <tr>
                                            <td><?= htmlspecialchars($group['id']) ?></td>
                            <td><?= htmlspecialchars($group['group_name']) ?></td>
                                            <td>
                                                <span class="badge <?= $group['category'] === 'Teaching Practice' ? 'bg-primary' : 'bg-success' ?>">
                                                    <?= htmlspecialchars($group['category']) ?>
                                                </span>
                                            </td>
                            <td>
                                <button type="button" 
                                        class="btn btn-sm btn-outline-primary"
                                        onclick="openEditModal(<?= $group['id'] ?>, '<?= htmlspecialchars($group['group_name']) ?>', '<?= htmlspecialchars($group['category']) ?>')">
                                    <i class="fas fa-edit"></i> Edit
                                </button>
                                <a href="?delete_id=<?= $group['id'] ?>"
                                                   class="btn btn-sm btn-outline-danger"
                                                   onclick="return confirm('Are you sure you want to delete this group?')">
                                    <i class="fas fa-trash"></i> Delete
                                </a>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                                <?php else: ?>
                        <tr>
                                        <td colspan="4" class="text-center">No groups found.</td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Edit Group Modal -->
<div class="modal fade" id="editGroupModal" tabindex="-1" aria-labelledby="editGroupModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="editGroupModalLabel">
                    <i class="fas fa-edit me-2"></i>Edit Group
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="post">
                <div class="modal-body">
                    <input type="hidden" name="group_id" id="edit_group_id">
                    <div class="mb-3">
                        <label for="edit_group_name" class="form-label">Group Name</label>
                        <input type="text" 
                               name="edit_group_name" 
                               id="edit_group_name" 
                               class="form-control" 
                               required>
                    </div>
                    <div class="mb-3">
                        <label for="edit_category" class="form-label">Category</label>
                        <select name="edit_category" id="edit_category" class="form-select">
                            <option value="Industrial Internship">Industrial Internship</option>
                            <option value="Teaching Practice">Teaching Practice</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" name="update_group" class="btn btn-primary">
                        <i class="fas fa-save me-2"></i>Update Group
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Auto-dismiss alert -->
<script>
    setTimeout(() => {
        const alertEl = document.getElementById('alert-message');
        if (alertEl) alertEl.remove();
    }, 3000);

    // Function to open edit modal
    function openEditModal(groupId, groupName, category) {
        document.getElementById('edit_group_id').value = groupId;
        document.getElementById('edit_group_name').value = groupName;
        document.getElementById('edit_category').value = category;
        
        // Show the modal
        const modal = new bootstrap.Modal(document.getElementById('editGroupModal'));
        modal.show();
    }
</script>

<?php include 'pi_footer.php'; ?>
