<?php
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

include '../config/database.php'; // Database connection

$successMessage = '';
$errorMessage = '';

// Retrieve the persistent setting from the database
$settingKey = 'post_internship_active';
$stmtSetting = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = ?");
$stmtSetting->execute([$settingKey]);
$setting = $stmtSetting->fetch(PDO::FETCH_ASSOC);
$postInternshipActive = ($setting && $setting['setting_value'] === '1') ? true : false;

// Retrieve the current academic year setting
$currentYearKey = 'current_academic_year';
$stmtCurrentYear = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = ?");
$stmtCurrentYear->execute([$currentYearKey]);
$currentYearSetting = $stmtCurrentYear->fetch(PDO::FETCH_ASSOC);
$currentAcademicYear = $currentYearSetting ? $currentYearSetting['setting_value'] : '2024-2025';

// Retrieve the registration deadline setting for current academic year
$deadlineKey = 'registration_end_date_' . $currentAcademicYear;
$stmtDeadline = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = ?");
$stmtDeadline->execute([$deadlineKey]);
$deadlineSetting = $stmtDeadline->fetch(PDO::FETCH_ASSOC);
$registrationDeadline = $deadlineSetting ? $deadlineSetting['setting_value'] : '';

// Retrieve the registration start date setting for current academic year
$startDateKey = 'registration_start_date_' . $currentAcademicYear;
$stmtStartDate = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = ?");
$stmtStartDate->execute([$startDateKey]);
$startDateSetting = $stmtStartDate->fetch(PDO::FETCH_ASSOC);
$registrationStartDate = $startDateSetting ? $startDateSetting['setting_value'] : '';

// Retrieve the manual registration status setting
$registrationStatusKey = 'registration_status_' . $currentAcademicYear;
$stmtStatus = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = ?");
$stmtStatus->execute([$registrationStatusKey]);
$statusSetting = $stmtStatus->fetch(PDO::FETCH_ASSOC);
$manualRegistrationStatus = $statusSetting ? $statusSetting['setting_value'] : 'auto'; // 'auto', 'open', 'closed'

// Handle success and error messages from session
if (isset($_SESSION['successMessage'])) {
    $successMessage = $_SESSION['successMessage'];
    unset($_SESSION['successMessage']);
}
if (isset($_SESSION['errorMessage'])) {
    $errorMessage = $_SESSION['errorMessage'];
    unset($_SESSION['errorMessage']);
}

// Handle CSV upload
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['upload_csv'])) {
    if (isset($_FILES['csv_file']) && $_FILES['csv_file']['error'] === UPLOAD_ERR_OK) {
        $fileTmpPath = $_FILES['csv_file']['tmp_name'];
        $fileName = $_FILES['csv_file']['name'];
        $fileType = $_FILES['csv_file']['type'];
        $academicYear = htmlspecialchars($_POST['academic_year'], ENT_QUOTES, 'UTF-8');

        $allowedTypes = ['text/csv', 'application/vnd.ms-excel'];
        if (in_array($fileType, $allowedTypes)) {
            if (($handle = fopen($fileTmpPath, 'r')) !== false) {
                // Skip header row
                fgetcsv($handle);
                $pdo->beginTransaction();
                try {
                    while (($data = fgetcsv($handle, 1000, ',')) !== false) {
                        $indexNumber = trim($data[0]);
                        if (!empty($indexNumber)) {
                            $stmt = $pdo->prepare('INSERT INTO valid_index_numbers (index_number, academic_year) VALUES (:index_number, :academic_year)');
                            $stmt->bindParam(':index_number', $indexNumber);
                            $stmt->bindParam(':academic_year', $academicYear);
                            $stmt->execute();
                        }
                    }
                    $pdo->commit();
                    fclose($handle);
                    $_SESSION['successMessage'] = 'CSV file uploaded successfully for academic year ' . $academicYear . '.';
                } catch (Exception $e) {
                    $pdo->rollBack();
                    $_SESSION['errorMessage'] = 'Error processing CSV file: ' . $e->getMessage();
                }
            } else {
                $_SESSION['errorMessage'] = 'Unable to open the CSV file.';
            }
        } else {
            $_SESSION['errorMessage'] = 'Please upload a valid CSV file.';
        }
    } else {
        $_SESSION['errorMessage'] = 'Please select a CSV file to upload.';
    }
    header('Location: upload_index_numbers.php');
    exit;
}

// Handle manual index number submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_index_number'])) {
    $indexNumber = htmlspecialchars($_POST['index_number'], ENT_QUOTES, 'UTF-8');
    $academicYear = htmlspecialchars($_POST['academic_year'], ENT_QUOTES, 'UTF-8');
    if (!empty($indexNumber) && !empty($academicYear)) {
        try {
            $stmt = $pdo->prepare('INSERT INTO valid_index_numbers (index_number, academic_year) VALUES (:index_number, :academic_year)');
            $stmt->bindParam(':index_number', $indexNumber);
            $stmt->bindParam(':academic_year', $academicYear);
            $stmt->execute();
            $_SESSION['successMessage'] = 'Index number added successfully for academic year ' . $academicYear . '.';
        } catch (Exception $e) {
            $_SESSION['errorMessage'] = 'Error: ' . $e->getMessage();
        }
    } else {
        $_SESSION['errorMessage'] = 'Index number and academic year cannot be empty.';
    }
    header('Location: upload_index_numbers.php');
    exit;
}

// Handle registration deadline setting
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['set_deadline'])) {
    $startDate = htmlspecialchars($_POST['registration_start_date'], ENT_QUOTES, 'UTF-8');
    $endDate = htmlspecialchars($_POST['registration_end_date'], ENT_QUOTES, 'UTF-8');
    $academicYear = htmlspecialchars($_POST['academic_year'], ENT_QUOTES, 'UTF-8');
    
    if (!empty($startDate) && !empty($endDate) && !empty($academicYear)) {
        // Validate that end date is after start date
        if (strtotime($endDate) <= strtotime($startDate)) {
            $_SESSION['errorMessage'] = 'End date must be after start date.';
        } else {
            try {
                // Save start date for the academic year
                $startDateKey = 'registration_start_date_' . $academicYear;
                $stmt = $pdo->prepare("REPLACE INTO settings (setting_key, setting_value, description) VALUES (?, ?, ?)");
                $stmt->execute([$startDateKey, $startDate, 'Student registration start date for ' . $academicYear . ' (YYYY-MM-DD HH:MM:SS)']);
                
                // Save end date for the academic year
                $endDateKey = 'registration_end_date_' . $academicYear;
                $stmt = $pdo->prepare("REPLACE INTO settings (setting_key, setting_value, description) VALUES (?, ?, ?)");
                $stmt->execute([$endDateKey, $endDate, 'Student registration end date for ' . $academicYear . ' (YYYY-MM-DD HH:MM:SS)']);
                
                $_SESSION['successMessage'] = 'Registration period set successfully for academic year ' . $academicYear . '.';
            } catch (Exception $e) {
                $_SESSION['errorMessage'] = 'Error setting registration period: ' . $e->getMessage();
            }
        }
    } else {
        $_SESSION['errorMessage'] = 'Please enter both start and end dates and select academic year.';
    }
    header('Location: upload_index_numbers.php');
    exit;
}

// Handle academic year change
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['change_academic_year'])) {
    $newAcademicYear = htmlspecialchars($_POST['new_academic_year'], ENT_QUOTES, 'UTF-8');
    if (!empty($newAcademicYear)) {
        try {
            // Update current academic year
            $stmt = $pdo->prepare("REPLACE INTO settings (setting_key, setting_value, description) VALUES (?, ?, ?)");
            $stmt->execute([$currentYearKey, $newAcademicYear, 'Current academic year for registration (YYYY-YYYY format)']);
            
            $_SESSION['successMessage'] = 'Current academic year changed to ' . $newAcademicYear . '.';
        } catch (Exception $e) {
            $_SESSION['errorMessage'] = 'Error changing academic year: ' . $e->getMessage();
        }
    } else {
        $_SESSION['errorMessage'] = 'Please enter a valid academic year.';
    }
    header('Location: upload_index_numbers.php');
    exit;
}

// Handle manual registration control
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['registration_control'])) {
    $action = htmlspecialchars($_POST['registration_control'], ENT_QUOTES, 'UTF-8');
    $academicYear = htmlspecialchars($_POST['academic_year'], ENT_QUOTES, 'UTF-8');
    
    if (!empty($action) && !empty($academicYear)) {
        try {
            $statusKey = 'registration_status_' . $academicYear;
            $newStatus = '';
            $message = '';
            
            switch ($action) {
                case 'open':
                    $newStatus = 'open';
                    $message = 'Registration manually opened for ' . $academicYear . '.';
                    break;
                case 'close':
                    $newStatus = 'closed';
                    $message = 'Registration manually closed for ' . $academicYear . '.';
                    break;
                case 'auto':
                    $newStatus = 'auto';
                    $message = 'Registration set to automatic mode for ' . $academicYear . ' (follows set dates).';
                    break;
                default:
                    $_SESSION['errorMessage'] = 'Invalid action.';
                    header('Location: upload_index_numbers.php');
                    exit;
            }
            
            $stmt = $pdo->prepare("REPLACE INTO settings (setting_key, setting_value, description) VALUES (?, ?, ?)");
            $stmt->execute([$statusKey, $newStatus, 'Manual registration status for ' . $academicYear . ' (auto/open/closed)']);
            
            $_SESSION['successMessage'] = $message;
        } catch (Exception $e) {
            $_SESSION['errorMessage'] = 'Error updating registration status: ' . $e->getMessage();
        }
    } else {
        $_SESSION['errorMessage'] = 'Invalid parameters.';
    }
    header('Location: upload_index_numbers.php');
    exit;
}

// Handle toggle action for Post-Internship functionality
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['toggle_action'])) {
    if ($_POST['toggle_action'] === 'activate') {
        // Update the persistent setting in the database
        $stmt = $pdo->prepare("REPLACE INTO settings (setting_key, setting_value) VALUES (?, ?)");
        $stmt->execute([$settingKey, '1']);
        $_SESSION['successMessage'] = 'Post-Internship functionality activated.';
    } elseif ($_POST['toggle_action'] === 'deactivate') {
        $stmt = $pdo->prepare("REPLACE INTO settings (setting_key, setting_value) VALUES (?, ?)");
        $stmt->execute([$settingKey, '0']);
        $_SESSION['successMessage'] = 'Post-Internship functionality deactivated.';
    }
    header('Location: upload_index_numbers.php');
    exit;
}

// Fetch distinct courses from student_details
$stmt = $pdo->prepare('SELECT DISTINCT `course` FROM student_details');
$stmt->execute();
$courses = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Index Numbers</title>
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body {
            background-color: #f5f5f5;
            color: #333;
            font-family: 'Arial', sans-serif;
        }
        .container {
            max-width: 900px;
            margin: 3rem auto;
        }
        .card {
            border: none;
            border-radius: 10px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
            transition: transform 0.3s ease;
        }
        .card:hover {
            transform: translateY(-5px);
        }
        .card-header {
            font-size: 1.25rem;
            border-bottom: none;
            background: #f8f9fa;
            text-align: center;
            padding: 1rem 0;
        }
        .btn {
            border-radius: 5px;
        }
        .btn-primary {
            background-color: #0066cc;
            border: none;
        }
        .btn-primary:hover {
            background-color: #0056b3;
        }
        .btn-info {
            background-color: #17a2b8;
            border: none;
        }
        .btn-info:hover {
            background-color: #138496;
        }
        .btn-success {
            background-color: #28a745;
            border: none;
        }
        .btn-success:hover {
            background-color: #218838;
        }
        .btn-warning {
            background-color: #ff9800;
            border: none;
        }
        .btn-warning:hover {
            background-color: #fb8c00;
        }
        .btn-danger {
            background-color: #dc3545;
            border: none;
        }
        .btn-danger:hover {
            background-color: #c82333;
        }
        .form-control {
            border: 1px solid #dee2e6;
            border-radius: 5px;
        }
        .alert {
            border-radius: 5px;
            font-size: 0.9rem;
        }
        .link-style {
            display: block;
            margin-top: 10px;
            color: #007bff;
            text-decoration: none;
            text-align: center;
        }
        .link-style:hover {
            text-decoration: underline;
        }
        h2 {
            font-weight: 600;
            text-align: center;
            margin-bottom: 2rem;
        }
        /* Toggle switch styling */
        .toggle-switch {
            position: relative;
            display: inline-block;
            width: 60px;
            height: 34px;
        }
        .toggle-switch input {
            opacity: 0;
            width: 0;
            height: 0;
        }
        .slider {
            position: absolute;
            cursor: pointer;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-color: #ccc;
            transition: 0.4s;
            border-radius: 34px;
        }
        .slider:before {
            position: absolute;
            content: "";
            height: 26px;
            width: 26px;
            left: 4px;
            bottom: 4px;
            background-color: white;
            transition: 0.4s;
            border-radius: 50%;
        }
        input:checked + .slider {
            background-color: #28a745;
        }
        input:checked + .slider:before {
            transform: translateX(26px);
        }
        .deadline-info {
            background-color: #e3f2fd;
            border: 1px solid #2196f3;
            border-radius: 5px;
            padding: 10px;
            margin-top: 10px;
            font-size: 0.9rem;
        }
        .deadline-info strong {
            color: #1976d2;
        }
    </style>
</head>
<body>
<?php include 'admin_header.php'; ?>

<div class="container">
    <h2>Manage Index Numbers & Download Acceptance Letters</h2>

    <?php if (!empty($successMessage)): ?>
        <div class="alert alert-success text-center">
            <?php echo htmlspecialchars($successMessage); ?>
        </div>
    <?php endif; ?>

    <?php if (!empty($errorMessage)): ?>
        <div class="alert alert-danger text-center">
            <?php echo htmlspecialchars($errorMessage); ?>
        </div>
    <?php endif; ?>

    <div class="row g-4">
        <!-- Academic Year Management -->
        <div class="col-12">
            <div class="card p-3">
                <h4 class="card-header">Academic Year Management</h4>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <form method="POST">
                                <div class="mb-3">
                                    <label for="new_academic_year" class="form-label">Change Current Academic Year:</label>
                                    <input type="text" class="form-control" name="new_academic_year" id="new_academic_year" 
                                           placeholder="YYYY-YYYY (e.g., 2024-2025)" pattern="\d{4}-\d{4}" required>
                                </div>
                                <div class="d-grid">
                                    <button type="submit" name="change_academic_year" class="btn btn-info">Change Academic Year</button>
                                </div>
                            </form>
                        </div>
                        <div class="col-md-6">
                            <div class="alert alert-info">
                                <strong>Current Academic Year:</strong> <?php echo htmlspecialchars($currentAcademicYear); ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- CSV Upload Form -->
        <div class="col-lg-6 col-md-12">
            <div class="card p-3">
                <h4 class="card-header">Upload CSV</h4>
                <div class="card-body">
                    <form method="POST" enctype="multipart/form-data">
                        <div class="mb-3">
                            <input type="file" class="form-control" name="csv_file" id="csv_file" required>
                        </div>
                        <div class="mb-3">
                            <label for="academic_year" class="form-label">Academic Year:</label>
                            <input type="text" class="form-control" name="academic_year" id="academic_year" value="<?php echo $currentAcademicYear; ?>" readonly>
                        </div>
                        <div class="d-grid mb-2">
                            <button type="submit" name="upload_csv" class="btn btn-primary">Upload CSV</button>
                        </div>
                        <a href="download_sample.php" class="link-style">Download Sample CSV</a>
                    </form>
                </div>
            </div>
        </div>

        <!-- Download Acceptance Letters by Course -->
        <div class="col-lg-6 col-md-12">
            <div class="card p-3">
                <h4 class="card-header">Download Acceptance Letters</h4>
                <div class="card-body">
                    <form method="POST" action="download_acceptance_letters.php">
                        <div class="mb-3">
                            <select name="course" class="form-select" required>
                                <option value="">Select Course</option>
                                <?php foreach ($courses as $course): ?>
                                    <option value="<?php echo htmlspecialchars($course['course'], ENT_QUOTES, 'UTF-8'); ?>">
                                        <?php echo htmlspecialchars($course['course'], ENT_QUOTES, 'UTF-8'); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="d-grid">
                            <button type="submit" name="download_acceptance_letters" class="btn btn-info">Download Letters</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <!-- Manual Add Form and Registration Deadline -->
    <div class="row g-4 mt-4">
        <div class="col-lg-6 col-md-12">
            <div class="card p-3">
                <h4 class="card-header">Add Index Number Manually</h4>
                <div class="card-body">
                    <form method="POST">
                        <div class="mb-3">
                            <input type="text" class="form-control" name="index_number" id="index_number" placeholder="Enter index number" required>
                        </div>
                        <div class="mb-3">
                            <label for="academic_year" class="form-label">Academic Year:</label>
                            <input type="text" class="form-control" name="academic_year" id="academic_year" value="<?php echo $currentAcademicYear; ?>" readonly>
                        </div>
                        <div class="d-grid">
                            <button type="submit" name="add_index_number" class="btn btn-success">Add Index Number</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <!-- Registration Deadline Setting -->
        <div class="col-lg-6 col-md-12">
            <div class="card p-3">
                <h4 class="card-header">Student Registration Period</h4>
                <div class="card-body">
                    <form method="POST">
                        <div class="mb-3">
                            <label for="academic_year" class="form-label">Academic Year:</label>
                            <input type="text" class="form-control" name="academic_year" id="academic_year" value="<?php echo $currentAcademicYear; ?>" readonly>
                        </div>
                        <div class="mb-3">
                            <label for="registration_start_date" class="form-label">Set Registration Start Date:</label>
                            <input type="datetime-local" class="form-control" name="registration_start_date" id="registration_start_date" 
                                   value="<?php echo $registrationStartDate ? date('Y-m-d\TH:i', strtotime($registrationStartDate)) : ''; ?>" required>
                        </div>
                        <div class="mb-3">
                            <label for="registration_end_date" class="form-label">Set Registration End Date:</label>
                            <input type="datetime-local" class="form-control" name="registration_end_date" id="registration_end_date" 
                                   value="<?php echo $registrationDeadline ? date('Y-m-d\TH:i', strtotime($registrationDeadline)) : ''; ?>" required>
                        </div>
                        <div class="d-grid">
                            <button type="submit" name="set_deadline" class="btn btn-warning">Set Registration Period</button>
                        </div>
                    </form>
                    <?php if ($registrationStartDate && $registrationDeadline): ?>
                        <div class="deadline-info">
                            <strong>Current Registration Period for <?php echo $currentAcademicYear; ?>:</strong>
                            <?php echo date('F j, Y \a\t g:i A', strtotime($registrationStartDate)); ?> - 
                            <?php echo date('F j, Y \a\t g:i A', strtotime($registrationDeadline)); ?>
                        </div>
                    <?php elseif ($registrationStartDate): ?>
                        <div class="deadline-info">
                            <strong>Current Registration Start for <?php echo $currentAcademicYear; ?>:</strong> <?php echo date('F j, Y \a\t g:i A', strtotime($registrationStartDate)); ?>
                        </div>
                    <?php elseif ($registrationDeadline): ?>
                        <div class="deadline-info">
                            <strong>Current Registration End for <?php echo $currentAcademicYear; ?>:</strong> <?php echo date('F j, Y \a\t g:i A', strtotime($registrationDeadline)); ?>
                        </div>
                    <?php else: ?>
                        <div class="deadline-info">
                            <strong>No registration period set for <?php echo $currentAcademicYear; ?>.</strong> Students can register at any time.
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

    <!-- Manual Registration Control -->
    <div class="row g-4 mt-4">
        <div class="col-12">
            <div class="card p-3">
                <h4 class="card-header">Manual Registration Control</h4>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-4">
                            <div class="alert alert-info">
                                <strong>Current Status:</strong> 
                                <?php 
                                switch($manualRegistrationStatus) {
                                    case 'open':
                                        echo '<span class="badge bg-success">Manually Open</span>';
                                        break;
                                    case 'closed':
                                        echo '<span class="badge bg-danger">Manually Closed</span>';
                                        break;
                                    case 'auto':
                                    default:
                                        echo '<span class="badge bg-primary">Automatic (Follows Dates)</span>';
                                        break;
                                }
                                ?>
                            </div>
                        </div>
                        <div class="col-md-8">
                            <form method="POST" class="d-inline">
                                <input type="hidden" name="academic_year" value="<?php echo $currentAcademicYear; ?>">
                                <button type="submit" name="registration_control" value="open" class="btn btn-success me-2">
                                    <i class="fas fa-unlock"></i> Open Registration
                                </button>
                                <button type="submit" name="registration_control" value="close" class="btn btn-danger me-2">
                                    <i class="fas fa-lock"></i> Close Registration
                                </button>
                                <button type="submit" name="registration_control" value="auto" class="btn btn-primary">
                                    <i class="fas fa-clock"></i> Set to Automatic
                                </button>
                            </form>
                        </div>
                    </div>
                    <div class="mt-3">
                        <small class="text-muted">
                            <strong>Manual Open:</strong> Registration is open regardless of dates<br>
                            <strong>Manual Close:</strong> Registration is closed regardless of dates<br>
                            <strong>Automatic:</strong> Registration follows the set start and end dates
                        </small>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Toggle for Post-Internship Functionality -->
    <div class="row g-4 mt-4">
        <div class="col-lg-6 col-md-12">
            <div class="card p-3">
                <h4 class="card-header">Post-Internship Functionality</h4>
                <div class="card-body text-center">
                    <form method="POST" id="toggleForm">
                        <label class="toggle-switch">
                            <input type="checkbox" name="toggle_post_internship" id="toggle_post_internship"
                                <?php echo ($postInternshipActive) ? 'checked' : ''; ?>>
                            <span class="slider"></span>
                        </label>
                        <!-- Hidden field for action will be appended by JavaScript -->
                    </form>
                    <br>
                    <p id="toggleStatus">
                        <?php 
                        echo ($postInternshipActive)
                            ? "Post-Internship functionality is activated."
                            : "Post-Internship functionality is deactivated.";
                        ?>
                    </p>
                    <br>
                    <?php if ($postInternshipActive): ?>
                        <a href="pi.php" class="btn btn-warning">Go to Post-Internship</a>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Bootstrap JS and Popper.js -->
<script src="https://cdn.jsdelivr.net/npm/@popperjs/core@2.10.2/dist/umd/popper.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.min.js"></script>

<script>
// Listen for changes on the toggle switch
document.getElementById('toggle_post_internship').addEventListener('change', function() {
    var form = document.getElementById('toggleForm');
    
    // Remove any existing hidden input if present
    var existing = document.querySelector('input[name="toggle_action"]');
    if (existing) {
        existing.parentNode.removeChild(existing);
    }
    
    // Create a new hidden input with the proper value based on toggle state
    var hiddenInput = document.createElement('input');
    hiddenInput.type = 'hidden';
    hiddenInput.name = 'toggle_action';
    hiddenInput.value = this.checked ? 'activate' : 'deactivate';
    form.appendChild(hiddenInput);
    
    // Optionally update the status text immediately (will be updated on page reload as well)
    document.getElementById('toggleStatus').innerText = this.checked 
        ? "Post-Internship functionality is activated." 
        : "Post-Internship functionality is deactivated.";
    
    // Submit the form automatically on toggle change
    form.submit();
});
</script>

</body>
</html>
