<?php
session_start();
require '../config/database.php';

// Admin session check
if (!isset($_SESSION['admin_id'])) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);
$studentResults = $input['student_results'] ?? [];

if (empty($studentResults)) {
    echo json_encode(['success' => false, 'message' => 'No student results provided']);
    exit;
}

try {
    $pdo->beginTransaction();
    
    $successCount = 0;
    $errorCount = 0;
    $errors = [];
    
    foreach ($studentResults as $result) {
        if (empty($result['student_id']) || $result['final_score'] === null) {
            continue;
        }
        
        // Check if this student already has a final summary record
        $stmtCheckExisting = $pdo->prepare("
            SELECT id FROM pi_student_grades 
            WHERE student_id = ? AND lecturer_id = 999999 AND grading_type = 'both'
            LIMIT 1
        ");
        $stmtCheckExisting->execute([$result['student_id']]);
        $existingRecord = $stmtCheckExisting->fetch(PDO::FETCH_ASSOC);
        
        try {
            if ($existingRecord) {
                // Update existing final summary record
                $stmtUpdate = $pdo->prepare("
                    UPDATE pi_student_grades 
                    SET 
                        final_score = ?,
                        report_score = ?,
                        presentation_score = ?,
                        graded_at = NOW()
                    WHERE id = ?
                ");
                $updateResult = $stmtUpdate->execute([
                    $result['final_score'],
                    $result['report_score'],
                    $result['presentation_score'],
                    $existingRecord['id']
                ]);
                
                if ($updateResult) {
                    $successCount++;
                } else {
                    $errorCount++;
                    $errors[] = "Failed to update student ID: " . $result['student_id'];
                }
            } else {
                // Create new final summary record
                $stmtInsert = $pdo->prepare("
                    INSERT INTO pi_student_grades (
                        student_id, 
                        lecturer_id, 
                        final_score, 
                        report_score, 
                        presentation_score, 
                        grading_type, 
                        graded_at
                    ) VALUES (?, 999999, ?, ?, ?, 'both', NOW())
                ");
                $insertResult = $stmtInsert->execute([
                    $result['student_id'],
                    $result['final_score'],
                    $result['report_score'],
                    $result['presentation_score']
                ]);
                
                if ($insertResult) {
                    $successCount++;
                } else {
                    $errorCount++;
                    $errors[] = "Failed to insert student ID: " . $result['student_id'];
                }
            }
        } catch (Exception $e) {
            $errorCount++;
            $errorMessage = $e->getMessage();
            
            // Provide more specific error messages for common database issues
            if (strpos($errorMessage, 'Duplicate entry') !== false) {
                $errors[] = "Student ID " . $result['student_id'] . ": Duplicate entry detected. This student already has a final summary record.";
            } elseif (strpos($errorMessage, 'Integrity constraint violation') !== false) {
                $errors[] = "Student ID " . $result['student_id'] . ": Database constraint violation. Please check if this student already has grades.";
            } else {
                $errors[] = "Error processing student ID " . $result['student_id'] . ": " . $errorMessage;
            }
        }
    }
    
    if ($errorCount === 0) {
        $pdo->commit();
        echo json_encode([
            'success' => true, 
            'message' => "All $successCount final results saved successfully"
        ]);
    } else {
        $pdo->rollBack();
        echo json_encode([
            'success' => false, 
            'message' => "Saved $successCount results, but $errorCount failed. Errors: " . implode('; ', $errors)
        ]);
    }
    
} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
}
?>
