<?php
session_start();
require '../config/database.php';

// Admin session check
if (!isset($_SESSION['admin_id'])) {
    header("Location: login");
    exit;
}

// Helper function to compute letter grade based on average score
function getLetterGrade($score)
{
    if ($score >= 80) {
        return 'A';
    } elseif ($score >= 75) {
        return 'B+';
    } elseif ($score >= 70) {
        return 'B';
    } elseif ($score >= 65) {
        return 'C+';
    } elseif ($score >= 60) {
        return 'C';
    } elseif ($score >= 55) {
        return 'D+';
    } elseif ($score >= 50) {
        return 'D';
    } else {
        return 'E';
    }
}

// Helper function to get grade color
function getGradeColor($score) {
    if ($score >= 80) return 'success';
    if ($score >= 70) return 'info';
    if ($score >= 60) return 'warning';
    return 'danger';
}

// Retrieve all lecturer groups
$stmtGroups = $pdo->query("SELECT * FROM pi_lecturer_groups ORDER BY group_name");
$groups = $stmtGroups->fetchAll(PDO::FETCH_ASSOC);

include('pi_header.php'); 
?>

<!-- Main Content -->
<div class="container mt-4">
    <!-- Page Header -->
    <div class="row mb-4">
        <div class="col-md-12">
            <h1 class="display-6 fw-bold text-primary">
                <i class="fas fa-chart-line me-2"></i>Student Scores Dashboard
            </h1>
            <p class="text-muted">View and analyze student performance across all groups</p>
        </div>
    </div>

    <!-- Filter and Export Section -->
    <div class="card shadow-sm mb-4">
        <div class="card-body">
            <form method="get" class="row g-3 align-items-end">
                <div class="col-md-8">
                    <label for="groupFilter" class="form-label fw-bold">
                        <i class="fas fa-filter me-2"></i>Filter by Group
                    </label>
                    <select id="groupFilter" name="group_id" class="form-select form-select-lg" onchange="this.form.submit()">
                        <option value="">-- Show All Groups --</option>
                        <?php foreach ($groups as $g): ?>
                            <option value="<?= $g['id'] ?>" <?= (isset($_GET['group_id']) && $_GET['group_id'] == $g['id']) ? 'selected' : '' ?>>
                                <?= htmlspecialchars($g['group_name']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-4 text-end">
                    <button type="button" class="btn btn-primary btn-sm me-2" id="calculateAllFinalBtn">
                        <i class="fas fa-calculator me-1"></i>Calculate Results
                    </button>
                    <a href="export_scores_excel.php<?= isset($_GET['group_id']) && $_GET['group_id'] !== '' ? '?group_id=' . $_GET['group_id'] : '' ?>" 
                       class="btn btn-success btn-lg">
                        <i class="fas fa-file-excel me-2"></i>Export to Excel
                    </a>
                </div>
            </form>
        </div>
    </div>
    
    <?php if ($groups): ?>
        <?php 
        $selectedGroupId = isset($_GET['group_id']) && $_GET['group_id'] !== '' ? $_GET['group_id'] : null;
        foreach ($groups as $group): 
            if ($selectedGroupId && $group['id'] != $selectedGroupId) continue;
        ?>
            <!-- Group Card -->
            <div class="card shadow-sm mb-4">
                <div class="card-header bg-primary text-white">
                    <h3 class="card-title mb-0">
                        <i class="fas fa-users me-2"></i><?php echo htmlspecialchars($group['group_name']); ?>
                    </h3>
                </div>
                <div class="card-body">
                    <?php
                    // Retrieve lecturers in this group
                    $stmtLecturers = $pdo->prepare("SELECT id, name FROM lecturer_login WHERE group_id = ?");
                    $stmtLecturers->execute([$group['id']]);
                    $lecturers = $stmtLecturers->fetchAll(PDO::FETCH_ASSOC);
                    
                    if ($lecturers):
                        // Create a comma-separated list of lecturer names
                        $lecturerNames = array_map(function($lect) {
                            return $lect['name'];
                        }, $lecturers);
                        $lecturerNamesStr = implode(', ', $lecturerNames);
                    ?>
                        <div class="alert alert-info mb-3">
                            <i class="fas fa-chalkboard-teacher me-2"></i>
                            <strong>Supervising Lecturers:</strong> <?php echo htmlspecialchars($lecturerNamesStr); ?>
                            <br>
                            <i class="fas fa-tag me-2"></i>
                            <strong>Group Type:</strong> <?php echo htmlspecialchars($group['category']); ?>
                        </div>
                    <?php else: ?>
                        <div class="alert alert-warning mb-3">
                            <i class="fas fa-exclamation-triangle me-2"></i>
                            <strong>No lecturers assigned to this group.</strong>
                        </div>
                    <?php endif; ?>
                    
                    <?php
                    // Retrieve students assigned to this group using lecturer_group_id
                    // Query to retrieve each student's details including their index number and average final score
                    try {
                                            $query = "
                        SELECT
                            s.id AS student_id,
                            CONCAT(s.surname, ' ', s.first_name, ' ', COALESCE(s.other_names, '')) AS student_name,
                            u.index_number,
                            s.course AS program,
                            u.group AS user_group,
                            ROUND(AVG(psg.final_score), 2) AS average_score
                            FROM student_details s
                            JOIN users u ON s.user_id = u.id
                            LEFT JOIN pi_student_grades psg ON s.id = psg.student_id
                        WHERE s.lecturer_group_id = ?
                        GROUP BY s.id
                        ORDER BY s.surname, s.first_name
                        ";
                    
                    $stmtStudents = $pdo->prepare($query);
                    $stmtStudents->execute([$group['id']]);
                        $students = $stmtStudents->fetchAll(PDO::FETCH_ASSOC);
                    } catch (PDOException $e) {
                        echo '<div class="alert alert-danger">Error fetching students: ' . htmlspecialchars($e->getMessage()) . '</div>';
                        $students = [];
                    }
                    ?>
                    
                    <?php if ($students): ?>
                        <div class="table-responsive">
                            <table class="table table-results">
                                <thead class="table-dark">
                                    <tr>
                                        <th><i class="fas fa-id-card me-1"></i>Index Number</th>
                                        <th><i class="fas fa-user me-1"></i>Student Name</th>
                                        <th><i class="fas fa-graduation-cap me-1"></i>Program</th>
                                        <th><i class="fas fa-file-alt me-1"></i>Report Score</th>
                                        <th><i class="fas fa-presentation me-1"></i>Presentation Score</th>
                                        <th><i class="fas fa-star me-1"></i>Final Score</th>
                                        <th><i class="fas fa-award me-1"></i>Grade</th>

                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($students as $student): ?>
                                        <tr>
                                            <td><strong><?php echo htmlspecialchars($student['index_number']); ?></strong></td>
                                            <td><?php echo htmlspecialchars($student['student_name']); ?></td>
                                            <td><?php echo htmlspecialchars($student['program']); ?></td>
                                            <?php if ($student['user_group'] === 'Teaching Practice'): ?>
                                                <td colspan="1" class="text-center text-muted">
                                                    <i class="fas fa-minus"></i> N/A
                                                </td>
                                                <td>
                                                    <?php
                                                    // Fetch average total_score from teaching_grades
                                                    $stmtTeach = $pdo->prepare("SELECT AVG(total_score) AS avg_score FROM teaching_grades WHERE student_id = ?");
                                                    $stmtTeach->execute([$student['student_id']]);
                                                    $teachScore = $stmtTeach->fetch(PDO::FETCH_ASSOC);
                                                    if ($teachScore && $teachScore['avg_score'] !== null) {
                                                        $finalScore = round($teachScore['avg_score'], 2);
                                                        echo htmlspecialchars($finalScore);
                                                    } else {
                                                        $finalScore = null;
                                                        echo '<span class="text-muted">Not Graded</span>';
                                                    }
                                                    ?>
                                                </td>
                                                <td>
                                                    <?php
                                                    if ($finalScore !== null) {
                                                        echo getLetterGrade($finalScore);
                                                    } else {
                                                        echo '<span class="text-muted">Not Graded</span>';
                                                    }
                                                    ?>
                                                </td>
                                                <td>
                                                    <button class="btn btn-primary btn-sm calculate-final-btn" 
                                                            data-student-id="<?php echo $student['student_id']; ?>"
                                                            data-student-name="<?php echo htmlspecialchars($student['student_name']); ?>">
                                                        <i class="fas fa-calculator me-1"></i>Calculate Final
                                                    </button>
                                                </td>
                                            <?php else: ?>
                                                <td>
                                                    <?php 
                                                    // First try to get existing report score from any grade record
                                                    $stmtReport = $pdo->prepare("
                                                        SELECT report_score FROM pi_student_grades 
                                                        WHERE student_id = ? AND report_score IS NOT NULL AND report_score > 0
                                                        ORDER BY graded_at DESC LIMIT 1
                                                    ");
                                                    $stmtReport->execute([$student['student_id']]);
                                                    $reportScore = $stmtReport->fetch(PDO::FETCH_ASSOC);
                                                    
                                                    // If no existing score, check final summary record
                                                    if (!$reportScore || $reportScore['report_score'] === null) {
                                                        $stmtReport = $pdo->prepare("
                                                            SELECT report_score FROM pi_student_grades 
                                                            WHERE student_id = ? AND lecturer_id = 999999 AND grading_type = 'both'
                                                            ORDER BY graded_at DESC LIMIT 1
                                                        ");
                                                        $stmtReport->execute([$student['student_id']]);
                                                        $reportScore = $stmtReport->fetch(PDO::FETCH_ASSOC);
                                                    }
                                                    
                                                    if ($reportScore && $reportScore['report_score'] !== null) {
                                                        echo htmlspecialchars($reportScore['report_score']);
                                                    } else {
                                                        echo '<span class="text-muted">Not Graded</span>';
                                                    }
                                                    ?>
                                                </td>
                                                <td>
                                                    <?php 
                                                    // First try to get existing presentation score from any grade record
                                                    $stmtPres = $pdo->prepare("
                                                        SELECT presentation_score FROM pi_student_grades 
                                                        WHERE student_id = ? AND presentation_score IS NOT NULL AND presentation_score > 0
                                                        ORDER BY graded_at DESC LIMIT 1
                                                    ");
                                                    $stmtPres->execute([$student['student_id']]);
                                                    $presScore = $stmtPres->fetch(PDO::FETCH_ASSOC);
                                                    
                                                    // If no existing score, check final summary record
                                                    if (!$presScore || $presScore['presentation_score'] === null) {
                                                        $stmtPres = $pdo->prepare("
                                                            SELECT presentation_score FROM pi_student_grades 
                                                            WHERE student_id = ? AND lecturer_id = 999999 AND grading_type = 'both'
                                                            ORDER BY graded_at DESC LIMIT 1
                                                        ");
                                                        $stmtPres->execute([$student['student_id']]);
                                                        $presScore = $stmtPres->fetch(PDO::FETCH_ASSOC);
                                                    }
                                                    
                                                    if ($presScore && $presScore['presentation_score'] !== null) {
                                                        echo htmlspecialchars($presScore['presentation_score']);
                                                    } else {
                                                        echo '<span class="text-muted">Not Graded</span>';
                                                    }
                                                    ?>
                                                </td>
                                                <td>
                                                    <?php
                                                    // First try to get existing final score from any grade record
                                                    $stmtFinal = $pdo->prepare("
                                                        SELECT final_score FROM pi_student_grades 
                                                        WHERE student_id = ? AND final_score IS NOT NULL AND final_score > 0
                                                        ORDER BY graded_at DESC LIMIT 1
                                                    ");
                                                    $stmtFinal->execute([$student['student_id']]);
                                                    $finalScore = $stmtFinal->fetch(PDO::FETCH_ASSOC);
                                                    
                                                    // If no existing score, check final summary record
                                                    if (!$finalScore || $finalScore['final_score'] === null) {
                                                        $stmtFinal = $pdo->prepare("
                                                            SELECT final_score FROM pi_student_grades 
                                                            WHERE student_id = ? AND lecturer_id = 999999 AND grading_type = 'both'
                                                            ORDER BY graded_at DESC LIMIT 1
                                                        ");
                                                        $stmtFinal->execute([$student['student_id']]);
                                                        $finalScore = $stmtFinal->fetch(PDO::FETCH_ASSOC);
                                                    }
                                                    
                                                    if ($finalScore && $finalScore['final_score'] !== null) {
                                                        echo htmlspecialchars($finalScore['final_score']);
                                                    } else {
                                                        echo '<span class="text-muted">Not Graded</span>';
                                                    }
                                                    ?>
                                                </td>
                                                <td>
                                                    <?php
                                                    if ($finalScore && $finalScore['final_score'] !== null) {
                                                        echo getLetterGrade($finalScore['final_score']);
                                                    } else {
                                                        echo '<span class="text-muted">Not Graded</span>';
                                                    }
                                                    ?>
                                                </td>
                                            <?php endif; ?>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php else: ?>
                        <div class="alert alert-info text-center">
                            <i class="fas fa-info-circle me-2"></i>
                            <strong>No students assigned to this group.</strong>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        <?php endforeach; ?>
    <?php else: ?>
        <div class="alert alert-warning text-center">
            <i class="fas fa-exclamation-triangle me-2"></i>
            <strong>No groups found.</strong>
        </div>
    <?php endif; ?>
</div>

<style>
/* Modern Results Table Styling */
.table-results {
    border-radius: 12px;
    overflow: hidden;
    box-shadow: 0 2px 12px rgba(31,42,56,0.07);
    background: #fff;
    margin-bottom: 2rem;
}
.table-results th {
    background: #1f2a38;
    color: #fff;
    font-weight: 600;
    text-transform: uppercase;
    font-size: 0.95rem;
    letter-spacing: 0.5px;
    border: none;
    padding: 0.85rem 0.75rem;
}
.table-results td {
    background: #f9fafb;
    border: none;
    padding: 0.85rem 0.75rem;
    vertical-align: middle;
    font-size: 1rem;
}
.table-results tbody tr:nth-child(even) td {
    background: #f1f3f6;
}
.table-results tbody tr:hover td {
    background: #e6eaf1;
    transition: background 0.2s;
}
.table-results td, .table-results th {
    border-right: 1px solid #e3e6ed;
}
.table-results td:last-child, .table-results th:last-child {
    border-right: none;
}

.card.shadow-sm, .card.shadow-lg {
    border-radius: 14px;
    box-shadow: 0 2px 16px rgba(31,42,56,0.10);
    border: none;
}

.card-header.bg-primary {
    border-radius: 14px 14px 0 0;
    background: #1f2a38 !important;
    color: #fff !important;
    border-bottom: 1px solid #e3e6ed;
}

.alert {
    border-radius: 10px;
    font-size: 1rem;
    box-shadow: 0 2px 8px rgba(31,42,56,0.07);
}

@media (max-width: 768px) {
    .table-results th, .table-results td {
        font-size: 0.95rem;
        padding: 0.65rem 0.5rem;
    }
    .card-header.bg-primary {
        font-size: 1.1rem;
    }
}



/* Modal Styling */
.modal-content {
    border-radius: 15px;
    border: none;
    box-shadow: 0 10px 30px rgba(0, 0, 0, 0.2);
}

.modal-header.bg-primary {
    border-radius: 15px 15px 0 0;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%) !important;
}

.modal-body {
    padding: 2rem;
}

/* Card Styling in Modal */
.card {
    border-radius: 12px;
    border: none;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
    transition: transform 0.2s ease;
}

.card:hover {
    transform: translateY(-2px);
}

.card-header {
    border-radius: 12px 12px 0 0;
    font-weight: 600;
}
</style>



<!-- JavaScript for Calculate All Final Results functionality -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Handle Calculate All Final Results button click
    document.getElementById('calculateAllFinalBtn').addEventListener('click', function() {
                    // Show loading state
            this.innerHTML = '<i class="fas fa-spinner fa-spin me-1"></i>Calculating...';
        this.disabled = true;
        
        // Get selected group ID if any
        const selectedGroupId = document.getElementById('groupFilter')?.value || '';
        
        // Start bulk calculation
        calculateAllFinalResults(selectedGroupId);
    });
    
    // Function to calculate final results for all students
    function calculateAllFinalResults(groupId = '') {
        fetch('calculate_all_final_results.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                group_id: groupId
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showBulkCalculationResults(data);
            } else {
                showError('Error calculating final results: ' + data.message);
                resetButton();
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showError('Error calculating final results. Please try again.');
            resetButton();
        });
    }
    
    // Function to display bulk calculation results
    function showBulkCalculationResults(data) {
        // Create and show results modal
        const modalHtml = `
            <div class="modal fade" id="bulkCalculationModal" tabindex="-1" aria-labelledby="bulkCalculationModalLabel" aria-hidden="true">
                <div class="modal-dialog modal-xl">
                    <div class="modal-content">
                        <div class="modal-header bg-success text-white">
                            <h5 class="modal-title" id="bulkCalculationModalLabel">
                                <i class="fas fa-check-circle me-2"></i>Calculation Complete & Results Saved
                            </h5>
                            <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                        </div>
                        <div class="modal-body">
                            <div class="alert alert-success">
                                <i class="fas fa-check-circle me-2"></i>
                                <strong>Calculation Complete:</strong> ${data.students_with_grades} students with BOTH report AND presentation scores processed and results automatically saved to database
                                ${data.students_without_grades > 0 ? `<br><small class="text-muted">${data.students_without_grades} students without complete grades were left untouched</small>` : ''}
                            </div>
                            
                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <div class="card border-primary">
                                        <div class="card-header bg-primary text-white">
                                            <h6 class="mb-0"><i class="fas fa-file-alt me-2"></i>Report Scores Summary</h6>
                                        </div>
                                        <div class="card-body">
                                            <p class="mb-2"><strong>Students with Report Scores:</strong> ${data.report_summary.count}</p>
                                            <p class="mb-2"><strong>Average Report Score:</strong> ${data.report_summary.average}</p>
                                            <p class="mb-0"><strong>Total Report Score:</strong> ${data.report_summary.total}</p>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="card border-success">
                                        <div class="card-header bg-success text-white">
                                            <h6 class="mb-0"><i class="fas fa-presentation me-2"></i>Presentation Scores Summary</h6>
                                        </div>
                                        <div class="card-body">
                                            <p class="mb-2"><strong>Students with Presentation Scores:</strong> ${data.report_summary.count}</p>
                                            <p class="mb-2"><strong>Average Presentation Score:</strong> ${data.presentation_summary.average}</p>
                                            <p class="mb-0"><strong>Total Presentation Score:</strong> ${data.presentation_summary.total}</p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row mb-3">
                                <div class="col-12">
                                    <div class="card border-warning">
                                        <div class="card-header bg-warning text-dark">
                                            <h6 class="mb-0"><i class="fas fa-calculator me-2"></i>Final Results Summary</h6>
                                        </div>
                                        <div class="card-body">
                                            <p class="mb-2"><strong>Students with Final Scores:</strong> ${data.final_summary.count}</p>
                                            <p class="mb-2"><strong>Average Final Score:</strong> ${data.final_summary.average}</p>
                                            <p class="mb-0"><strong>Grade Distribution:</strong></p>
                                            <div class="row mt-2">
                                                <div class="col-md-2">
                                                    <span class="badge bg-success">A: ${data.grade_distribution.A || 0}</span>
                                                </div>
                                                <div class="col-md-2">
                                                    <span class="badge bg-info">B+: ${data.grade_distribution.B_plus || 0}</span>
                                                </div>
                                                <div class="col-md-2">
                                                    <span class="badge bg-info">B: ${data.grade_distribution.B || 0}</span>
                                                </div>
                                                <div class="col-md-2">
                                                    <span class="badge bg-warning">C+: ${data.grade_distribution.C_plus || 0}</span>
                                                </div>
                                                <div class="col-md-2">
                                                    <span class="mb-2"><strong>Grade Distribution:</strong></p>
                                            <div class="row mt-2">
                                                <div class="col-md-2">
                                                    <span class="badge bg-success">A: ${data.grade_distribution.A || 0}</span>
                                                </div>
                                                <div class="mb-2"><strong>Grade Distribution:</strong></p>
                                            <div class="row mt-2">
                                                <div class="col-md-2">
                                                    <span class="badge bg-success">A: ${data.grade_distribution.A || 0}</span>
                                                </div>
                                                <div class="col-md-2">
                                                    <span class="badge bg-info">B+: ${data.grade_distribution.B_plus || 0}</span>
                                                </div>
                                                <div class="col-md-2">
                                                    <span class="badge bg-info">B: ${data.grade_distribution.B || 0}</span>
                                                </div>
                                                <div class="col-md-2">
                                                    <span class="badge bg-warning">C+: ${data.grade_distribution.C_plus || 0}</span>
                                                </div>
                                                <div class="col-md-2">
                                                    <span class="badge bg-warning">C: ${data.grade_distribution.C || 0}</span>
                                                </div>
                                                <div class="col-md-2">
                                                    <span class="badge bg-danger">D/E: ${data.grade_distribution.D_E || 0}</span>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-12">
                                    <div class="card border-info">
                                        <div class="card-header bg-info text-white">
                                            <h6 class="mb-0"><i class="fas fa-list me-2"></i>Student Results</h6>
                                        </div>
                                        <div class="card-body">
                                            <div class="table-responsive" style="max-height: 400px; overflow-y: auto;">
                                                <table class="table table-sm table-striped">
                                                    <thead class="table-dark">
                                                        <tr>
                                                            <th>Student Name</th>
                                                            <th>Program</th>
                                                            <th>Report Score</th>
                                                            <th>Presentation Score</th>
                                                            <th>Final Score</th>
                                                            <th>Grade</th>
                                                        </tr>
                                                    </thead>
                                                    <tbody>
                                                        ${data.student_results.map(student => `
                                                            <tr>
                                                                <td>${student.student_name}</td>
                                                                <td>${student.program}</td>
                                                                <td>${student.report_score || 'N/A'}</td>
                                                                <td>${student.presentation_score || 'N/A'}</td>
                                                                <td><strong>${student.final_score || 'N/A'}</strong></td>
                                                                <td><span class="badge bg-${getGradeColor(student.letter_grade)}">${student.letter_grade || 'N/A'}</span></td>
                                                            </tr>
                                                        `).join('')}
                                                    </tbody>
                                                </table>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="modal-footer">
                            <div class="text-start text-muted small me-auto">
                                <i class="fas fa-info-circle me-1"></i>
                                Only students with BOTH report AND presentation scores were processed. Students with incomplete grades remain unchanged.
                            </div>
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                        </div>
                    </div>
                </div>
            </div>
        `;
        
        // Remove existing modal if any
        const existingModal = document.getElementById('bulkCalculationModal');
        if (existingModal) {
            existingModal.remove();
        }
        
        // Add modal to body
        document.body.insertAdjacentHTML('beforeend', modalHtml);
        
        // Show modal
        const modal = new bootstrap.Modal(document.getElementById('bulkCalculationModal'));
        modal.show();
        
        // Auto-save results after calculation
        saveAllResults(data.student_results);
        
        // Reset button
        resetButton();
    }
    
    // Function to save all results
    function saveAllResults(studentResults) {
        fetch('save_all_final_results.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                student_results: studentResults
            })
        })
        .then(response => response.json())
        .then(result => {
            if (result.success) {
                // Show success message and reload page after 2 seconds
                document.getElementById('bulkCalculationModal').querySelector('.modal-body').innerHTML = 
                    '<div class="alert alert-success text-center"><i class="fas fa-check-circle me-2"></i>All final results saved successfully! Page will reload in 2 seconds...</div>';
                
                // Reload page after 2 seconds to show updated scores
                setTimeout(() => {
                    location.reload();
                }, 2000);
            } else {
                // Show error message
                document.getElementById('bulkCalculationModal').querySelector('.modal-body').innerHTML = 
                    '<div class="alert alert-danger text-center"><i class="fas fa-exclamation-triangle me-2"></i>Error saving results: ' + result.message + '</div>';
            }
        })
        .catch(error => {
            console.error('Error:', error);
            // Show error message
            document.getElementById('bulkCalculationModal').querySelector('.modal-body').innerHTML = 
                '<div class="alert alert-danger text-center"><i class="fas fa-exclamation-triangle me-2"></i>Error saving results. Please try again.</div>';
        });
    }
    
    // Helper function to get grade color
    function getGradeColor(grade) {
        if (grade === 'A') return 'success';
        if (grade === 'B+' || grade === 'B') return 'info';
        if (grade === 'C+' || grade === 'C') return 'warning';
        return 'danger';
    }
    
    // Function to show error
    function showError(message) {
        // Create and show error alert
        const alertHtml = `
            <div class="alert alert-danger alert-dismissible fade show" role="alert" style="position: fixed; top: 20px; right: 20px; z-index: 9999; min-width: 300px;">
                <i class="fas fa-exclamation-triangle me-2"></i>${message}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        `;
        document.body.insertAdjacentHTML('beforeend', alertHtml);
        
        // Auto-remove after 5 seconds
        setTimeout(() => {
            const alert = document.querySelector('.alert-danger');
            if (alert) alert.remove();
        }, 5000);
    }
    
    // Function to reset button
    function resetButton() {
        const btn = document.getElementById('calculateAllFinalBtn');
        btn.innerHTML = '<i class="fas fa-calculator me-1"></i>Calculate Results';
        btn.disabled = false;
    }
});
</script>

<?php include('pi_footer.php'); ?>
