<?php
// Start output buffering to prevent accidental output
ob_start();
require '../vendor/autoload.php'; // Ensure you have PHPSpreadsheet installed
include '../config/database.php';

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Worksheet\PageSetup;

// ----------------------------------------------------------
// 1) Fetch all lecturers and their assigned districts
// ----------------------------------------------------------
$query = '
    SELECT 
        ll.id AS lecturer_id,
        ll.name AS lecturer_name,
        lad.district AS assigned_district,
        lad.region AS assigned_region
    FROM lecturer_login ll
    JOIN lecturer_assigned_districts lad ON ll.id = lad.lecturer_id
    ORDER BY ll.name ASC, lad.region ASC, lad.district ASC
';
$stmt = $pdo->prepare($query);
$stmt->execute();
$lecturerDistricts = $stmt->fetchAll(PDO::FETCH_ASSOC);

// ----------------------------------------------------------
// 2) Fetch all students with "submitted" status INCLUDING telephone
// ----------------------------------------------------------
$query = '
    SELECT 
        id AS student_id,
        lecturer_id,
        district,
        region,
        first_name,
        surname,
        organisation_name,
        location,
        telephone
    FROM student_details 
    WHERE status = "submitted"
    ORDER BY lecturer_id ASC, region ASC, district ASC, surname ASC
';
$stmt = $pdo->prepare($query);
$stmt->execute();
$submittedStudents = $stmt->fetchAll(PDO::FETCH_ASSOC);

// ----------------------------------------------------------
// 3) Organize data by lecturer and region/district
// ----------------------------------------------------------
$lecturerData = [];
$totalStudentsAssigned = 0;

foreach ($lecturerDistricts as $row) {
    $lecturerId = $row['lecturer_id'];
    $lecturerName = ucwords(strtolower($row['lecturer_name'])); // Proper case
    $region = ucwords(strtolower($row['assigned_region']));
    $district = ucwords(strtolower($row['assigned_district']));

    if (!isset($lecturerData[$lecturerId])) {
        $lecturerData[$lecturerId] = [
            'name' => $lecturerName,
            'regions' => [],
            'total_students' => 0
        ];
    }
    if (!isset($lecturerData[$lecturerId]['regions'][$region])) {
        $lecturerData[$lecturerId]['regions'][$region] = [];
    }

    // Ensure each region/district entry exists (even if no students)
    $lecturerData[$lecturerId]['regions'][$region][$district] = [];
}

// ----------------------------------------------------------
// 4) Place each "submitted" student under the correct lecturer
// ----------------------------------------------------------
foreach ($submittedStudents as $student) {
    $lecturerId = $student['lecturer_id'];
    $region = ucwords(strtolower($student['region']));
    $district = ucwords(strtolower($student['district']));

    // Only assign if that region & district exists under that lecturer
    if (isset($lecturerData[$lecturerId]['regions'][$region][$district])) {
        $lecturerData[$lecturerId]['regions'][$region][$district][] = [
            'name'         => ucwords(strtolower($student['first_name'] . ' ' . $student['surname'])), 
            'organisation' => $student['organisation_name'],
            'location'     => $student['location'],
            'telephone'    => $student['telephone']
        ];
        $lecturerData[$lecturerId]['total_students']++;
        $totalStudentsAssigned++;
    }
}

// ----------------------------------------------------------
// 5) Calculate summary stats
// ----------------------------------------------------------
$totalSubmittedStudents   = count($submittedStudents);
$totalUnassignedStudents  = $totalSubmittedStudents - $totalStudentsAssigned;

// ----------------------------------------------------------
// 6) Create the Spreadsheet
// ----------------------------------------------------------
$spreadsheet = new Spreadsheet();

// ----------------------------------------------------------
// 6a) Summary Sheet
// ----------------------------------------------------------
$summarySheet = $spreadsheet->getActiveSheet();
$summarySheet->setTitle("Summary");

// Set to Landscape
$summarySheet->getPageSetup()->setOrientation(PageSetup::ORIENTATION_LANDSCAPE);

// Set Summary Headers
$summarySheet->setCellValue('A1', 'Summary of Assigned Students');
$summarySheet->getStyle('A1')->getFont()->setBold(true)->setSize(14);
$summarySheet->mergeCells('A1:B1');

$summarySheet->setCellValue('A3', 'Total Students with Submitted Status:');
$summarySheet->setCellValue('B3', $totalSubmittedStudents);
$summarySheet->getStyle('A3')->getFont()->setBold(true);

$summarySheet->setCellValue('A4', 'Total Students Assigned to Lecturers:');
$summarySheet->setCellValue('B4', $totalStudentsAssigned);
$summarySheet->getStyle('A4')->getFont()->setBold(true);

$summarySheet->setCellValue('A5', 'Total Students Not Assigned to Any Lecturer:');
$summarySheet->setCellValue('B5', $totalUnassignedStudents);
$summarySheet->getStyle('A5')->getFont()->setBold(true);

// Lecturer-wise student count
$summarySheet->setCellValue('A7', 'Lecturer Name');
$summarySheet->setCellValue('B7', 'Number of Assigned Students');
$summarySheet->getStyle('A7:B7')->getFont()->setBold(true);

$summaryRow = 8;
foreach ($lecturerData as $lecturerId => $lecturer) {
    $summarySheet->setCellValue("A$summaryRow", $lecturer['name']);
    $summarySheet->setCellValue("B$summaryRow", $lecturer['total_students']);
    $summaryRow++;
}

// Auto-size columns A and B
foreach (range('A', 'B') as $col) {
    $summarySheet->getColumnDimension($col)->setAutoSize(true);
}

// ----------------------------------------------------------
// 6b) Create a separate sheet for each lecturer
// ----------------------------------------------------------
foreach ($lecturerData as $lecturerId => $lecturer) {
    $lecturerName = $lecturer['name'];

    // Create a new worksheet for this lecturer
    $sheet = new \PhpOffice\PhpSpreadsheet\Worksheet\Worksheet($spreadsheet, $lecturerName);
    $spreadsheet->addSheet($sheet);

    // Set to Landscape
    $sheet->getPageSetup()->setOrientation(PageSetup::ORIENTATION_LANDSCAPE);

    $row = 1;

    // Lecturer Name as a header
    $sheet->setCellValue("A$row", "Lecturer: " . $lecturerName);
    $sheet->getStyle("A$row")->getFont()->setBold(true)->setSize(16);
    $sheet->mergeCells("A$row:D$row");
    $sheet->getStyle("A$row")->getAlignment()->setHorizontal(\PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER);
    $row += 2;

    // Regions and Districts
    foreach ($lecturer['regions'] as $region => $districts) {
        // Region Title
        $sheet->setCellValue("A$row", strtoupper($region));
        $sheet->getStyle("A$row")->getFont()->setBold(true)->setSize(14);
        $row++;

        foreach ($districts as $district => $students) {
            // District Title
            $sheet->setCellValue("A$row", "District: " . strtoupper($district));
            $sheet->getStyle("A$row")->getFont()->setBold(true)->setSize(12);
            $row++;

            // Table Headers (Now we have 4 columns: Name, Organisation, Location, Telephone)
            $sheet->setCellValue("A$row", "Student Name");
            $sheet->setCellValue("B$row", "Organisation");
            $sheet->setCellValue("C$row", "Location");
            $sheet->setCellValue("D$row", "Telephone");
            $sheet->getStyle("A$row:D$row")->getFont()->setBold(true);
            $row++;

            if (!empty($students)) {
                foreach ($students as $student) {
                    $sheet->setCellValue("A$row", $student['name']);
                    $sheet->setCellValue("B$row", $student['organisation']);
                    $sheet->setCellValue("C$row", $student['location']);
                    // *** NEW: Telephone value ***
                    $sheet->setCellValue("D$row", $student['telephone']);
                    $row++;
                }
            } else {
                $sheet->setCellValue("A$row", "No Students Assigned");
                $sheet->getStyle("A$row")->getFont()->setItalic(true);
                $row++;
            }

            $row += 2; // Blank line for spacing
        }
    }

    // Add total for this lecturer at the bottom
    $sheet->setCellValue("A$row", "Total Students Assigned: " . $lecturer['total_students']);
    $sheet->getStyle("A$row")->getFont()->setBold(true)->setSize(12);

    // Auto-size columns A through D
    foreach (range('A', 'D') as $col) {
        $sheet->getColumnDimension($col)->setAutoSize(true);
    }
}

// ----------------------------------------------------------
// 7) Set Active sheet back to Summary and output
// ----------------------------------------------------------
$spreadsheet->setActiveSheetIndex(0);

// Prepare to send the file to browser
header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
header('Content-Disposition: attachment; filename="Assigned_Students_' . date('Y-m-d') . '.xlsx"');
header('Cache-Control: max-age=0');

// Write file to output
$writer = new Xlsx($spreadsheet);
$writer->save('php://output');
exit();
