<?php
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

include '../config/database.php';

// Fetch distinct regions for filtering (from student_details)
$query = 'SELECT DISTINCT region 
          FROM student_details 
          WHERE district IS NOT NULL 
            AND district != ""';
$stmt = $pdo->prepare($query);
$stmt->execute();
$regions = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get the selected filters
$regionFilter        = isset($_GET['region_filter']) ? htmlspecialchars($_GET['region_filter'], ENT_QUOTES, 'UTF-8') : '';
$districtFilter      = isset($_GET['district_filter']) ? htmlspecialchars($_GET['district_filter'], ENT_QUOTES, 'UTF-8') : '';
$teachingPractice    = (isset($_GET['group_filter']) && $_GET['group_filter'] === 'Teaching Practice') ? 'Teaching Practice' : '';
$industrialInternship= (isset($_GET['group_filter_industrial']) && $_GET['group_filter_industrial'] === 'Industrial Internship') ? 'Industrial Internship' : '';
$unassignedFilter    = isset($_GET['unassigned_filter']) && $_GET['unassigned_filter'] === 'on';
$sortByOrganisation  = (isset($_GET['sort_by_organisation']) && $_GET['sort_by_organisation'] === 'true');

// Build the base query
$query = '
    SELECT 
        sd.id AS student_id,
        sd.first_name,
        sd.surname,
        sd.district,
        sd.region,
        sd.lecturer_id,
        sd.organisation_name,
        sd.location,
        u.`group` AS user_group
    FROM student_details sd
    JOIN users u ON sd.user_id = u.id
    WHERE sd.status = "submitted"
      AND sd.district IS NOT NULL
      AND sd.district != ""';

// Apply filters
if ($regionFilter) {
    $query .= ' AND sd.region = :region_filter';
}
if ($districtFilter) {
    $query .= ' AND sd.district = :district_filter';
}

// Build OR logic for the group checkboxes
$groupConditions = [];
if ($teachingPractice) {
    $groupConditions[] = 'u.`group` = :group_teaching';
}
if ($industrialInternship) {
    $groupConditions[] = 'u.`group` = :group_industrial';
}
if (!empty($groupConditions)) {
    $query .= ' AND (' . implode(' OR ', $groupConditions) . ')';
}

// Unassigned?
if ($unassignedFilter) {
    $query .= ' AND sd.lecturer_id IS NULL';
}

// Sort?
if ($sortByOrganisation) {
    $query .= ' ORDER BY sd.organisation_name ASC';
}

// Prepare statement
$stmt = $pdo->prepare($query);

// Bind parameters
if ($regionFilter) {
    $stmt->bindParam(':region_filter', $regionFilter);
}
if ($districtFilter) {
    $stmt->bindParam(':district_filter', $districtFilter);
}
if ($teachingPractice) {
    $stmt->bindParam(':group_teaching', $teachingPractice);
}
if ($industrialInternship) {
    $stmt->bindParam(':group_industrial', $industrialInternship);
}

// Execute and fetch
$stmt->execute();
$students = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch lecturers and their assigned districts
$query = 'SELECT ll.id AS lecturer_id, ll.name, ll.profile_image, lad.district
          FROM lecturer_login ll
          JOIN lecturer_assigned_districts lad ON ll.id = lad.lecturer_id';
$stmt = $pdo->prepare($query);
$stmt->execute();
$lecturers = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Build an array keyed by district => [lecturer_id => lecturer_info]
$lecturersByDistrict = [];
foreach ($lecturers as $lect) {
    $district = $lect['district'];
    $lectId   = $lect['lecturer_id'];

    if (!isset($lecturersByDistrict[$district])) {
        $lecturersByDistrict[$district] = [];
    }
    // Only add if not already present
    if (!array_key_exists($lectId, $lecturersByDistrict[$district])) {
        $lecturersByDistrict[$district][$lectId] = $lect;
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Assign Students to Lecturers</title>
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.1.1/css/all.min.css">
    <link rel="stylesheet" href="button.css">
    <style>
        /* Ensure buttons have no rounded corners */
        .btn {
            border-radius: 0;
        }
    </style>
</head>
<body>

<?php include 'admin_header.php'; ?>

<div class="container mt-4">
    <h2>Assign Students to Lecturers</h2>

    <!-- FILTERS FORM (GET) -->
    <form method="GET" action="assign_students.php" class="mb-4">
        <div class="row">
            <!-- Region Filter -->
            <div class="col-md-4">
                <select name="region_filter" class="form-select" onchange="this.form.submit()">
                    <option value="">All Regions</option>
                    <?php foreach ($regions as $region): ?>
                        <option value="<?php echo htmlspecialchars($region['region'], ENT_QUOTES, 'UTF-8'); ?>"
                            <?php echo $regionFilter === $region['region'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($region['region'], ENT_QUOTES, 'UTF-8'); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <!-- District Filter -->
            <div class="col-md-4">
                <select name="district_filter" id="district_filter" class="form-select" onchange="this.form.submit()">
                    <option value="">All Districts</option>
                    <!-- Populated by JS below -->
                </select>
            </div>

            <!-- Checkboxes -->
            <div class="col-md-4">
                <div class="form-check">
                    <input class="form-check-input"
                           type="checkbox"
                           name="group_filter"
                           value="Teaching Practice"
                           id="groupFilter"
                           onchange="this.form.submit()"
                           <?php echo ($teachingPractice === 'Teaching Practice') ? 'checked' : ''; ?>>
                    <label class="form-check-label" for="groupFilter">Teaching Practice</label>
                </div>
                <div class="form-check">
                    <input class="form-check-input"
                           type="checkbox"
                           name="group_filter_industrial"
                           value="Industrial Internship"
                           id="groupFilterIndustrial"
                           onchange="this.form.submit()"
                           <?php echo ($industrialInternship === 'Industrial Internship') ? 'checked' : ''; ?>>
                    <label class="form-check-label" for="groupFilterIndustrial">Industrial Internship</label>
                </div>
                <div class="form-check">
                    <input class="form-check-input"
                           type="checkbox"
                           name="unassigned_filter"
                           id="unassignedFilter"
                           onchange="this.form.submit()"
                           <?php echo $unassignedFilter ? 'checked' : ''; ?>>
                    <label class="form-check-label" for="unassignedFilter">Show Unassigned Students</label>
                </div>
            </div>
        </div>

        <!-- Sort by Organisation Button -->
        <div class="mt-3">
            <button type="submit" name="sort_by_organisation" value="true" class="btn btn-secondary">
                Sort by Organisation
            </button>
        </div>

        <!-- View Assigned Students Button -->
        <div class="mt-3">
            <a href="view_assigned_students.php" class="btn btn-info">
                View Assigned Students
            </a>
        </div>
    </form>

    <!-- ASSIGNMENT FORM (POST) -->
    <form id="assignmentForm" method="POST" action="process_assignment.php">

        <!-- Preserve current filters on POST (so we can redirect back to them later) -->
        <input type="hidden" name="region_filter" 
               value="<?php echo htmlspecialchars($regionFilter, ENT_QUOTES, 'UTF-8'); ?>">
        <input type="hidden" name="district_filter" 
               value="<?php echo htmlspecialchars($districtFilter, ENT_QUOTES, 'UTF-8'); ?>">
        <input type="hidden" name="group_filter" 
               value="<?php echo htmlspecialchars($teachingPractice, ENT_QUOTES, 'UTF-8'); ?>">
        <input type="hidden" name="group_filter_industrial" 
               value="<?php echo htmlspecialchars($industrialInternship, ENT_QUOTES, 'UTF-8'); ?>">
        <input type="hidden" name="unassigned_filter" 
               value="<?php echo $unassignedFilter ? 'on' : ''; ?>">
        <input type="hidden" name="sort_by_organisation" 
               value="<?php echo $sortByOrganisation ? 'true' : ''; ?>">

        <table class="table table-striped">
            <thead>
                <tr>
                    <th>Student Name</th>
                    <th>Chosen District</th>
                    <th>Organisation</th>
                    <th>Town</th>
                    <th>Available Lecturers</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($students as $student): ?>
                    <tr>
                        <td>
                            <?php echo htmlspecialchars($student['first_name'] . ' ' . $student['surname'], ENT_QUOTES, 'UTF-8'); ?>
                        </td>
                        <td>
                            <?php echo htmlspecialchars($student['district'], ENT_QUOTES, 'UTF-8'); ?>
                        </td>
                        <td>
                            <?php echo htmlspecialchars($student['organisation_name'], ENT_QUOTES, 'UTF-8'); ?>
                        </td>
                        <td>
                            <?php echo htmlspecialchars($student['location'], ENT_QUOTES, 'UTF-8'); ?>
                        </td>
                        <td>
                            <div class="d-flex align-items-center">
                                <select name="lecturer_assignment[<?php echo htmlspecialchars($student['student_id'], ENT_QUOTES, 'UTF-8'); ?>]"
                                        class="form-select me-2"
                                        <?php echo $student['lecturer_id'] ? 'disabled' : ''; ?>>
                                    <option value="">Select Lecturer</option>
                                    <?php 
                                    // Show only lecturers assigned to the student's chosen district
                                    $district = $student['district'];
                                    if (isset($lecturersByDistrict[$district])) {
                                        foreach ($lecturersByDistrict[$district] as $lectInfo) {
                                            $lectId   = $lectInfo['lecturer_id'];
                                            $lectName = $lectInfo['name'];
                                            $selected = ($student['lecturer_id'] == $lectId) ? 'selected' : '';
                                            echo '<option value="'.htmlspecialchars($lectId, ENT_QUOTES, 'UTF-8').'" '.$selected.'>'
                                                 .htmlspecialchars($lectName, ENT_QUOTES, 'UTF-8')
                                                 .'</option>';
                                        }
                                    } else {
                                        echo '<option value="" disabled>No lecturers available for this district</option>';
                                    }
                                    ?>
                                </select>
                                <?php if ($student['lecturer_id']): ?>
                                    <button type="button"
                                            class="btn btn-sm btn-outline-danger undo-assignment"
                                            data-student-id="<?php echo htmlspecialchars($student['student_id'], ENT_QUOTES, 'UTF-8'); ?>">
                                        <i class="fas fa-undo"></i>
                                    </button>
                                <?php endif; ?>
                            </div>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>

        <button type="submit" class="btn btn-primary">Assign Lecturers to Students</button>
    </form>
</div>

<!-- Bootstrap JS and Popper.js -->
<script src="https://cdn.jsdelivr.net/npm/@popperjs/core@2.10.2/dist/umd/popper.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.min.js"></script>

<script>
// DYNAMICALLY FETCH DISTRICTS WHEN REGION CHANGES
document.querySelector('select[name="region_filter"]').addEventListener('change', function() {
    const region = this.value;
    if (region) {
        fetch('get_districts.php?region=' + encodeURIComponent(region))
        .then(response => response.json())
        .then(data => {
            const districtSelect = document.querySelector('#district_filter');
            districtSelect.innerHTML = '<option value="">All Districts</option>';
            data.forEach(function(d) {
                districtSelect.innerHTML += '<option value="' + d + '">' + d + '</option>';
            });
        })
        .catch(error => console.error('Error fetching districts:', error));
    } else {
        document.querySelector('#district_filter').innerHTML = '<option value="">All Districts</option>';
    }
    this.form.submit();
});

// PRE-POPULATE THE DISTRICT FILTER (IF REGION WAS ALREADY SELECTED)
window.addEventListener('DOMContentLoaded', function() {
    const regionSelect = document.querySelector('select[name="region_filter"]');
    const region = regionSelect.value;

    if (region) {
        fetch('get_districts.php?region=' + encodeURIComponent(region))
        .then(response => response.json())
        .then(data => {
            const districtSelect = document.querySelector('#district_filter');
            districtSelect.innerHTML = '<option value="">All Districts</option>';
            data.forEach(function(d) {
                districtSelect.innerHTML += '<option value="' + d + '">' + d + '</option>';
            });
            // Reselect if a district filter was used
            const selectedDistrict = '<?php echo $districtFilter; ?>';
            if (selectedDistrict) {
                districtSelect.value = selectedDistrict;
            }
        })
        .catch(error => console.error('Error fetching districts:', error));
    }
});

// UNDO ASSIGNMENT (AJAX)
document.querySelectorAll('.undo-assignment').forEach(btn => {
    btn.addEventListener('click', function() {
        const studentId = this.getAttribute('data-student-id');
        fetch('undo_assignment.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ studentId })
        })
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                alert('Assignment undone successfully.');
                window.location.reload();
            } else {
                alert('Error undoing assignment.');
            }
        })
        .catch(err => console.error('Error:', err));
    });
});
</script>

</body>
</html>
