<?php
/**
 * assign_student.php  |  Allocate or clear students from PI groups
 * ------------------------------------------------------------------
 * • Session gate (admin_id → login.php)
 * • All POST handlers run before any output, so redirects work
 * • Alerts auto-dismiss after 3 s
 */

 /* -------------------- SESSION GUARD ----------------------------- */
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit();
}

 /* -------------------- DEV ERROR DISPLAY ------------------------- */
error_reporting(E_ALL);
ini_set('display_errors', 1);

 /* -------------------- DATABASE ---------------------------------- */
require '../config/database.php';

 /* -------------------- HELPERS & CONSTANTS ----------------------- */
$courseAbbreviations = [
    'Accounting'                              => 'ACCT',
    'Procurement and Supply Chain Management' => 'PSCM',
    'Marketing and Entrepreneurship'          => 'Mkt&E',
    'Business Information Systems'            => 'BIS',
    'Human Resource Management'               => 'HRM',
    'Banking and Finance'                     => 'B&F',
    'Risk and Insurance'                      => 'R&I',
];
function redirectSelf(): void
{
    header('Location: ' . $_SERVER['PHP_SELF']);
    exit();
}

 /* ================================================================ */
 /*  >>> HANDLE ANY POST ACTION BEFORE OUTPUT <<<                    */
 /* ================================================================ */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    /* ---------- Manual assignment ---------- */
    if (isset($_POST['assign'])) {
        $student_id = $_POST['student_id'] ?? '';
        $group_id   = $_POST['group_id']   ?? '';

        if ($student_id && $group_id) {
            try {
                $stmt = $pdo->prepare(
                    "UPDATE student_details
                     SET lecturer_group_id = :group_id
                     WHERE id = :student_id"
                );
                $stmt->execute([
                    'group_id'   => $group_id,
                    'student_id' => $student_id
                ]);
                $_SESSION['success'] = 'Student assigned successfully.';
            } catch (PDOException $e) {
                $_SESSION['error'] = 'Error assigning student: ' . $e->getMessage();
            }
        } else {
            $_SESSION['warning'] = 'Please select both a student and a group.';
        }
        redirectSelf();
    }

    /* ---------- Remove individual ---------- */
    if (isset($_POST['remove_individual'])) {
        $student_id = $_POST['student_id'] ?? '';
        if ($student_id) {
            try {
                $stmt = $pdo->prepare(
                    "UPDATE student_details
                     SET lecturer_group_id = NULL
                     WHERE id = :student_id"
                );
                $stmt->execute(['student_id' => $student_id]);
                $_SESSION['success'] = 'Student removed from group.';
            } catch (PDOException $e) {
                $_SESSION['error'] = 'Error removing student: ' . $e->getMessage();
            }
        }
        redirectSelf();
    }

    /* ---------- Random assignment ---------- */
    if (isset($_POST['random_assign'])) {
        try {
            // Get students eligible for random assignment (unassigned, have lecturer, not Teaching Practice)
            $auto_students = $pdo->query(
                "SELECT s.id, s.lecturer_id, u.group AS student_group
                 FROM student_details s
                 JOIN users u ON s.user_id = u.id
                 WHERE s.lecturer_group_id IS NULL
                   AND s.lecturer_id IS NOT NULL
                   AND u.group <> 'Teaching Practice'"
            )->fetchAll(PDO::FETCH_ASSOC);

            if ($auto_students) {
                $lookup  = $pdo->prepare("SELECT group_id FROM lecturer_login WHERE id = :lid LIMIT 1");
                $group_category_check = $pdo->prepare("SELECT category FROM pi_lecturer_groups WHERE id = :gid LIMIT 1");
                $update  = $pdo->prepare(
                    "UPDATE student_details
                     SET lecturer_group_id = :gid
                     WHERE id = :sid"
                );

                $assigned_count = 0;
                $skipped_count = 0;

                foreach ($auto_students as $stu) {
                    $lookup->execute(['lid' => $stu['lecturer_id']]);
                    $gid = $lookup->fetchColumn();
                    
                    if ($gid) {
                        // Check if the group category matches the student's internship type
                        $group_category_check->execute(['gid' => $gid]);
                        $group_category = $group_category_check->fetchColumn();
                        
                        if ($group_category && $group_category === $stu['student_group']) {
                        $update->execute(['gid' => $gid, 'sid' => $stu['id']]);
                            $assigned_count++;
                        } else {
                            $skipped_count++;
                        }
                    }
                }
                
                if ($assigned_count > 0) {
                    $_SESSION['success'] = "Random assignment completed: {$assigned_count} students assigned.";
                    if ($skipped_count > 0) {
                        $_SESSION['success'] .= " {$skipped_count} students skipped (category mismatch).";
                    }
                } else {
                    $_SESSION['warning'] = "No students could be assigned due to category mismatches.";
                }
            } else {
                $_SESSION['warning'] = 'No students available for random assign.';
            }
        } catch (PDOException $e) {
            $_SESSION['error'] = 'Random assign failed: ' . $e->getMessage();
        }
        redirectSelf();
    }

    /* ---------- Random assignment for Teaching Practice students only ---------- */
    if (isset($_POST['random_assign_teaching'])) {
        try {
            // Get Teaching Practice students eligible for random assignment
            $teaching_students = $pdo->query(
                "SELECT s.id, s.lecturer_id, u.group AS student_group
                 FROM student_details s
                 JOIN users u ON s.user_id = u.id
                 WHERE s.lecturer_group_id IS NULL
                   AND s.lecturer_id IS NOT NULL
                   AND u.group = 'Teaching Practice'"
            )->fetchAll(PDO::FETCH_ASSOC);

            if ($teaching_students) {
                $lookup  = $pdo->prepare("SELECT group_id FROM lecturer_login WHERE id = :lid LIMIT 1");
                $group_category_check = $pdo->prepare("SELECT category FROM pi_lecturer_groups WHERE id = :gid LIMIT 1");
                $update  = $pdo->prepare(
                    "UPDATE student_details
                     SET lecturer_group_id = :gid
                     WHERE id = :sid"
                );

                $assigned_count = 0;
                $skipped_count = 0;

                foreach ($teaching_students as $stu) {
                    $lookup->execute(['lid' => $stu['lecturer_id']]);
                    $gid = $lookup->fetchColumn();
                    
                    if ($gid) {
                        // Check if the group category matches Teaching Practice
                        $group_category_check->execute(['gid' => $gid]);
                        $group_category = $group_category_check->fetchColumn();
                        
                        if ($group_category && $group_category === 'Teaching Practice') {
                            $update->execute(['gid' => $gid, 'sid' => $stu['id']]);
                            $assigned_count++;
                        } else {
                            $skipped_count++;
                        }
                    }
                }
                
                if ($assigned_count > 0) {
                    $_SESSION['success'] = "Teaching Practice assignment completed: {$assigned_count} students assigned.";
                    if ($skipped_count > 0) {
                        $_SESSION['success'] .= " {$skipped_count} students skipped (lecturer not in Teaching Practice group).";
                    }
                } else {
                    $_SESSION['warning'] = "No Teaching Practice students could be assigned. Lecturers may not be in Teaching Practice groups.";
                }
            } else {
                $_SESSION['warning'] = 'No Teaching Practice students available for assignment.';
            }
        } catch (PDOException $e) {
            $_SESSION['error'] = 'Teaching Practice assignment failed: ' . $e->getMessage();
        }
        redirectSelf();
    }

    /* ---------- Random assignment for Industrial Internship students only ---------- */
    if (isset($_POST['random_assign_industrial'])) {
        try {
            // Get Industrial Internship students eligible for random assignment
            $industrial_students = $pdo->query(
                "SELECT s.id, s.lecturer_id, u.group AS student_group
                 FROM student_details s
                 JOIN users u ON s.user_id = u.id
                 WHERE s.lecturer_group_id IS NULL
                   AND s.lecturer_id IS NOT NULL
                   AND u.group = 'Industrial Internship'"
            )->fetchAll(PDO::FETCH_ASSOC);

            if ($industrial_students) {
                $lookup  = $pdo->prepare("SELECT group_id FROM lecturer_login WHERE id = :lid LIMIT 1");
                $group_category_check = $pdo->prepare("SELECT category FROM pi_lecturer_groups WHERE id = :gid LIMIT 1");
                $update  = $pdo->prepare(
                    "UPDATE student_details
                     SET lecturer_group_id = :gid
                     WHERE id = :sid"
                );

                $assigned_count = 0;
                $skipped_count = 0;

                foreach ($industrial_students as $stu) {
                    $lookup->execute(['lid' => $stu['lecturer_id']]);
                    $gid = $lookup->fetchColumn();
                    
                    if ($gid) {
                        // Check if the group category matches Industrial Internship
                        $group_category_check->execute(['gid' => $gid]);
                        $group_category = $group_category_check->fetchColumn();
                        
                        if ($group_category && $group_category === 'Industrial Internship') {
                            $update->execute(['gid' => $gid, 'sid' => $stu['id']]);
                            $assigned_count++;
                        } else {
                            $skipped_count++;
                        }
                    }
                }
                
                if ($assigned_count > 0) {
                    $_SESSION['success'] = "Industrial Internship assignment completed: {$assigned_count} students assigned.";
                    if ($skipped_count > 0) {
                        $_SESSION['success'] .= " {$skipped_count} students skipped (lecturer not in Industrial Internship group).";
                    }
                } else {
                    $_SESSION['warning'] = "No Industrial Internship students could be assigned. Lecturers may not be in Industrial Internship groups.";
                }
            } else {
                $_SESSION['warning'] = 'No Industrial Internship students available for assignment.';
            }
        } catch (PDOException $e) {
            $_SESSION['error'] = 'Industrial Internship assignment failed: ' . $e->getMessage();
        }
        redirectSelf();
    }

    /* ---------- Remove ALL students ---------- */
    if (isset($_POST['remove_all'])) {
        try {
            $pdo->exec("UPDATE student_details SET lecturer_group_id = NULL
                        WHERE lecturer_group_id IS NOT NULL");
            $_SESSION['success'] = 'All students removed from groups.';
        } catch (PDOException $e) {
            $_SESSION['error'] = 'Error clearing groups: ' . $e->getMessage();
        }
        redirectSelf();
    }

    /* ---------- Distribute all unassigned students equally across groups ---------- */
    if (isset($_POST['distribute_all_equally'])) {
        $summary = [];
        foreach ([
            'Industrial Internship' => 'Industrial Internship',
            'Teaching Practice' => 'Teaching Practice'
        ] as $category => $label) {
            // Get all unassigned students in this category
            $students = $pdo->prepare(
                "SELECT s.id FROM student_details s
                 JOIN users u ON s.user_id = u.id
                 WHERE s.lecturer_group_id IS NULL
                   AND u.group = :category"
            );
            $students->execute(['category' => $category]);
            $studentIds = $students->fetchAll(PDO::FETCH_COLUMN);

            // Get all available groups in this category
            $groups = $pdo->prepare(
                "SELECT id FROM pi_lecturer_groups WHERE category = :category"
            );
            $groups->execute(['category' => $category]);
            $groupIds = $groups->fetchAll(PDO::FETCH_COLUMN);

            $assigned = 0;
            if ($studentIds && $groupIds) {
                $update = $pdo->prepare(
                    "UPDATE student_details SET lecturer_group_id = :gid WHERE id = :sid"
                );
                $groupCount = count($groupIds);
                $i = 0;
                foreach ($studentIds as $sid) {
                    $gid = $groupIds[$i % $groupCount];
                    $update->execute(['gid' => $gid, 'sid' => $sid]);
                    $assigned++;
                    $i++;
                }
                $summary[] = "$assigned $label students distributed equally among $groupCount groups.";
            } else {
                $summary[] = "No unassigned $label students or no available $label groups.";
            }
        }
        $_SESSION['success'] = implode(' ', $summary);
        redirectSelf();
    }
}

 /* ================================================================ */
 /*  >>> FETCH DATA FOR DISPLAY <<<                                  */
 /* ================================================================ */

/* Students still unassigned */
$students = $pdo->query(
    "SELECT s.id, s.first_name, s.surname, u.group, s.course
     FROM student_details s
     JOIN users u ON s.user_id = u.id
     WHERE s.lecturer_group_id IS NULL
     ORDER BY u.group DESC, s.first_name ASC"
)->fetchAll(PDO::FETCH_ASSOC);

/* Get selected group from GET */
$selectedGroup = $_GET['group_id'] ?? '';

/* Groups with at least one lecturer */
$groups = $pdo->query(
    "SELECT g.id, g.group_name, g.category
     FROM pi_lecturer_groups g
     JOIN lecturer_login l ON g.id = l.group_id
     GROUP BY g.id
     ORDER BY g.category ASC, g.group_name ASC"
)->fetchAll(PDO::FETCH_ASSOC);

/* Current allocations, filtered by group if selected */
$allocationsQuery = "
    SELECT s.id AS student_id,
            CONCAT(s.first_name, ' ', s.surname) AS student_name,
            s.course,
            g.id AS group_id,
           g.group_name,
           g.category
     FROM student_details s
     JOIN pi_lecturer_groups g ON s.lecturer_group_id = g.id
";
$params = [];
if ($selectedGroup) {
    $allocationsQuery .= " WHERE g.id = :group_id";
    $params['group_id'] = $selectedGroup;
}
$allocationsQuery .= " ORDER BY g.category ASC, g.group_name ASC";
$stmt = $pdo->prepare($allocationsQuery);
$stmt->execute($params);
$allocations = $stmt->fetchAll(PDO::FETCH_ASSOC);

/* ================================================================ */
/*  >>> OUTPUT (HTML) <<<                                            */
/* ================================================================ */
include 'pi_header.php';
?>

<div class="container mt-5">

    <!-- Flash messages -->
    <?php foreach (['success','error','warning'] as $type): ?>
        <?php if (!empty($_SESSION[$type])): ?>
            <div class="alert alert-<?= $type === 'error' ? 'danger' : $type ?> alert-dismissible fade show">
                <?= htmlspecialchars($_SESSION[$type]) ?>
            </div>
            <?php unset($_SESSION[$type]); ?>
        <?php endif; ?>
    <?php endforeach; ?>

    <!-- Manual assignment card -->
    <div class="row">
        <div class="col-md-8 mx-auto">
            <div class="card shadow-sm p-3">
                <h4 class="text-center mb-3">Manually Assign Student to a Group</h4>

                <form method="post">
                    <div class="mb-3">
                        <label class="form-label">Select Student</label>
                        <select name="student_id" class="form-control form-control-sm" required>
                            <option value="">-- Select Student --</option>
                            <?php foreach ($students as $s): ?>
                                <?php
                                    $name = htmlspecialchars($s['first_name'] . ' ' . $s['surname']);
                                    $abbr = $courseAbbreviations[$s['course']] ?? $s['course'];
                                ?>
                                <option value="<?= $s['id'] ?>"><?= "$name ($abbr)" ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Select Group</label>
                        <select name="group_id" class="form-control form-control-sm" required>
                            <option value="">-- Select Group --</option>
                            <?php foreach ($groups as $g): ?>
                                <option value="<?= $g['id'] ?>">
                                    <?= htmlspecialchars($g['group_name']) ?> (<?= htmlspecialchars($g['category']) ?>)
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <button type="submit" name="assign" class="btn btn-primary btn-sm w-100">
                        Manually Assign Student
                    </button>
                </form>
            </div>
        </div>
    </div>

    <!-- Action buttons -->
    <div class="text-center mt-4">
        <div class="row">
            <div class="col-md-3">
                <form method="post" class="d-inline">
                    <button type="submit" name="random_assign_teaching" class="btn btn-primary w-100">
                        <i class="fas fa-chalkboard-teacher"></i> Assign Teaching Practice
                    </button>
                </form>
            </div>
            <div class="col-md-3">
        <form method="post" class="d-inline">
                    <button type="submit" name="random_assign_industrial" class="btn btn-secondary w-100">
                        <i class="fas fa-industry"></i> Assign Industrial Internship
            </button>
        </form>
            </div>
            <div class="col-md-3">
        <form method="post" class="d-inline">
                    <button type="submit" name="remove_all" class="btn btn-danger w-100"
                    onclick="return confirm('Are you sure you want to remove all students?');">
                        <i class="fas fa-trash"></i> Remove All Students
                    </button>
                </form>
            </div>
            <div class="col-md-3">
                <form method="post" class="d-inline">
                    <button type="submit" name="distribute_all_equally" class="btn btn-info w-100">
                        <i class="fas fa-share-alt"></i> Distribute Equally
            </button>
        </form>
            </div>
        </div>
    </div>

    <!-- Group filter -->
    <form method="get" class="mb-3">
        <label for="groupFilter">Filter by Group:</label>
        <select id="groupFilter" name="group_id" class="form-control form-control-sm w-25 d-inline" onchange="this.form.submit()">
            <option value="">-- Show All Groups --</option>
            <?php foreach ($groups as $g): ?>
                <option value="<?= $g['id'] ?>" <?= ($selectedGroup == $g['id']) ? 'selected' : '' ?>>
                    <?= htmlspecialchars($g['group_name']) ?> (<?= htmlspecialchars($g['category']) ?>)
                </option>
            <?php endforeach; ?>
        </select>
    </form>

    <!-- Selected group heading -->
    <h4 class="text-center mt-4" id="selectedGroupName">All Groups</h4>

    <!-- Current Allocations Table -->
    <div class="row mt-4">
        <div class="col-md-12">
            <div class="card shadow-sm">
                <div class="card-header">
                    <h5 class="mb-0">Current Student Allocations</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-striped table-sm">
        <thead>
            <tr>
                                    <th>Student Name</th>
                                    <th>Course</th>
                                    <th>Group</th>
                                    <th>Category</th>
                <th>Action</th>
            </tr>
        </thead>
        <tbody>
                                <?php if ($allocations): ?>
                                    <?php foreach ($allocations as $allocation): ?>
                                        <tr>
                                            <td><?= htmlspecialchars($allocation['student_name']) ?></td>
                                            <td><?= htmlspecialchars($allocation['course']) ?></td>
                                            <td><?= htmlspecialchars($allocation['group_name']) ?></td>
                                            <td>
                                                <span class="badge <?= $allocation['category'] === 'Teaching Practice' ? 'bg-primary' : 'bg-success' ?>">
                                                    <?= htmlspecialchars($allocation['category']) ?>
                                                </span>
                    </td>
                    <td>
                        <form method="post" style="display:inline;">
                                                    <input type="hidden" name="student_id" value="<?= $allocation['student_id'] ?>">
                                                    <button type="submit" name="remove_individual" class="btn btn-danger btn-sm"
                                                            onclick="return confirm('Remove this student from group?');">
                                Remove
                            </button>
                        </form>
                    </td>
                </tr>
            <?php endforeach; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="5" class="text-center">No students currently allocated.</td>
                                    </tr>
            <?php endif; ?>
        </tbody>
    </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
    /* Auto-hide alerts after 3 s */
    setTimeout(() => {
        document.querySelectorAll('.alert').forEach(el => {
            el.classList.remove('show');          // Bootstrap fade
            el.addEventListener('transitionend', () => el.remove());
        });
    }, 3000);

    /* Client-side group filter */
    function filterGroup() {
        const sel = document.getElementById('groupFilter').value;
        const rows = document.querySelectorAll('#studentTable tbody tr');
        const heading = document.getElementById('selectedGroupName');

        heading.textContent = sel === ''
            ? 'All Groups'
            : document.querySelector(`#groupFilter option[value="${sel}"]`).textContent;

        rows.forEach(r => r.style.display =
            sel === '' || r.dataset.group === sel ? '' : 'none'
        );
    }
</script>

<?php include 'pi_footer.php'; ?>
