<?php
/**
 * assign_lecturer.php  |  Assign or remove lecturers from PI groups
 * ---------------------------------------------------------------
 * Session guard – identical to your other protected pages.
 */
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

if (!isset($_SESSION['admin_id'])) {       // same session key and redirect as elsewhere
    header('Location: login');
    exit();
}

/* ------------------------------------------------------------------ */
/*  Development-time error display (remove or lower level in prod)     */
/* ------------------------------------------------------------------ */
error_reporting(E_ALL);
ini_set('display_errors', 1);

/* ------------------------------------------------------------------ */
/*  Database connection                                                */
/* ------------------------------------------------------------------ */
require '../config/database.php';

/* ------------------------------------------------------------------ */
/*  Fetch groups (for the select box)                                  */
/* ------------------------------------------------------------------ */
try {
    $groups = $pdo->query(
        "SELECT * FROM pi_lecturer_groups ORDER BY group_name ASC"
    )->fetchAll();
} catch (PDOException $e) {
    die("Error fetching groups: " . $e->getMessage());
}

/* ------------------------------------------------------------------ */
/*  Fetch lecturers without a group                                    */
/* ------------------------------------------------------------------ */
try {
    $lecturers = $pdo->query(
        "SELECT id, name FROM lecturer_login WHERE group_id IS NULL ORDER BY name ASC"
    )->fetchAll();
} catch (PDOException $e) {
    die("Error fetching lecturers: " . $e->getMessage());
}

/* ------------------------------------------------------------------ */
/*  Fetch current allocations                                          */
/* ------------------------------------------------------------------ */
try {
    $allocations = $pdo->query(
        "SELECT l.id   AS lecturer_id,
                l.name AS lecturer_name,
                g.id   AS group_id,
                g.group_name
         FROM lecturer_login l
         JOIN pi_lecturer_groups g ON l.group_id = g.id
         ORDER BY g.group_name ASC"
    )->fetchAll();
} catch (PDOException $e) {
    die("Error fetching allocations: " . $e->getMessage());
}

/* ------------------------------------------------------------------ */
/*  Assign lecturer to a group                                         */
/* ------------------------------------------------------------------ */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['assign'])) {
    $lecturer_id = $_POST['lecturer_id'] ?? '';
    $group_id    = $_POST['group_id']    ?? '';

    if ($lecturer_id && $group_id) {
        try {
            $stmt = $pdo->prepare(
                "UPDATE lecturer_login SET group_id = :group_id WHERE id = :lecturer_id"
            );
            $stmt->execute([
                'group_id'    => $group_id,
                'lecturer_id' => $lecturer_id
            ]);

            $_SESSION['success'] = "Lecturer assigned successfully!";
        } catch (PDOException $e) {
            $_SESSION['error'] = "Error: " . $e->getMessage();
        }
    } else {
        $_SESSION['warning'] = "Please select both a lecturer and a group.";
    }
    header("Location: assign_lecturer.php");
    exit();
}

/* ------------------------------------------------------------------ */
/*  Remove lecturer from a group                                       */
/* ------------------------------------------------------------------ */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete'])) {
    $lecturer_id = $_POST['lecturer_id'] ?? '';

    if ($lecturer_id) {
        try {
            $stmt = $pdo->prepare(
                "UPDATE lecturer_login SET group_id = NULL WHERE id = :lecturer_id"
            );
            $stmt->execute(['lecturer_id' => $lecturer_id]);

            $_SESSION['success'] = "Lecturer removed from group!";
        } catch (PDOException $e) {
            $_SESSION['error'] = "Error: " . $e->getMessage();
        }
    }
    header("Location: assign_lecturer.php");
    exit();
}

include 'pi_header.php';  // common header
?>

<!-- ----------------------------------------------------------------- -->
<!--  Page Content                                                     -->
<!-- ----------------------------------------------------------------- -->
<div class="container mt-4">
    <?php
    foreach (['success', 'error', 'warning'] as $type) {
        if (isset($_SESSION[$type])) {
            echo "<div class='alert alert-{$type}'>" .
                 htmlspecialchars($_SESSION[$type]) .
                 "</div>";
            unset($_SESSION[$type]);
        }
    }
    ?>

    <div class="d-flex justify-content-center">
        <div class="col-md-6">
            <div class="card shadow-sm p-3">
                <h4 class="text-center mb-3">Assign Lecturer to a Group</h4>
                <form method="post">
                    <div class="mb-3">
                        <label class="form-label">Select Lecturer</label>
                        <select name="lecturer_id" class="form-control form-control-sm" required>
                            <option value="">-- Select Lecturer --</option>
                            <?php foreach ($lecturers as $lecturer): ?>
                                <option value="<?= $lecturer['id'] ?>">
                                    <?= htmlspecialchars($lecturer['name']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Select Group</label>
                        <select name="group_id" class="form-control form-control-sm" required>
                            <option value="">-- Select Group --</option>
                            <?php foreach ($groups as $group): ?>
                                <option value="<?= $group['id'] ?>">
                                    <?= htmlspecialchars($group['group_name']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <button type="submit" name="assign" class="btn btn-primary btn-sm w-100">
                        Assign Lecturer
                    </button>
                </form>
            </div>
        </div>
    </div>

    <div class="col-md-12 mt-4">
        <h4>Current Lecturer Allocations</h4>
        <table class="table table-striped table-sm">
            <thead>
                <tr>
                    <th>Group Name</th>
                    <th>Lecturer</th>
                    <th>Action</th>
                </tr>
            </thead>
            <tbody>
                <?php if ($allocations): ?>
                    <?php foreach ($allocations as $allocation): ?>
                        <tr>
                            <td><?= htmlspecialchars($allocation['group_name']) ?></td>
                            <td><?= htmlspecialchars($allocation['lecturer_name']) ?></td>
                            <td>
                                <form method="post" style="display:inline;">
                                    <input type="hidden" name="lecturer_id"
                                           value="<?= $allocation['lecturer_id'] ?>">
                                    <button type="submit" name="delete"
                                            class="btn btn-danger btn-sm"
                                            onclick="return confirm('Remove this lecturer?');">
                                            Remove
                                    </button>
                                </form>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php else: ?>
                    <tr><td colspan="3" class="text-center">No lecturers assigned yet.</td></tr>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<?php include 'pi_footer.php'; ?>  <!-- common footer -->
