<?php
// Start output buffering (if desired)
ob_start();

// Start session if not already started
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Check if lecturer is logged in
if (!isset($_SESSION['lecturer_id'])) {
    header('Location: login.php');
    exit;
}

// Include database connection once
include '../config/database.php';

// ----------------------------------------------------------------------------
// THIS SECTION MERGES CODE FROM lect_header.php
// (fetch lecturer data, build navbar, show success/error alerts, etc.)
// ----------------------------------------------------------------------------

// Fetch lecturer info for navbar display
$lecturer_id = $_SESSION['lecturer_id'] ?? '';
$lecturer_first_name = '';

if ($lecturer_id) {
    $stmtLect = $pdo->prepare('SELECT name FROM lecturer_login WHERE id = ?');
    $stmtLect->execute([$lecturer_id]);
    $lectRow = $stmtLect->fetch();
    if ($lectRow) {
        // First name only for the navbar
        $lecturer_first_name = htmlspecialchars(explode(' ', $lectRow['name'])[0], ENT_QUOTES, 'UTF-8');
    }
}

// ----------------------------------------------------------------------------
// BEGIN HTML
// ----------------------------------------------------------------------------
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <!-- Mobile responsive meta tag -->
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Enter Results</title>
    
    <!-- Bootstrap CSS (you can switch to 5.3 if you like) -->
    <link
      href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css"
      rel="stylesheet"
    >
    
    <!-- Font Awesome (if needed for icons) -->
    <link
      rel="stylesheet"
      href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css"
    >
    
    <!-- Your custom CSS (optional) -->
    <link rel="icon" type="image/png" href="../public/img/sob_logo.png">
    <link href="style.css" rel="stylesheet">
    
    <style>
      /* Example styles merged from lect_header.php or your custom styling */
      .modal-header {
          background-color: #002047;
          color: white;
      }
      .btn-custom-send {
          background-color: #002047;
          color: white;
      }
      .btn-custom-send:hover {
          background-color: #001533;
          color: white;
      }

      @media (max-width: 576px) {
          body {
              font-size: 0.9rem;
          }
          h2 {
              font-size: 1.5rem;
          }
          .btn {
              font-size: 0.9rem;
              padding: 0.5rem 0.8rem;
          }
          .container {
              padding: 0 10px;
          }
      }
    </style>
</head>
<body>

<!-- NAVBAR (merged from lect_header.php) -->
<nav class="navbar navbar-expand-lg navbar-dark" style="background-color: #002047;">
  <div class="container-fluid">
    <a class="navbar-brand fw-bold" href="#">
      <img src="../public/img/sob_logo.png" alt="Logo" width="30" height="30" class="d-inline-block align-top">
      Lecturer Portal
    </a>
    <button class="navbar-toggler" type="button" data-bs-toggle="collapse"
            data-bs-target="#navbarNav" aria-controls="navbarNav"
            aria-expanded="false" aria-label="Toggle navigation">
      <span class="navbar-toggler-icon"></span>
    </button>

    <div class="collapse navbar-collapse" id="navbarNav">
      <?php if (!empty($lecturer_id)): ?>
      <ul class="navbar-nav ms-auto">
        <li class="nav-item"><a class="nav-link" href="index">Students</a></li>
        <li class="nav-item"><a class="nav-link" href="region">Region</a></li>
        <li class="nav-item"><a class="nav-link" href="mapping">Mapping</a></li>
        <li class="nav-item"><a class="nav-link" href="results">Results</a></li>
        <li class="nav-item">
          <a class="nav-link" href="#" data-bs-toggle="modal"
             data-bs-target="#messageModal">Message</a>
        </li>
        <li class="nav-item dropdown">
          <a class="nav-link dropdown-toggle" href="#" id="navbarDropdown"
             role="button" data-bs-toggle="dropdown"
             aria-expanded="false">
            <?php echo $lecturer_first_name; ?>
          </a>
          <ul class="dropdown-menu dropdown-menu-end" aria-labelledby="navbarDropdown">
            <li><a class="dropdown-item" href="logout.php">Logout</a></li>
          </ul>
        </li>
      </ul>
      <?php endif; ?>
    </div>
  </div>
</nav>

<!-- SUCCESS/ERROR ALERTS (if any) -->
<div class="container mt-4">
  <?php if (isset($_SESSION['success'])): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
      <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
      <button type="button" class="btn-close" data-bs-dismiss="alert"
              aria-label="Close"></button>
    </div>
  <?php endif; ?>
  
  <?php if (isset($_SESSION['error'])): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
      <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
      <button type="button" class="btn-close" data-bs-dismiss="alert"
              aria-label="Close"></button>
    </div>
  <?php endif; ?>
</div>

<!-- MAIN CONTENT: "Enter Results" (from results.php) -->
<div class="container mt-3">
<?php
// -------------------------------------------------------------------
// The existing results.php logic (fetch assigned students, etc.)
// -------------------------------------------------------------------
$lecturerId = $_SESSION['lecturer_id'];

// Fetch assigned students' details and their saved results
$stmt = $pdo->prepare('
    SELECT sd.id AS student_id, sd.first_name, sd.surname, u.index_number,
           sr.marks, sr.grade, sr.gp, sr.description, sr.status
    FROM student_details sd
    JOIN users u ON sd.user_id = u.id
    LEFT JOIN student_results sr
           ON sd.id = sr.student_id
          AND sr.lecturer_id = ?
    WHERE sd.lecturer_id = ?
');
$stmt->execute([$lecturerId, $lecturerId]);
$students = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Check if results have been submitted
$resultsSubmitted = false;
foreach ($students as $student) {
    if ($student['status'] === 'submitted') {
        $resultsSubmitted = true;
        break;
    }
}
?>

    <h2>Enter Results</h2>
    <?php if (empty($students)): ?>
        <div class="alert alert-warning">No students assigned to you yet.</div>
    <?php elseif ($resultsSubmitted): ?>
        <div class="alert alert-info">
            Results have been submitted and cannot be edited.
        </div>
        <div class="table-responsive">
            <table class="table table-striped">
                <thead>
                    <tr>
                        <th>Student Name</th>
                        <th>Index Number</th>
                        <th>Marks</th>
                        <th class="d-none d-sm-table-cell">Grade</th>
                        <th class="d-none d-sm-table-cell">GP</th>
                        <th class="d-none d-sm-table-cell">Description</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($students as $stud): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($stud['first_name'].' '.$stud['surname']); ?></td>
                            <td><?php echo htmlspecialchars($stud['index_number']); ?></td>
                            <td><?php echo htmlspecialchars($stud['marks']); ?></td>
                            <td class="d-none d-sm-table-cell"><?php echo htmlspecialchars($stud['grade']); ?></td>
                            <td class="d-none d-sm-table-cell"><?php echo htmlspecialchars($stud['gp']); ?></td>
                            <td class="d-none d-sm-table-cell"><?php echo htmlspecialchars($stud['description']); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php else: ?>
        <!-- Show form to enter or edit results -->
        <form method="POST" action="save_submit_results.php">
            <div class="table-responsive">
                <table class="table table-striped">
                    <thead>
                        <tr>
                            <th>Student Name</th>
                            <th>Index Number</th>
                            <th>Marks</th>
                            <th class="d-none d-sm-table-cell">Grade</th>
                            <th class="d-none d-sm-table-cell">GP</th>
                            <th class="d-none d-sm-table-cell">Description</th>
                        </tr>
                    </thead>
                    <tbody>
                    <?php foreach ($students as $index => $stud): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($stud['first_name'].' '.$stud['surname']); ?></td>
                            <td><?php echo htmlspecialchars($stud['index_number']); ?></td>
                            <td>
                                <input
                                  type="number" name="marks[]" min="0" max="100"
                                  class="form-control"
                                  value="<?php echo htmlspecialchars($stud['marks']); ?>"
                                  required
                                  oninput="calculateGrade(this.value, <?php echo $index; ?>)"
                                >
                            </td>
                            <td
                              id="grade-<?php echo $index; ?>"
                              class="d-none d-sm-table-cell"
                            >
                              <?php echo htmlspecialchars($stud['grade']); ?>
                            </td>
                            <td
                              id="gp-<?php echo $index; ?>"
                              class="d-none d-sm-table-cell"
                            >
                              <?php echo htmlspecialchars($stud['gp']); ?>
                            </td>
                            <td
                              id="description-<?php echo $index; ?>"
                              class="d-none d-sm-table-cell"
                            >
                              <?php echo htmlspecialchars($stud['description']); ?>
                            </td>

                            <!-- Hidden fields to pass data to save_submit_results.php -->
                            <input type="hidden" name="student_ids[]" value="<?php echo htmlspecialchars($stud['student_id']); ?>">
                            <input type="hidden" name="index_numbers[]" value="<?php echo htmlspecialchars($stud['index_number']); ?>">
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            <button
              type="submit" name="action" value="save"
              class="btn btn-secondary"
            >
              Save Results
            </button>
            <button
              type="submit" name="action" value="submit"
              class="btn btn-primary"
            >
              Submit Results
            </button>
        </form>
    <?php endif; ?>
</div><!-- /.container for the main content -->

<!-- FOOTER (merged from lect_footer.php) -->
<footer class="footer mt-5 py-3" style="background: #002047;">
  <div class="container text-center">
    <span class="text-white">© 2024 Lecturer Portal. All rights reserved.</span>
  </div>
</footer>

<!-- SEND MESSAGE MODAL (if needed from lect_header.php) -->
<div class="modal fade" id="messageModal" tabindex="-1"
     aria-labelledby="messageModalLabel" aria-hidden="true">
  <div class="modal-dialog">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="messageModalLabel">Send Message</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"
                aria-label="Close"></button>
      </div>
      <form method="POST" action="send_message.php">
        <div class="modal-body">
          <div class="mb-3">
            <label for="recipients" class="form-label">Recipients</label>
            <select multiple class="form-control" id="recipients" name="recipients[]">
              <?php
              // Populate student list for message recipients (if needed)
              $stmtMsg = $pdo->prepare('
                SELECT user_id, CONCAT(first_name, " ", surname) AS name 
                FROM student_details 
                WHERE lecturer_id = ?
              ');
              $stmtMsg->execute([$lecturer_id]);
              $studentsList = $stmtMsg->fetchAll();
              foreach ($studentsList as $s) {
                  echo '<option value="'.htmlspecialchars($s['user_id']).'">'
                       .htmlspecialchars($s['name'])
                       .'</option>';
              }
              ?>
            </select>
          </div>
          <div class="form-check mb-3">
            <input type="checkbox" class="form-check-input"
                   id="sendToAll" name="send_to_all" value="1">
            <label class="form-check-label" for="sendToAll">Send to all students</label>
          </div>
          <div class="mb-3">
            <label for="subject" class="form-label">Subject</label>
            <input type="text" class="form-control" id="subject" name="subject" required>
          </div>
          <div class="mb-3">
            <label for="message" class="form-label">Message</label>
            <textarea class="form-control" id="message" name="message" rows="3" required></textarea>
          </div>
        </div>
        <div class="modal-footer">
          <button
            type="button" class="btn btn-danger"
            data-bs-dismiss="modal"
          >
            Close
          </button>
          <button type="submit" class="btn btn-custom-send">
            Send
          </button>
        </div>
      </form>
    </div>
  </div>
</div>

<!-- SCRIPTS (merged from lect_footer.php, etc.) -->
<script
  src="https://cdn.jsdelivr.net/npm/@popperjs/core@2.10.2/dist/umd/popper.min.js">
</script>
<script
  src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.min.js">
</script>

<script>
  // "Calculate Grade" logic from results.php
  function calculateGrade(mark, index) {
    let grade, gp, description;
    mark = parseFloat(mark);

    if (mark >= 80 && mark <= 100) {
      grade = 'A';   gp = 4.0; description = 'Excellent';
    } else if (mark >= 75 && mark <= 79) {
      grade = 'B+';  gp = 3.5; description = 'Very Good';
    } else if (mark >= 70 && mark <= 74) {
      grade = 'B';   gp = 3.0; description = 'Good';
    } else if (mark >= 65 && mark <= 69) {
      grade = 'C+';  gp = 2.5; description = 'Average';
    } else if (mark >= 60 && mark <= 64) {
      grade = 'C';   gp = 2.0; description = 'Fair';
    } else if (mark >= 55 && mark <= 59) {
      grade = 'D+';  gp = 1.5; description = 'Barely Satisfactory';
    } else if (mark >= 50 && mark <= 54) {
      grade = 'D';   gp = 1.0; description = 'Weak Pass';
    } else if (mark >= 0 && mark <= 49) {
      grade = 'E';   gp = 0.0; description = 'Fail';
    } else {
      grade = ''; gp = ''; description = '';
    }

    document.getElementById('grade-' + index).innerText = grade;
    document.getElementById('gp-' + index).innerText = gp;
    document.getElementById('description-' + index).innerText = description;
  }

  // Handle "Send to all" checkbox for the message modal
  document.addEventListener('DOMContentLoaded', function() {
    let sendToAll = document.getElementById('sendToAll');
    let recipients = document.getElementById('recipients');
    if (sendToAll && recipients) {
      sendToAll.addEventListener('change', function() {
        for (let i = 0; i < recipients.options.length; i++) {
          recipients.options[i].selected = this.checked;
        }
      });
    }
  });
</script>

</body>
</html>
<?php
ob_end_flush();
?>
