<?php
session_start();
if (!isset($_SESSION['lecturer_id'])) {
    header('Location: login');
    exit;
}

include '../includes/lecturer_header.php';
include '../config/database.php';

$statusMessage = '';
$lecturerLoginId = htmlspecialchars($_SESSION['lecturer_id'], ENT_QUOTES, 'UTF-8');
$preferenceId = htmlspecialchars($_GET['id'], ENT_QUOTES, 'UTF-8');

// Fetch current preference
$stmt = $pdo->prepare('SELECT preferred_region, preferred_districts FROM lecturer_preferences WHERE id = ? AND lecturer_id = ?');
$stmt->execute([$preferenceId, $lecturerLoginId]);
$preference = $stmt->fetch(PDO::FETCH_ASSOC);

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (!$preference) {
        $statusMessage = 'Invalid preference ID.';
    } else {
        $preferredRegion = htmlspecialchars($_POST['preferredRegion'], ENT_QUOTES, 'UTF-8');
        $preferredDistricts = implode(',', array_map(function ($district) {
            return htmlspecialchars($district, ENT_QUOTES, 'UTF-8');
        }, $_POST['preferredDistricts']));

        // Update the preference
        try {
            $stmt = $pdo->prepare('UPDATE lecturer_preferences SET preferred_region = ?, preferred_districts = ? WHERE id = ? AND lecturer_id = ?');
            $stmt->execute([$preferredRegion, $preferredDistricts, $preferenceId, $lecturerLoginId]);
            $statusMessage = 'Preference updated successfully.';
            header('Location: region');
            exit;
        } catch (PDOException $e) {
            $statusMessage = 'Error updating preference: ' . htmlspecialchars($e->getMessage(), ENT_QUOTES, 'UTF-8');
        }
    }
}

// Fetch regions and districts for form
$stmt = $pdo->prepare('SELECT DISTINCT region FROM districts');
$stmt->execute();
$regions = $stmt->fetchAll(PDO::FETCH_ASSOC);

$stmt = $pdo->prepare('SELECT district FROM districts WHERE region = ?');
$stmt->execute([htmlspecialchars($preference['preferred_region'], ENT_QUOTES, 'UTF-8')]);
$districts = $stmt->fetchAll(PDO::FETCH_ASSOC);

$currentDistricts = array_map(function ($district) {
    return htmlspecialchars($district, ENT_QUOTES, 'UTF-8');
}, explode(',', $preference['preferred_districts']));
?>

<div class="container mt-4">
    <h2>Edit Preferred Region and Districts</h2>

    <?php if ($statusMessage): ?>
        <div class="alert alert-info" id="statusMessage">
            <?php echo htmlspecialchars($statusMessage, ENT_QUOTES, 'UTF-8'); ?>
        </div>
    <?php endif; ?>

    <form method="POST" action="edit_preference?id=<?php echo htmlspecialchars($preferenceId, ENT_QUOTES, 'UTF-8'); ?>">
        <div class="mb-3">
            <label for="preferredRegion" class="form-label">Preferred Region</label>
            <select class="form-control" id="preferredRegion" name="preferredRegion" required>
                <option value="">Select Region</option>
                <?php foreach ($regions as $region): ?>
                    <option value="<?php echo htmlspecialchars($region['region'], ENT_QUOTES, 'UTF-8'); ?>" <?php echo htmlspecialchars($preference['preferred_region'], ENT_QUOTES, 'UTF-8') === $region['region'] ? 'selected' : ''; ?>><?php echo htmlspecialchars($region['region'], ENT_QUOTES, 'UTF-8'); ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="mb-3">
            <label for="preferredDistricts" class="form-label">Preferred Districts</label>
            <select class="form-control" id="preferredDistricts" name="preferredDistricts[]" multiple required>
                <?php foreach ($districts as $district): ?>
                    <option value="<?php echo htmlspecialchars($district['district'], ENT_QUOTES, 'UTF-8'); ?>" <?php echo in_array($district['district'], $currentDistricts) ? 'selected' : ''; ?>><?php echo htmlspecialchars($district['district'], ENT_QUOTES, 'UTF-8'); ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        <button type="submit" class="btn btn-primary">Update Preferences</button>
    </form>
</div>

<script>
    document.addEventListener('DOMContentLoaded', function () {
        // Fetch districts based on the selected region
        document.getElementById('preferredRegion').addEventListener('change', function () {
            const region = this.value;
            const districtSelect = document.getElementById('preferredDistricts');
            districtSelect.innerHTML = '';
            if (region) {
                fetch('../../public/fetch_districts.php?region=' + encodeURIComponent(region))
                    .then(response => response.json())
                    .then(data => {
                        data.forEach(district => {
                            const option = document.createElement('option');
                            option.value = district.district;
                            option.textContent = district.district;
                            districtSelect.appendChild(option);
                        });
                    });
            }
        });

        // Trigger change event to load districts if a region is already selected
        const regionSelect = document.getElementById('preferredRegion');
        if (regionSelect.value) {
            regionSelect.dispatchEvent(new Event('change'));
        }
    });
</script>

<?php include '../../includes/footer.php'; ?>
