<?php
session_start();
require_once '../config/database.php';

if (!isset($_SESSION['sys_admin_id'])) {
    header('Location: login.php');
    exit;
}

$stmt = $pdo->prepare('SELECT * FROM sys_admin WHERE id = :id');
$stmt->bindParam(':id', $_SESSION['sys_admin_id']);
$stmt->execute();
$admin = $stmt->fetch();

if (!$admin) {
    header('Location: login.php');
    exit;
}

// Pagination setup
$records_per_page = 10;
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$offset = ($page - 1) * $records_per_page;

// Fetch audit logs with pagination
$stmt = $pdo->prepare("SELECT * FROM audit_log ORDER BY timestamp DESC LIMIT :offset, :records_per_page");
$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmt->bindValue(':records_per_page', $records_per_page, PDO::PARAM_INT);
$stmt->execute();
$logs = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get total records count
$total_stmt = $pdo->query("SELECT COUNT(*) FROM audit_log");
$total_records = $total_stmt->fetchColumn();
$total_pages = ceil($total_records / $records_per_page);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Audit Log Viewer</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body {
            background-color: #f4f4f4;
            font-family: Arial, sans-serif;
            padding: 20px;
        }
        .table-container {
            background-color: #ffffff;
            border-radius: 6px;
            padding: 20px;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
            margin-top: 20px;
            max-width: 1200px;
            margin: auto;
        }
        h2 {
            text-align: center;
            margin-bottom: 20px;
            font-weight: 600;
            color: #444;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin: 20px 0;
            font-size: 15px;
            color: #333;
        }
        thead th {
            background-color: #f8f9fa;
            font-weight: 600;
            padding: 12px;
            text-align: center;
            border-bottom: 2px solid #e9ecef;
        }
        tbody tr {
            border-bottom: 1px solid #e9ecef;
        }
        tbody tr:nth-child(odd) {
            background-color: #f8f9fa;
        }
        tbody tr:hover {
            background-color: #f1f3f5;
        }
        td {
            padding: 10px;
            text-align: left;
        }
        .center-text {
            text-align: center;
        }
        /* Highlight Tampered Rows */
        .tampered {
            background-color: #ffe6e6;
        }
        .integrity-check {
            font-weight: bold;
            padding: 6px;
            border-radius: 4px;
        }
        .tampered .integrity-check {
            color: #c0392b;
        }
        .intact .integrity-check {
            color: #27ae60;
        }
        /* Pagination Styling */
        .pagination {
            justify-content: center;
            margin-top: 20px;
        }
    </style>
</head>
<body>

<div class="container table-container">
    <h2>Audit Log Viewer</h2>

    <table>
        <thead>
            <tr>
                <th class="center-text">Username</th>
                <th class="center-text">Action Description</th>
                <th class="center-text">Affected Table</th>
                <th class="center-text">Affected Record ID</th>
                <th class="center-text">Timestamp</th>
                <th class="center-text">IP Address</th>
                <th class="center-text">User Agent</th>
                <th class="center-text">Integrity Check</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($logs as $log): ?>
                <tr class="<?= $log['is_tampered'] ? 'tampered' : 'intact' ?>">
                    <td class="center-text"><?= htmlspecialchars($log['username']) ?></td>
                    <td><?= htmlspecialchars($log['action']) ?></td>
                    <td class="center-text"><?= htmlspecialchars($log['affected_table']) ?></td>
                    <td class="center-text"><?= htmlspecialchars($log['affected_id']) ?></td>
                    <td><?= htmlspecialchars($log['timestamp']) ?></td>
                    <td><?= htmlspecialchars($log['ip_address']) ?></td>
                    <td><?= htmlspecialchars($log['user_agent']) ?></td>
                    <td class="center-text integrity-check"><?= $log['is_tampered'] ? 'Tampered' : 'Intact' ?></td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>

    <!-- Pagination -->
    <nav>
        <ul class="pagination">
            <?php
            $visible_pages = 5;
            $start_page = max(1, $page - floor($visible_pages / 2));
            $end_page = min($total_pages, $start_page + $visible_pages - 1);

            // Adjust if near the end
            if ($end_page - $start_page + 1 < $visible_pages) {
                $start_page = max(1, $end_page - $visible_pages + 1);
            }

            // "First" and "Previous" links
            if ($page > 1) {
                echo '<li class="page-item"><a class="page-link" href="?page=1">First</a></li>';
                echo '<li class="page-item"><a class="page-link" href="?page=' . ($page - 1) . '">Previous</a></li>';
            }

            // Page number links
            for ($i = $start_page; $i <= $end_page; $i++) {
                $active_class = ($i == $page) ? 'active' : '';
                echo '<li class="page-item ' . $active_class . '"><a class="page-link" href="?page=' . $i . '">' . $i . '</a></li>';
            }

            // "Next" and "Last" links
            if ($page < $total_pages) {
                echo '<li class="page-item"><a class="page-link" href="?page=' . ($page + 1) . '">Next</a></li>';
                echo '<li class="page-item"><a class="page-link" href="?page=' . $total_pages . '">Last</a></li>';
            }
            ?>
        </ul>
    </nav>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
