<?php
// Include the database connection file
include '../config/database.php'; // Ensure the path to the database connection file is correct

// Handle resubmit when the Resubmit button is clicked
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['resubmit'])) {
    $user_id = $_POST['user_id'];
    
    // Update the status to 'saved' for the user to make corrections
    $update_stmt = $pdo->prepare("UPDATE student_details SET status = 'saved' WHERE user_id = :user_id");
    $update_stmt->execute(['user_id' => $user_id]);

    // Redirect to the same page after the status update
    header("Location: gps_check.php?status=saved");
    exit;
}

// Fetch students with 'submitted' status from the database
$sql = "SELECT * FROM student_details WHERE status = 'submitted'";
$stmt = $pdo->query($sql);

// Initialize an array to store users with errors
$error_users = [];

// GPS Validation functions (matches first code logic)
function isValidGPS($gps_location) {
    // Regular expressions for coordinates and Google Maps URL
    $coordinateRegex = '/^-?\d{1,3}\.\d+,-?\d{1,3}\.\d+$/'; 
    $googleMapsUrlRegex = '/^https:\/\/(maps\.app\.goo\.gl\/[A-Za-z0-9]+|www\.google\.com\/maps\/.*)$/';
    $ghanaPostRegex = '/^[A-Za-z0-9+-]{5,}(\s*,?\s*[A-Za-z\s]+)?$/'; // Ghana Post GPS
    
    return preg_match($coordinateRegex, $gps_location) ||
           preg_match($googleMapsUrlRegex, $gps_location) ||
           preg_match($ghanaPostRegex, $gps_location);
}

// Loop through each user to check for errors
while ($row = $stmt->fetch()) {
    $user_id = $row['user_id'];
    $gps_location = $row['gps_location'];
    $acceptance_letter = $row['acceptance_letter'];

    // Validate GPS location and acceptance letter
    $gps_valid = isValidGPS($gps_location);
    $acceptance_valid = !empty($acceptance_letter);

    // Add to error array if validation fails (either GPS or acceptance letter)
    if (!$gps_valid || !$acceptance_valid) {
        $row['error'] = [
            'gps_error' => !$gps_valid ? 'Invalid GPS location or URL' : '',
            'acceptance_letter_error' => !$acceptance_valid ? 'No acceptance letter uploaded' : ''
        ];
        $error_users[] = $row;
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>GPS and Acceptance Letter Validation</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0-alpha1/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body {
            background-color: #f4f4f4;
            font-family: Arial, sans-serif;
            color: #333;
        }

        .container {
            max-width: 90%;
            margin: 50px auto;
            padding: 20px;
            background-color: white;
            border-radius: 6px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
        }

        h2 {
            font-size: 20px;
            color: #222;
            margin-bottom: 20px;
            text-align: center;
        }

        .table-responsive {
            max-width: 100%;
        }

        .table {
            margin-top: 10px;
            width: 100%;
            border-collapse: collapse;
        }

        .table th, .table td {
            text-align: left;
            padding: 12px;
            word-wrap: break-word;
        }

        .table th {
            font-weight: bold;
            text-transform: uppercase;
            background-color: #f8f9fa;
            color: #555;
            border-bottom: 1px solid #ddd;
        }

        .table td {
            border-bottom: 1px solid #f1f1f1;
        }

        .alert {
            margin-top: 20px;
            font-size: 14px;
        }

        .btn-resubmit {
            background-color: #ff4d4d;
            border: none;
            color: white;
            font-size: 14px;
            padding: 5px 10px;
            border-radius: 4px;
            transition: background-color 0.3s ease;
        }

        .btn-resubmit:hover {
            background-color: #e04444;
        }

        .btn-back {
            background-color: #5a6268;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 4px;
            font-size: 14px;
            transition: background-color 0.3s ease;
            text-align: center;
            text-decoration: none;
        }

        .btn-back:hover {
            background-color: #4e555b;
        }

        .text-center {
            margin-top: 20px;
        }

        /* Adjust for smaller screens */
        @media (max-width: 768px) {
            .table th, .table td {
                font-size: 12px;
                padding: 8px;
            }

            .btn-resubmit {
                padding: 4px 8px;
                font-size: 12px;
            }
        }
    </style>
</head>
<body>

<div class="container mt-5">
    <h2>Users with Errors</h2>
    
    <?php if (!empty($error_users)): ?>
    <div class="table-responsive">
        <table class="table table-bordered">
            <thead>
                <tr>
                    <th>Full Name</th>
                    <th>Telephone</th>
                    <th>Email</th>
                    <th>GPS Location</th>
                    <th>Acceptance Letter</th>
                    <th>Error</th>
                    <th>Resubmit</th>
                </tr>
            </thead>
            <tbody>
            <?php foreach ($error_users as $user): ?>
                <tr>
                    <td><?= htmlspecialchars($user['surname'] . ' ' . $user['first_name'], ENT_QUOTES, 'UTF-8') ?></td>
                    <td><?= htmlspecialchars($user['telephone'], ENT_QUOTES, 'UTF-8') ?></td>
                    <td><?= htmlspecialchars($user['email'], ENT_QUOTES, 'UTF-8') ?></td>
                    <td><?= htmlspecialchars($user['gps_location'], ENT_QUOTES, 'UTF-8') ?></td>
                    <td><?= !empty($user['acceptance_letter']) ? 'Uploaded' : 'Not Uploaded' ?></td>
                    <td>
                        <?= !empty($user['error']['gps_error']) ? htmlspecialchars($user['error']['gps_error'], ENT_QUOTES, 'UTF-8') . '<br>' : '' ?>
                        <?= !empty($user['error']['acceptance_letter_error']) ? htmlspecialchars($user['error']['acceptance_letter_error'], ENT_QUOTES, 'UTF-8') : '' ?>
                    </td>
                    <td>
                        <form method="POST" action="">
                            <input type="hidden" name="user_id" value="<?= htmlspecialchars($user['user_id'], ENT_QUOTES, 'UTF-8') ?>">
                            <button type="submit" name="resubmit" class="btn-resubmit">Resubmit</button>
                        </form>
                    </td>
                </tr>
            <?php endforeach; ?>
            </tbody>
        </table>
    </div>
    <?php else: ?>
    <div class="alert alert-success" role="alert">
        All users have valid GPS locations and acceptance letters.
    </div>
    <?php endif; ?>

    <div class="text-center">
        <a href="index.php" class="btn-back">Back to Dashboard</a>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0-alpha1/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>


