<?php
session_start();
if (!isset($_SESSION['lecturer_id'])) {
    header('Location: login');
    exit;
}

include 'lect_header.php';
include '../config/database.php';

$lecturerId = $_SESSION['lecturer_id'];

// Fetch lecturer's name and SMS status
$stmt = $pdo->prepare('SELECT name, sms_sent FROM lecturer_login WHERE id = ?');
$stmt->execute([$lecturerId]);
$lecturer = $stmt->fetch();
$lecturerName = $lecturer ? htmlspecialchars($lecturer['name'], ENT_QUOTES, 'UTF-8') : '';
$smsSent = $lecturer ? $lecturer['sms_sent'] : 1;

// Fetch assigned students' details along with their index numbers
$stmt = $pdo->prepare('
    SELECT sd.id, sd.first_name, sd.surname, u.index_number, sd.telephone, sd.organisation_name, sd.location, sd.district, sd.region, sd.gps_location 
    FROM student_details sd 
    JOIN users u ON sd.user_id = u.id
    WHERE sd.lecturer_id = ?
');
$stmt->execute([$lecturerId]);
$students = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<div class="container mt-4">
    <h2>Assigned Students</h2>
    <?php if (empty($students)): ?>
        <div class="alert alert-info" role="alert">
            You have not been assigned any students yet.
        </div>
    <?php else: ?>
        <!-- Hidden print header (shown only in print) -->
        <div id="printHeader" class="d-none">
            <h3>Students assigned to <?php echo $lecturerName; ?></h3>
        </div>

        <!-- Desktop/Large Screen Table (hidden below md by Bootstrap class d-none d-md-block) -->
        <div class="table-responsive d-none d-md-block">
            <table class="table table-striped" id="studentsTable">
                <thead>
                    <tr>
                        <!-- Removed “No.” column -->
                        <th>Name</th>
                        <th>Index Number</th>
                        <th>Telephone Number</th>
                        <th>Name of Organisation</th>
                        <th>Location</th>
                        <th>District/Region</th>
                        <th>GPS Location</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($students as $index => $student): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($student['first_name'] . ' ' . $student['surname'], ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><?php echo htmlspecialchars($student['index_number'], ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><?php echo htmlspecialchars($student['telephone'], ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><?php echo htmlspecialchars($student['organisation_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><?php echo htmlspecialchars($student['location'], ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><?php echo htmlspecialchars($student['district'] . '/' . substr($student['region'], 0, 2), ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><?php echo htmlspecialchars($student['gps_location'], ENT_QUOTES, 'UTF-8'); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <!-- Mobile Accordion (visible below md) -->
        <div class="accordion d-md-none" id="studentsAccordion">
            <?php foreach ($students as $index => $student): ?>
                <div class="accordion-item">
                    <h2 class="accordion-header" id="heading<?php echo $index; ?>">
                        <button class="accordion-button collapsed"
                                type="button" data-bs-toggle="collapse"
                                data-bs-target="#collapse<?php echo $index; ?>"
                                aria-expanded="false"
                                aria-controls="collapse<?php echo $index; ?>">
                            <?php echo htmlspecialchars($student['first_name'] . ' ' . $student['surname'], ENT_QUOTES, 'UTF-8'); ?>
                        </button>
                    </h2>
                    <div id="collapse<?php echo $index; ?>" 
                         class="accordion-collapse collapse"
                         aria-labelledby="heading<?php echo $index; ?>"
                         data-bs-parent="#studentsAccordion">
                        <div class="accordion-body">
                            <p><strong>Index Number:</strong>
                                <?php echo htmlspecialchars($student['index_number'], ENT_QUOTES, 'UTF-8'); ?>
                            </p>
                            <p><strong>Telephone Number:</strong>
                                <?php echo htmlspecialchars($student['telephone'], ENT_QUOTES, 'UTF-8'); ?>
                            </p>
                            <p><strong>Name of Organisation:</strong>
                                <?php echo htmlspecialchars($student['organisation_name'], ENT_QUOTES, 'UTF-8'); ?>
                            </p>
                            <p><strong>Location:</strong>
                                <?php echo htmlspecialchars($student['location'], ENT_QUOTES, 'UTF-8'); ?>
                            </p>
                            <p><strong>District/Region:</strong>
                                <?php echo htmlspecialchars($student['district'] . '/' . substr($student['region'], 0, 2), ENT_QUOTES, 'UTF-8'); ?>
                            </p>
                            <p><strong>GPS Location:</strong>
                                <?php echo htmlspecialchars($student['gps_location'], ENT_QUOTES, 'UTF-8'); ?>
                            </p>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>

        <!-- Action Buttons -->
        <button class="btn btn-outline-custom mt-3" 
                id="sendSmsButton" 
                <?php echo $smsSent ? 'disabled' : ''; ?>>
            Send Bulk SMS
        </button>
        <!-- Hide the Print button on mobile (below md) -->
        <button class="btn btn-outline-custom mt-3 d-none d-md-inline-block" 
                id="printButton">
            Print
        </button>
    <?php endif; ?>
</div>

<!-- SMS Modal -->
<div class="modal fade" id="smsModal" tabindex="-1" aria-labelledby="smsModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="smsModalLabel">Compose SMS</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <form id="smsForm">
                    <div class="mb-3">
                        <label for="smsMessage" class="form-label">Message</label>
                        <textarea class="form-control" id="smsMessage" rows="3" maxlength="160">
You have been assigned to <?php echo $lecturerName; ?>. You will be visited for supervision at your organisation soon.
                        </textarea>
                        <div id="charCount" class="form-text">0/160 characters used</div>
                    </div>
                    <button type="submit" class="btn btn-outline-custom">Send SMS</button>
                </form>
            </div>
        </div>
    </div>
</div>

<?php include 'lect_footer.php'; ?>

<script>
document.getElementById('sendSmsButton').addEventListener('click', function() {
    const smsModal = new bootstrap.Modal(document.getElementById('smsModal'));
    smsModal.show();
});

document.getElementById('smsMessage').addEventListener('input', function() {
    const charCount = this.value.length;
    document.getElementById('charCount').textContent = `${charCount}/160 characters used`;
});

document.getElementById('smsForm').addEventListener('submit', function(event) {
    event.preventDefault();
    const message = document.getElementById('smsMessage').value;
    if (message.length <= 160) {
        sendBulkSMS(message);
    } else {
        alert('Message exceeds 160 characters.');
    }
});

document.getElementById('printButton').addEventListener('click', function() {
    const printHeader = document.getElementById('printHeader');
    printHeader.classList.remove('d-none');
    window.print();
    printHeader.classList.add('d-none');
});

function sendBulkSMS(message) {
    const xhr = new XMLHttpRequest();
    xhr.open('POST', 'send_sms.php', true);
    xhr.setRequestHeader('Content-Type', 'application/json');
    xhr.onreadystatechange = function() {
        if (xhr.readyState === XMLHttpRequest.DONE) {
            const response = JSON.parse(xhr.responseText);
            alert(response.message);
            if (response.status === 'success') {
                // Disable the button after a successful send
                document.getElementById('sendSmsButton').disabled = true;

                setTimeout(function() {
                    const smsModal = bootstrap.Modal.getInstance(document.getElementById('smsModal'));
                    smsModal.hide();
                }, 3000);
            }
        }
    };
    const payload = {
        message: message,
        students: <?php echo json_encode(array_map(function($student) {
            return htmlspecialchars($student['telephone'], ENT_QUOTES, 'UTF-8');
        }, $students)); ?>
    };
    xhr.send(JSON.stringify(payload));
}
</script>

<style>
@media print {
    /* Hide everything by default */
    body * {
        visibility: hidden;
    }

    /* Make the print header and the table (and its children) visible in print */
    #printHeader, #studentsTable, #studentsTable * {
        visibility: visible;
    }

    /* IMPORTANT: Override Bootstrap's .d-none so the table can actually show in print */
    .table-responsive.d-none.d-md-block {
        display: block !important;
        visibility: visible;
    }

    #printHeader {
        position: absolute;
        top: 0;
        left: 0;
        width: 100%;
        text-align: center;
        margin-bottom: 20px;
    }
    #studentsTable {
        position: absolute;
        top: 50px;
        left: 0;
        width: 100%;
    }

    /* Force landscape mode (optional) */
    @page {
        size: landscape;
    }
}
</style>
