<?php
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

require '../config/database.php';

// =====================================================
// Step 1: Process an unassignment (Return) action if requested
// =====================================================
if (isset($_GET['return_student_id'])) {
    // Sanitize/validate the student id as needed
    $studentId = $_GET['return_student_id'];

    // Unassign the lecturer by setting lecturer_id to NULL
    $stmt = $pdo->prepare("UPDATE student_details SET lecturer_id = NULL WHERE id = ?");
    $stmt->execute([$studentId]);

    // Optionally add a flash message here

    // Build the redirect URL using existing GET parameters but remove the return_student_id parameter.
    $queryParams = $_GET;
    unset($queryParams['return_student_id']);
    $redirectUrl = $_SERVER['PHP_SELF'] . (!empty($queryParams) ? '?' . http_build_query($queryParams) : '');
    header("Location: " . $redirectUrl);
    exit;
}

// =====================================================
// Function to fetch summary
// =====================================================
function fetchSummary($pdo) {
    $summary = [];

    // Total submitted students
    $query = 'SELECT COUNT(*) AS total FROM student_details WHERE status = "submitted"';
    $stmt = $pdo->prepare($query);
    $stmt->execute();
    $summary['total_submitted'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'];

    // Total assigned students
    $query = 'SELECT COUNT(*) AS total 
              FROM student_details 
              WHERE status = "submitted" 
                AND lecturer_id IS NOT NULL';
    $stmt = $pdo->prepare($query);
    $stmt->execute();
    $summary['total_assigned'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'];

    // Total unassigned students
    $summary['total_unassigned'] = $summary['total_submitted'] - $summary['total_assigned'];

    return $summary;
}

// =====================================================
// Function to fetch lecturers data (modified to include student id)
// =====================================================
function fetchLecturersData($pdo) {
    $query = '
        SELECT 
            ll.id AS lecturer_id,
            ll.name AS lecturer_name,
            lad.region AS assigned_region,
            lad.district AS assigned_district,
            sd.id AS student_id,              -- New: Student id for unassignment
            sd.first_name,
            sd.surname,
            sd.organisation_name,
            sd.location
        FROM lecturer_login ll
        LEFT JOIN lecturer_assigned_districts lad 
               ON ll.id = lad.lecturer_id
        LEFT JOIN student_details sd 
               ON sd.lecturer_id = ll.id
              AND sd.district = lad.district
              AND sd.status = "submitted"
        ORDER BY ll.name ASC, lad.region ASC, lad.district ASC, sd.surname ASC
    ';
    $stmt = $pdo->prepare($query);
    $stmt->execute();
    $data = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Organize data by lecturer, region, and district
    $lecturerData = [];
    foreach ($data as $row) {
        $lecturerId   = $row['lecturer_id'];
        $lecturerName = ucwords(strtolower($row['lecturer_name']));
        $region       = ucwords(strtolower($row['assigned_region']));
        $district     = ucwords(strtolower($row['assigned_district']));

        if (!isset($lecturerData[$lecturerId])) {
            $lecturerData[$lecturerId] = [
                'name'           => $lecturerName,
                'regions'        => [],
                'total_students' => 0
            ];
        }

        if (!isset($lecturerData[$lecturerId]['regions'][$region])) {
            $lecturerData[$lecturerId]['regions'][$region] = [];
        }

        if (!isset($lecturerData[$lecturerId]['regions'][$region][$district])) {
            $lecturerData[$lecturerId]['regions'][$region][$district] = [];
        }

        // Only count a student if there's a first_name (meaning there's actual data)
        if (!empty($row['first_name'])) {
            $lecturerData[$lecturerId]['regions'][$region][$district][] = [
                'id'           => $row['student_id'], // Store student id
                'name'         => ucwords(strtolower($row['first_name'] . ' ' . $row['surname'])),
                'organisation' => $row['organisation_name'],
                'location'     => $row['location']
            ];
            $lecturerData[$lecturerId]['total_students']++;
        }
    }

    return $lecturerData;
}

// Fetch summary and lecturers data
$summary      = fetchSummary($pdo);
$lecturerData = fetchLecturersData($pdo);

// Get filters
$selectedLecturer = $_GET['lecturer'] ?? '';
$selectedRegion   = $_GET['region']   ?? '';

// Attempt to find the chosen lecturer in our data
$chosenLecturer = null;
if (!empty($selectedLecturer) && isset($lecturerData[$selectedLecturer])) {
    $chosenLecturer = $lecturerData[$selectedLecturer];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>View Assigned Students</title>
    <!-- Bootstrap CSS -->
    <link 
      href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" 
      rel="stylesheet"
    >
    <!-- Font Awesome (optional for icons) -->
    <link 
      rel="stylesheet" 
      href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.1.1/css/all.min.css"
    >
    <style>
        /* Hide elements that are not to be printed */
        @media print {
            .d-print-none {
                display: none !important;
            }
        }
    </style>
    <script>
      // Function to print only the results area
      function printResults() {
          var divToPrint = document.getElementById("printableArea");
          var newWin = window.open("", "Print-Window");
          newWin.document.open();
          // You can include additional styles for printing as needed
          newWin.document.write("<html><head><title>Print Results</title>");
          newWin.document.write('<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css">');
          newWin.document.write("<style>table { width: 100%; border-collapse: collapse; } th, td { border: 1px solid #000; padding: 8px; }</style>");
          newWin.document.write("</head><body onload='window.print(); window.close();'>");
          newWin.document.write(divToPrint.innerHTML);
          newWin.document.write("</body></html>");
          newWin.document.close();
      }
    </script>
</head>
<body>

<!-- Include your admin_header (which has the sidebar) -->
<?php include 'admin_header.php'; ?>

<div class="container mt-4">
    <h1 class="mb-4">Assigned Students Summary</h1>

    <!-- Back to Assign Students -->
    <div class="mb-4">
        <a href="assign_students" class="btn btn-secondary d-print-none">
            <i class="fas fa-arrow-left"></i> Back to Assign Students
        </a>
    </div>

    <!-- Export Options -->
    <div class="mb-4 d-print-none">
        <a href="export_assigned_students.php?type=summary" class="btn btn-info">
            <i class="fas fa-file-excel"></i> Export Summary Only
        </a>
        <a href="export_assigned_students.php?type=all" class="btn btn-success">
            <i class="fas fa-file-excel"></i> Export All Details
        </a>
        <form action="export_assigned_students.php" method="GET" style="display: inline;">
            <input type="hidden" name="type" value="lecturer">
            <select name="lecturer" class="form-select d-inline-block w-auto" required>
                <option value="">Select Lecturer</option>
                <?php foreach ($lecturerData as $lectId => $lectr): ?>
                    <option value="<?= $lectId ?>"><?= $lectr['name'] ?></option>
                <?php endforeach; ?>
            </select>
            <button type="submit" class="btn btn-primary">Export Lecturer Sheet</button>
        </form>
    </div>

    <!-- Summary Section -->
    <div class="card mb-4 d-print-none">
        <div class="card-body">
            <h4>Summary</h4>
            <p><strong>Total Submitted Students:</strong> <?= $summary['total_submitted'] ?></p>
            <p><strong>Total Assigned Students:</strong> <?= $summary['total_assigned'] ?></p>
            <p><strong>Total Unassigned Students:</strong> <?= $summary['total_unassigned'] ?></p>
        </div>
    </div>

    <!-- Filter Section -->
    <div class="card mb-4 d-print-none">
        <div class="card-body">
            <h4>Filter</h4>
            <form id="filterForm" method="GET">
                <div class="row">
                    <!-- Lecturer Filter -->
                    <div class="col-md-6 mb-3">
                        <label for="lecturerFilter" class="form-label">Select Lecturer</label>
                        <select name="lecturer" id="lecturerFilter" class="form-select">
                            <option value="">-- Choose a Lecturer --</option>
                            <?php foreach ($lecturerData as $lectId => $lectr): ?>
                                <option 
                                  value="<?= $lectId ?>"
                                  <?php if ($selectedLecturer == $lectId) echo 'selected'; ?>
                                >
                                    <?= $lectr['name'] ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <!-- Region Filter -->
                    <div class="col-md-6 mb-3">
                        <label for="regionFilter" class="form-label">Select Region</label>
                        <select name="region" id="regionFilter" class="form-select">
                            <option value="">-- All Regions --</option>
                            <?php
                            // Gather all possible regions
                            $allRegions = [];
                            foreach ($lecturerData as $lectr) {
                                foreach (array_keys($lectr['regions']) as $reg) {
                                    $allRegions[$reg] = $reg;
                                }
                            }
                            ?>
                            <?php foreach ($allRegions as $reg): ?>
                                <option 
                                  value="<?= $reg ?>"
                                  <?php if ($selectedRegion == $reg) echo 'selected'; ?>
                                >
                                    <?= $reg ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>
                <!-- Filter Button -->
                <button type="submit" class="btn btn-primary d-print-none">Filter</button>
            </form>
        </div>
    </div>

    <!-- Results Section (printable area) -->
    <div id="printableArea" class="card">
        <div class="card-body">
            <h4>Results</h4>
            
            <?php if (empty($selectedLecturer)): ?>
                <!-- No lecturer selected -->
                <p class="text-muted">
                    No lecturer selected. Please select a lecturer above.
                </p>

            <?php elseif (!$chosenLecturer): ?>
                <!-- Lecturer selected, but not found in data array -->
                <p class="text-danger">
                    No data found for the selected lecturer.
                </p>

            <?php else: ?>
                <!-- Print Button (visible only on screen) -->
                <div class="d-print-none mb-3">
                    <button onclick="printResults()" class="btn btn-outline-secondary">
                        <i class="fas fa-print"></i> Print
                    </button>
                </div>
                
                <!-- Show data for the single chosen lecturer -->
                <h5 class="mt-4">
                    <?= $chosenLecturer['name'] ?>
                    (<?= $chosenLecturer['total_students'] ?> students assigned)
                </h5>

                <!-- A single table for all regions/districts -->
                <table class="table table-striped table-bordered mt-3">
                    <thead>
                        <tr>
                            <th scope="col">Region</th>
                            <th scope="col">District</th>
                            <th scope="col">Student Name</th>
                            <th scope="col">Organisation</th>
                            <th scope="col">Location</th>
                            <th scope="col" class="d-print-none">Action</th> <!-- Action column hidden in print -->
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($chosenLecturer['regions'] as $region => $districts): ?>
                            <!-- If user selected a region, skip any that doesn't match -->
                            <?php if ($selectedRegion && $selectedRegion != $region) continue; ?>

                            <?php foreach ($districts as $district => $students): ?>
                                <?php if (!empty($students)): ?>
                                    <!-- Row per student -->
                                    <?php foreach ($students as $student): ?>
                                        <tr>
                                            <td><?= $region ?></td>
                                            <td><?= $district ?></td>
                                            <td><?= $student['name'] ?></td>
                                            <td><?= $student['organisation'] ?></td>
                                            <td><?= $student['location'] ?></td>
                                            <td class="d-print-none">
                                                <?php
                                                // Build the return URL to include the current filter parameters along with return_student_id
                                                $currentParams = $_GET;
                                                $currentParams['return_student_id'] = $student['id'];
                                                $returnUrl = $_SERVER['PHP_SELF'] . '?' . http_build_query($currentParams);
                                                ?>
                                                <a 
                                                  href="<?= $returnUrl ?>"
                                                  class="btn btn-warning btn-sm"
                                                  onclick="return confirm('Are you sure you want to unassign this student?');"
                                                >
                                                    Return
                                                </a>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            <?php endforeach; ?>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- Bootstrap Bundle JS -->
<script 
  src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js">
</script>

</body>
</html>
