<?php
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
if (!isset($_SESSION['admin_id'])) {
    header('Location: login');
    exit;
}

// Database connection settings
include '../config/database.php';
// Data Source Name (DSN)
$dsn = "mysql:host=" . htmlspecialchars($host, ENT_QUOTES, 'UTF-8') . ";dbname=" . htmlspecialchars($db, ENT_QUOTES, 'UTF-8') . ";charset=" . htmlspecialchars($charset, ENT_QUOTES, 'UTF-8');

$options = [
    PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION, // Enable exceptions for errors
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,       // Set default fetch mode to associative array
    PDO::ATTR_EMULATE_PREPARES   => false,                  // Disable emulation of prepared statements
];

try {
    // Create a PDO instance (connect to the database)
    $pdo = new PDO($dsn, $user, $pass, $options);
} catch (\PDOException $e) {
    // Handle connection error
    throw new \PDOException(htmlspecialchars($e->getMessage(), ENT_QUOTES, 'UTF-8'), (int)$e->getCode());
}

// Fetch all students with status and filter parameters
$status = isset($_GET['status']) ? $_GET['status'] : 'all';
$group = isset($_GET['group_filter']) ? $_GET['group_filter'] : '';
$region = isset($_GET['region_filter']) ? $_GET['region_filter'] : '';

// Prepare the base query with a JOIN to the `users` table for group filtering
$query = '
    SELECT sd.*, u.`group` 
    FROM student_details sd
    JOIN users u ON sd.user_id = u.id
    WHERE 1';

// Add status filtering
if ($status === 'submitted') {
    $query .= ' AND sd.status = "submitted"';
} elseif ($status === 'saved') {
    $query .= ' AND sd.status = "saved"';
}

// Add group filtering if selected (from users table)
if (!empty($group)) {
    $query .= ' AND u.`group` = :group';
}

// Add region filtering if selected
if (!empty($region)) {
    $query .= ' AND sd.region = :region';
}

$stmt = $pdo->prepare($query);

// Bind parameters if group or region filters are applied
if (!empty($group)) {
    $stmt->bindParam(':group', $group);
}
if (!empty($region)) {
    $stmt->bindParam(':region', $region);
}

$stmt->execute();
$students = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch distinct groups from the users table
$groupStmt = $pdo->prepare('SELECT DISTINCT `group` FROM users');
$groupStmt->execute();
$groups = $groupStmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch distinct regions from the student_details table
$regionStmt = $pdo->prepare('SELECT DISTINCT region FROM student_details');
$regionStmt->execute();
$regions = $regionStmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Student List</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="admin_style.css" rel="stylesheet">
    <!-- FontAwesome for icons -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <style>
        .container {
            max-width: 1200px;
        }

        .filter-row {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 15px;
        }

        .form-select, .btn {
            max-width: 200px;
            margin-right: 10px;
        }

        .btn-outline-tomato {
            border-color: tomato;
            color: tomato;
        }

        .btn-outline-tomato:hover {
            background-color: tomato;
            color: white;
        }

        .table {
            margin-top: 15px;
        }

        .modal-body p {
            margin-bottom: 10px; /* Add spacing between elements */
            word-wrap: break-word; /* Ensures the URL breaks and wraps */
            white-space: normal;   /* Prevents text from staying on one line */
        }
    </style>
</head>
<body>

<?php include 'admin_header.php'; ?>

<div class="container mt-4">
    <h2 class="mb-3">All Students</h2>

    <div class="filter-row">
        <div class="d-flex">
            <select id="groupFilter" name="group_filter" class="form-select">
                <option value="">Filter by Group</option>
                <?php foreach ($groups as $group): ?>
                    <option value="<?php echo htmlspecialchars($group['group'], ENT_QUOTES, 'UTF-8'); ?>" <?php echo isset($_GET['group_filter']) && $_GET['group_filter'] === $group['group'] ? 'selected' : ''; ?>>
                        <?php echo htmlspecialchars($group['group'], ENT_QUOTES, 'UTF-8'); ?>
                    </option>
                <?php endforeach; ?>
            </select>
            <select id="regionFilter" name="region_filter" class="form-select">
                <option value="">Filter by Region</option>
                <?php foreach ($regions as $region): ?>
                    <option value="<?php echo htmlspecialchars($region['region'], ENT_QUOTES, 'UTF-8'); ?>" <?php echo isset($_GET['region_filter']) && $_GET['region_filter'] === $region['region'] ? 'selected' : ''; ?>>
                        <?php echo htmlspecialchars($region['region'], ENT_QUOTES, 'UTF-8'); ?>
                    </option>
                <?php endforeach; ?>
            </select>
            <button id="filterButton" class="btn btn-outline-tomato" onclick="applyFilters()">Filter</button>
        </div>

        <!-- Status filter buttons -->
        <div>
            <a href="?status=submitted" class="btn btn-outline-success">View Submitted</a>
            <a href="?status=saved" class="btn btn-outline-warning">View Saved</a>
            <a href="?status=all" class="btn btn-outline-primary">View All</a>
        </div>
    </div>

    <?php if (empty($students)): ?>
        <p>No students found.</p>
    <?php else: ?>
        <table class="table table-striped" id="studentTable">
            <thead>
                <tr>
                    <th>Full Name</th>
                    <th>Email</th>
                    <th>Telephone</th>
                    <th>Course</th>
                    <th>Organisation</th>
                    <th>Region</th>
                    <th>Status</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($students as $student): ?>
                    <tr data-group="<?php echo htmlspecialchars($student['group'], ENT_QUOTES, 'UTF-8'); ?>" data-region="<?php echo htmlspecialchars($student['region'], ENT_QUOTES, 'UTF-8'); ?>">
                        <td><?php echo htmlspecialchars($student['first_name'], ENT_QUOTES, 'UTF-8') . ' ' . htmlspecialchars($student['surname'], ENT_QUOTES, 'UTF-8'); ?></td>
                        <td><?php echo htmlspecialchars($student['email'], ENT_QUOTES, 'UTF-8'); ?></td>
                        <td><?php echo htmlspecialchars($student['telephone'], ENT_QUOTES, 'UTF-8'); ?></td>
                        <td><?php echo htmlspecialchars($student['course'], ENT_QUOTES, 'UTF-8'); ?></td>
                        <td><?php echo htmlspecialchars($student['organisation_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                        <td><?php echo htmlspecialchars($student['region'], ENT_QUOTES, 'UTF-8'); ?></td>
                        <td><?php echo htmlspecialchars($student['status'], ENT_QUOTES, 'UTF-8'); ?></td>
                        <td><i class="fas fa-eye" style="cursor:pointer;" data-bs-toggle="modal" data-bs-target="#studentDetailsModal" onclick="showStudentDetails(<?php echo htmlspecialchars(json_encode($student), ENT_QUOTES, 'UTF-8'); ?>)"></i></td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    <?php endif; ?>
</div>

<!-- Modal for Student Details -->
<div class="modal fade" id="studentDetailsModal" tabindex="-1" aria-labelledby="studentDetailsModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="studentDetailsModalLabel">Student Details</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body" id="studentDetailsContent">
                <!-- Details will be injected here by JavaScript -->
            </div>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/@popperjs/core@2.10.2/dist/umd/popper.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.min.js"></script>
<script>
    function applyFilters() {
        const groupFilter = document.getElementById('groupFilter').value;
        const regionFilter = document.getElementById('regionFilter').value;
        const url = new URL(window.location.href);
        
        // Add group filter
        if (groupFilter) {
            url.searchParams.set('group_filter', groupFilter);
        } else {
            url.searchParams.delete('group_filter');
        }
        
        // Add region filter
        if (regionFilter) {
            url.searchParams.set('region_filter', regionFilter);
        } else {
            url.searchParams.delete('region_filter');
        }

        window.location.href = url.toString(); // Reload the page with the selected filters
    }

    function showStudentDetails(student) {
        var content = `
            <p><strong>Surname:</strong> ${student.surname}</p>
            <p><strong>First Name:</strong> ${student.first_name}</p>
            <p><strong>Other Names:</strong> ${student.other_names}</p>
            <p><strong>Telephone:</strong> ${student.telephone}</p>
            <p><strong>Email:</strong> ${student.email}</p>
            <p><strong>Course:</strong> ${student.course}</p>
            <p><strong>Organisation:</strong> ${student.organisation_name}</p>
            <p><strong>Region:</strong> ${student.region}</p>
            <p><strong>Status:</strong> ${student.status}</p>
            <p><strong>GPS Location:</strong> ${student.gps_location ? student.gps_location : 'Not Provided'}</p>
            ${student.acceptance_letter ? `<p><strong>Acceptance Letter:</strong> <a href="../uploads/${student.acceptance_letter}" target="_blank">View</a></p>` : ''}
        `;
        document.getElementById('studentDetailsContent').innerHTML = content;
    }
</script>
</body>
</html>
