<?php
// Include database connection
require_once '../config/database.php';

try {
    // Fetch distinct groups from users table
    $queryGroups = "SELECT DISTINCT `group` FROM users";
    $stmtGroups = $pdo->query($queryGroups);
    $groups = $stmtGroups->fetchAll();

    // Fetch distinct courses from student_details table (submitted only)
    $queryCourses = "SELECT DISTINCT course FROM student_details WHERE status = 'submitted'";
    $stmtCourses = $pdo->query($queryCourses);
    $courses = $stmtCourses->fetchAll();

    // Initialize filters
    $selectedGroups = $_GET['group'] ?? [];
    $selectedCourses = $_GET['course'] ?? [];

    // Base query for filtered statistics (students)
    $baseQuery = "
        SELECT u.group, sd.course, COUNT(sd.id) as total
        FROM users u
        LEFT JOIN student_details sd ON u.id = sd.user_id
        WHERE sd.status = 'submitted'
    ";

    // Apply filters
    $conditions = [];
    if (!empty($selectedGroups)) {
        $placeholders = implode(',', array_fill(0, count($selectedGroups), '?'));
        $conditions[] = "u.group IN ($placeholders)";
    }
    if (!empty($selectedCourses)) {
        $placeholders = implode(',', array_fill(0, count($selectedCourses), '?'));
        $conditions[] = "sd.course IN ($placeholders)";
    }

    // Append conditions to query
    if (!empty($conditions)) {
        $baseQuery .= " AND " . implode(" AND ", $conditions);
    }
    $baseQuery .= " GROUP BY u.group, sd.course";

    // Prepare and execute the query
    $stmt = $pdo->prepare($baseQuery);
    $bindIndex = 1;
    foreach (array_merge($selectedGroups, $selectedCourses) as $value) {
        $stmt->bindValue($bindIndex++, $value);
    }
    $stmt->execute();
    $filteredStats = $stmt->fetchAll();

    // Query for total registered students
    $queryRegistered = "SELECT COUNT(*) as total_registered FROM student_details WHERE status = 'submitted'";
    $stmtRegistered = $pdo->query($queryRegistered);
    $resultRegistered = $stmtRegistered->fetch();
    $totalRegistered = $resultRegistered['total_registered'];

    // Query for total lecturers and departmental breakdown
    $queryLecturers = "
        SELECT department, COUNT(id) as total 
        FROM lecturer_login 
        GROUP BY department
    ";
    $stmtLecturers = $pdo->query($queryLecturers);
    $lecturerStats = $stmtLecturers->fetchAll();

    // Fetch lecturer names grouped by department
    $queryLecturerNames = "SELECT name, department FROM lecturer_login";
    $stmtLecturerNames = $pdo->query($queryLecturerNames);

    // Group lecturer names by department
    $lecturersByDepartment = [];
    while ($row = $stmtLecturerNames->fetch(PDO::FETCH_ASSOC)) {
        $lecturersByDepartment[$row['department']][] = $row['name'];
    }

} catch (PDOException $e) {
    die("Error fetching data: " . $e->getMessage());
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Statistics Report</title>
    <!-- Bootstrap 5 CSS CDN -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        .sidebar {
            border-right: 1px solid #ddd;
            padding-right: 15px;
            min-height: 100vh;
        }
        .main-content {
            padding-left: 15px;
        }
        .section-title {
            font-size: 1.5rem;
            font-weight: bold;
            padding: 10px;
            color: #fff;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .students-header {
            background-color: #007bff; /* Bootstrap primary */
        }
        .lecturers-header {
            background-color: #28a745; /* Bootstrap success */
        }
        .toggle-btn {
            cursor: pointer;
        }
    </style>
</head>
<body>
    <div class="container-fluid mt-5">
        <div class="row">
            <!-- Sidebar Filters -->
            <div class="col-md-3 sidebar">
                <h5>Filters</h5>
                <form method="GET">
                    <div class="mb-3">
                        <h6>Filter by Group</h6>
                        <?php foreach ($groups as $group): ?>
                            <div class="form-check">
                                <input 
                                    class="form-check-input" 
                                    type="checkbox" 
                                    name="group[]" 
                                    value="<?= htmlspecialchars($group['group']) ?>" 
                                    id="group-<?= htmlspecialchars($group['group']) ?>"
                                    <?= in_array($group['group'], $selectedGroups) ? 'checked' : '' ?>>
                                <label class="form-check-label" for="group-<?= htmlspecialchars($group['group']) ?>">
                                    <?= htmlspecialchars($group['group']) ?>
                                </label>
                            </div>
                        <?php endforeach; ?>
                    </div>

                    <div class="mb-3">
                        <h6>Filter by Department</h6>
                        <?php foreach ($courses as $course): ?>
                            <div class="form-check">
                                <input 
                                    class="form-check-input" 
                                    type="checkbox" 
                                    name="course[]" 
                                    value="<?= htmlspecialchars($course['course']) ?>" 
                                    id="course-<?= htmlspecialchars($course['course']) ?>"
                                    <?= in_array($course['course'], $selectedCourses) ? 'checked' : '' ?>>
                                <label class="form-check-label" for="course-<?= htmlspecialchars($course['course']) ?>">
                                    <?= htmlspecialchars($course['course']) ?>
                                </label>
                            </div>
                        <?php endforeach; ?>
                    </div>

                    <button type="submit" class="btn btn-primary">Apply</button>
                    <a href="stat-report.php" class="btn btn-secondary">Reset</a>
                </form>
            </div>

            <!-- Main Content -->
            <div class="col-md-9 main-content">
                <!-- Total Registered Students -->
                <div>
                    <h2 class="section-title students-header">
                        Total Registered Students 
                        <span class="badge bg-light text-dark"><?= htmlspecialchars($totalRegistered) ?></span>
                    </h2>
                    <ul class="list-group">
                        <?php foreach ($filteredStats as $stat): ?>
                            <li class="list-group-item d-flex justify-content-between align-items-center">
                                <?= htmlspecialchars($stat['group']) ?> - <?= htmlspecialchars($stat['course']) ?>
                                <span class="badge bg-primary"><?= htmlspecialchars($stat['total']) ?></span>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                </div>

                <!-- Total Lecturers -->
                <div class="mt-4">
                    <h2 class="section-title lecturers-header">
                        Total Lecturers 
                        <span class="badge bg-light text-dark"><?= htmlspecialchars(array_sum(array_column($lecturerStats, 'total'))) ?></span>
                    </h2>
                    <ul class="list-group">
                        <?php foreach ($lecturerStats as $lecturer): ?>
                            <li 
                                class="list-group-item d-flex justify-content-between align-items-center toggle-btn"
                                onclick="openLecturerModal(
                                    '<?= htmlspecialchars($lecturer['department']) ?>',
                                    <?= json_encode($lecturersByDepartment[$lecturer['department']] ?? []) ?>
                                )">
                                <?= htmlspecialchars($lecturer['department']) ?>
                                <span class="badge bg-success"><?= htmlspecialchars($lecturer['total']) ?></span>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            </div>
        </div>
    </div>

    <!-- Modal -->
    <div class="modal fade" id="lecturerModal" tabindex="-1" aria-labelledby="lecturerModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="lecturerModalLabel">Lecturers in <span id="modalDepartmentName"></span></h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body" id="lecturerModalBody">
                    <!-- Lecturer names will be dynamically added here -->
                </div>
            </div>
        </div>
    </div>

    <!-- Bootstrap 5 JS Bundle (includes Popper.js) -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function openLecturerModal(department, lecturers) {
            document.getElementById('modalDepartmentName').textContent = department;
            const modalBody = document.getElementById('lecturerModalBody');
            modalBody.innerHTML = "";

            if (lecturers.length > 0) {
                lecturers.forEach(lecturer => {
                    const listItem = document.createElement('p');
                    listItem.textContent = lecturer;
                    modalBody.appendChild(listItem);
                });
            } else {
                modalBody.textContent = "No lecturers found for this department.";
            }
        }
    </script>
</body>
</html>