<?php
// Start session if it's not already started
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Check if the admin is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Location: login');
    exit;
}

include '../config/database.php'; // Include the database connection file

// Initialize success and error messages
$successMessage = '';
$errorMessage = '';

// Handle CSV Upload
if (isset($_POST['upload'])) {
    if (isset($_FILES['csv_file']) && $_FILES['csv_file']['error'] == 0) {
        $fileTmpPath = $_FILES['csv_file']['tmp_name'];
        $fileName = $_FILES['csv_file']['name'];
        $fileExtension = pathinfo($fileName, PATHINFO_EXTENSION);

        if ($fileExtension == 'csv') {
            if (($handle = fopen($fileTmpPath, "r")) !== FALSE) {
                $pdo->beginTransaction(); // Begin transaction for multiple inserts
                try {
                    $errorRows = []; // To keep track of failed insertions

                    while (($data = fgetcsv($handle, 1000, ",")) !== FALSE) {
                        $staff_id = trim($data[0]);

                        // Ensure staff_id length matches valid_staff_id's varchar(20)
                        if (strlen($staff_id) > 20) {
                            $errorRows[] = "Staff ID '$staff_id' exceeds the allowed length of 20 characters.";
                            continue;
                        }

                        // Insert staff_id into valid_staff_id table
                        $stmt = $pdo->prepare("INSERT INTO valid_staff_id (staff_id) VALUES (:staff_id)");
                        try {
                            $stmt->bindParam(':staff_id', $staff_id);
                            $stmt->execute();
                        } catch (Exception $e) {
                            $errorRows[] = "Failed to insert '$staff_id': " . $e->getMessage();
                        }
                    }

                    $pdo->commit(); // Commit the transaction
                    fclose($handle);

                    if (count($errorRows) > 0) {
                        $errorMessage = "Some rows failed to insert:<br>" . implode("<br>", $errorRows);
                    } else {
                        $successMessage = "Staff IDs uploaded successfully.";
                    }
                } catch (Exception $e) {
                    $pdo->rollBack(); // Rollback transaction on error
                    $errorMessage = "Failed to upload CSV: " . $e->getMessage();
                }
            } else {
                $errorMessage = "Unable to open the CSV file.";
            }
        } else {
            $errorMessage = "Invalid file type. Please upload a CSV file.";
        }
    } else {
        $errorMessage = "Error uploading the file.";
    }

    // PRG Pattern - Redirect to the same page to avoid form resubmission
    header("Location: " . $_SERVER['PHP_SELF']);
    exit();
}

// Handle Edit Action
if (isset($_POST['edit'])) {
    $staff_id = $_POST['staff_id'];
    $name = $_POST['name'];

    // Update the lecturer's name in the lecturer_login table
    $stmt = $pdo->prepare("UPDATE lecturer_login SET name = :name WHERE staff_id = :staff_id");
    $stmt->bindParam(':name', $name);
    $stmt->bindParam(':staff_id', $staff_id);
    $stmt->execute();

    $successMessage = "Lecturer name updated successfully.";

    // PRG Pattern - Redirect to the same page to avoid form resubmission
    header("Location: " . $_SERVER['PHP_SELF']);
    exit();
}

// Handle Delete Action
if (isset($_POST['delete'])) {
    $staff_id = $_POST['staff_id'];

    // Delete the staff record from lecturer_login
    $stmt = $pdo->prepare("DELETE FROM lecturer_login WHERE staff_id = :staff_id");
    $stmt->bindParam(':staff_id', $staff_id);
    $stmt->execute();

    $successMessage = "Staff member deleted successfully.";

    // PRG Pattern - Redirect to the same page to avoid form resubmission
    header("Location: " . $_SERVER['PHP_SELF']);
    exit();
}

// Fetch the list of valid staff IDs and the corresponding lecturer details
$query = "SELECT v.staff_id, l.name, 
          CASE 
              WHEN l.staff_id IS NOT NULL THEN 'Registered' 
              ELSE 'Not Registered' 
          END AS status 
          FROM valid_staff_id v 
          LEFT JOIN lecturer_login l ON v.staff_id = l.staff_id";
$statement = $pdo->query($query);
$staffList = $statement->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Staff IDs</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        .container-form {
            max-width: 500px; /* Shortened form width */
            margin: 0 auto; /* Center the form */
            padding: 30px;
        }

        .table {
            margin-top: 30px;
        }

        .card {
            border-radius: 8px;
            padding: 20px;
            box-shadow: 0 4px 8px rgba(0,0,0,0.1);
        }

        .action-icons i {
            font-size: 18px;
            cursor: pointer;
            margin-right: 10px;
        }
    </style>
</head>
<body>

<?php include 'admin_header.php'; ?>

<div class="container mt-5">
    <div class="container-form">
        <div class="card">
            <h2 class="text-center">Upload Staff IDs</h2>

            <!-- Success/Error Messages -->
            <?php if (!empty($successMessage)): ?>
                <div class="alert alert-success text-center">
                    <?php echo htmlspecialchars($successMessage); ?>
                </div>
            <?php endif; ?>
            <?php if (!empty($errorMessage)): ?>
                <div class="alert alert-danger text-center">
                    <?php echo htmlspecialchars($errorMessage); ?>
                </div>
            <?php endif; ?>

            <form action="" method="POST" enctype="multipart/form-data">
                <div class="mb-3">
                    <label for="csv_file" class="form-label">Choose CSV File</label>
                    <input type="file" name="csv_file" id="csv_file" class="form-control" required>
                </div>
                <button type="submit" name="upload" class="btn btn-primary w-100">Upload</button>
            </form>
        </div>
    </div>

    <!-- Display the staff list -->
    <h3 class="text-center mt-5">Staff Registration Status</h3>
    <table class="table table-striped mt-3">
        <thead>
            <tr>
                <th>Staff ID</th>
                <th>Lecturer Name</th>
                <th>Status</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php if (!empty($staffList)): ?>
                <?php foreach ($staffList as $staff): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($staff['staff_id']); ?></td>
                        <td><?php echo htmlspecialchars($staff['name'] ?? ''); ?></td>
                        <td><?php echo htmlspecialchars($staff['status']); ?></td>
                        <td class="action-icons">
                            <!-- Edit Icon -->
                            <a href="#" data-bs-toggle="modal" data-bs-target="#editModal<?php echo $staff['staff_id']; ?>"><i class="fas fa-edit text-primary"></i></a>
                            <!-- Delete Icon -->
                            <a href="#" data-bs-toggle="modal" data-bs-target="#deleteModal<?php echo $staff['staff_id']; ?>"><i class="fas fa-trash text-danger"></i></a>
                        </td>
                    </tr>

                    <!-- Edit Modal -->
                    <div class="modal fade" id="editModal<?php echo $staff['staff_id']; ?>" tabindex="-1" aria-labelledby="editModalLabel" aria-hidden="true">
                        <div class="modal-dialog">
                            <div class="modal-content">
                                <div class="modal-header">
                                    <h5 class="modal-title" id="editModalLabel">Edit Lecturer Name</h5>
                                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                </div>
                                <div class="modal-body">
                                    <form action="" method="POST">
                                        <input type="hidden" name="staff_id" value="<?php echo $staff['staff_id']; ?>">
                                        <div class="mb-3">
                                            <label for="name" class="form-label">Lecturer Name</label>
                                            <input type="text" class="form-control" name="name" value="<?php echo htmlspecialchars($staff['name']); ?>" required>
                                        </div>
                                        <div class="modal-footer">
                                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                            <button type="submit" name="edit" class="btn btn-primary">Save Changes</button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Delete Modal -->
                    <div class="modal fade" id="deleteModal<?php echo $staff['staff_id']; ?>" tabindex="-1" aria-labelledby="deleteModalLabel" aria-hidden="true">
                        <div class="modal-dialog">
                            <div class="modal-content">
                                <div class="modal-header">
                                    <h5 class="modal-title" id="deleteModalLabel">Delete Staff Member</h5>
                                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                </div>
                                <div class="modal-body">
                                    <p>Are you sure you want to delete this staff member with Staff ID: <?php echo htmlspecialchars($staff['staff_id']); ?>?</p>
                                </div>
                                <div class="modal-footer">
                                    <form action="" method="POST">
                                        <input type="hidden" name="staff_id" value="<?php echo $staff['staff_id']; ?>">
                                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                        <button type="submit" name="delete" class="btn btn-danger">Delete</button>
                                    </form>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php else: ?>
                <tr>
                    <td colspan="4" class="text-center">No staff registered yet.</td>
                </tr>
            <?php endif; ?>
        </tbody>
    </table>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/js/all.min.js"></script>
</body>
</html>
