<?php
session_start();
// Optionally ensure an admin is logged in:
// if (!isset($_SESSION['admin_id'])) {
//     header('Location: login.php');
//     exit;
// }

include '../config/database.php'; // Adjust path as needed

/**********************************************
 * 1. Get filter inputs (region & district)
 **********************************************/
$regionFilter   = isset($_GET['region'])   ? trim($_GET['region'])   : '';
$districtFilter = isset($_GET['district']) ? trim($_GET['district']) : '';

/**********************************************
 * 2. Fetch all distinct regions for the filter dropdown
 **********************************************/
$stmt = $pdo->query('SELECT DISTINCT region FROM student_details ORDER BY region');
$allRegions = $stmt->fetchAll(PDO::FETCH_COLUMN);

/**********************************************
 * 3. Fetch relevant districts for the selected region
 *    If no region is selected, fetch all distinct districts.
 **********************************************/
if (!empty($regionFilter)) {
    // Only get districts from the chosen region
    $stmt = $pdo->prepare('
        SELECT DISTINCT district
          FROM student_details
         WHERE region = :selectedRegion
         ORDER BY district
    ');
    $stmt->execute([':selectedRegion' => $regionFilter]);
} else {
    // Fetch districts across all regions
    $stmt = $pdo->query('SELECT DISTINCT district FROM student_details ORDER BY district');
}
$allDistricts = $stmt->fetchAll(PDO::FETCH_COLUMN);

/**********************************************
 * 4. Build the main query to group by region, district
 *    Then apply filters if provided
 **********************************************/
$query = "
    SELECT region, district, COUNT(*) AS total_students
      FROM student_details
      WHERE 1=1
";

// Optional: If you only want certain statuses, e.g. submitted, add:
// $query .= " AND status = 'submitted'";

if (!empty($regionFilter)) {
    $query .= " AND region = :regionFilter";
}
if (!empty($districtFilter)) {
    $query .= " AND district = :districtFilter";
}

$query .= " GROUP BY region, district ORDER BY region, district";

/**********************************************
 * 5. Prepare & Bind
 **********************************************/
$stmt = $pdo->prepare($query);

if (!empty($regionFilter)) {
    $stmt->bindParam(':regionFilter', $regionFilter, PDO::PARAM_STR);
}
if (!empty($districtFilter)) {
    $stmt->bindParam(':districtFilter', $districtFilter, PDO::PARAM_STR);
}

$stmt->execute();
$results = $stmt->fetchAll(PDO::FETCH_ASSOC);

/**********************************************
 * 6. Find the maximum student count
 *    (for the progress bar percentage)
 **********************************************/
$maxStudents = 0;
foreach ($results as $row) {
    if ($row['total_students'] > $maxStudents) {
        $maxStudents = $row['total_students'];
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Student Counts by Region and District</title>
  <!-- Bootstrap 5 CSS -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">

  <style>
    .table td, .table th {
      vertical-align: middle;
    }
    .progress {
      height: 25px;
    }
  </style>
</head>
<body>

<div class="container mt-4">
    <h2 class="mb-4">Filter Student Counts by Region and District</h2>

    <!-- 7. Filter Form -->
    <form method="GET" action="region_district_counts.php" class="row g-3 mb-4">
        <div class="col-md-4">
            <label for="region" class="form-label">Select Region (optional)</label>
            <select name="region" id="region" class="form-select" onchange="this.form.submit()">
                <option value="">All Regions</option>
                <?php foreach ($allRegions as $reg): ?>
                    <option value="<?php echo htmlspecialchars($reg, ENT_QUOTES, 'UTF-8'); ?>"
                        <?php echo ($regionFilter === $reg) ? 'selected' : ''; ?>>
                        <?php echo htmlspecialchars($reg, ENT_QUOTES, 'UTF-8'); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="col-md-4">
            <label for="district" class="form-label">Select District (optional)</label>
            <select name="district" id="district" class="form-select" onchange="this.form.submit()">
                <option value="">All Districts</option>
                <?php foreach ($allDistricts as $dist): ?>
                    <option value="<?php echo htmlspecialchars($dist, ENT_QUOTES, 'UTF-8'); ?>"
                        <?php echo ($districtFilter === $dist) ? 'selected' : ''; ?>>
                        <?php echo htmlspecialchars($dist, ENT_QUOTES, 'UTF-8'); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>
        <!-- You could add a "Submit" button if you prefer, but the onchange above auto-submits -->
    </form>

    <!-- 8. Display Results in a Table -->
    <?php if (!empty($results)): ?>
        <table class="table table-striped">
          <thead class="table-dark">
            <tr>
              <th scope="col">Region</th>
              <th scope="col">District</th>
              <th scope="col">Total Students</th>
              <th scope="col">Visual</th>
            </tr>
          </thead>
          <tbody>
            <?php foreach ($results as $row): ?>
              <?php
                $region   = htmlspecialchars($row['region'], ENT_QUOTES, 'UTF-8');
                $district = htmlspecialchars($row['district'], ENT_QUOTES, 'UTF-8');
                $count    = (int)$row['total_students'];
                $percentage = ($maxStudents > 0) ? ($count / $maxStudents) * 100 : 0;
              ?>
              <tr>
                <td><?php echo $region; ?></td>
                <td><?php echo $district; ?></td>
                <td><?php echo $count; ?></td>
                <td style="max-width: 200px;">
                  <div class="progress">
                    <div class="progress-bar"
                         role="progressbar"
                         style="width: <?php echo $percentage; ?>%;"
                         aria-valuenow="<?php echo round($percentage); ?>"
                         aria-valuemin="0"
                         aria-valuemax="100">
                      <?php echo round($percentage); ?>%
                    </div>
                  </div>
                </td>
              </tr>
            <?php endforeach; ?>
          </tbody>
        </table>
    <?php else: ?>
        <p class="text-muted">No results found for the selected filter(s).</p>
    <?php endif; ?>
</div>

<!-- Bootstrap 5 JS (includes Popper) -->
<script src="https://cdn.jsdelivr.net/npm/@popperjs/core@2.10.2/dist/umd/popper.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.min.js"></script>
</body>
</html>
