<?php
/**
 * Protect this page by starting a session and checking if 'admin_id' is set.
 * Modify 'login.php' to the correct login script/path in your environment.
 */
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');  // Redirect to your login page
    exit();
}

include '../config/database.php'; // Include the PDO connection

$filter_region = isset($_GET['region']) ? $_GET['region'] : '';
$filter_type   = isset($_GET['type'])   ? $_GET['type']   : 'assigned';

try {
    // Fetch all regions for filter dropdown
    $stmtRegions = $pdo->query("SELECT DISTINCT region FROM districts");
    $regions = $stmtRegions->fetchAll(PDO::FETCH_COLUMN);

    // Fetch assigned districts
    $queryAssigned = "
        SELECT d.region, d.district, ll.name AS lecturer_name 
        FROM districts d 
        JOIN lecturer_assigned_districts lad ON d.district = lad.district 
        JOIN lecturer_login ll ON lad.lecturer_id = ll.id
    ";
    if ($filter_region) {
        $queryAssigned .= " WHERE d.region = ?";
    }
    $stmtAssigned = $pdo->prepare($queryAssigned);
    $stmtAssigned->execute($filter_region ? [$filter_region] : []);
    $districts_by_region = [];
    while ($row = $stmtAssigned->fetch()) {
        $region = $row['region'];
        $districts_by_region[$region][] = [
            'district'      => $row['district'],
            'lecturer_name' => $row['lecturer_name']
        ];
    }

    // Fetch unassigned districts and students who selected them
    $queryUnassigned = "
        SELECT DISTINCT d.region, d.district, sd.surname, sd.first_name 
        FROM districts d 
        LEFT JOIN lecturer_assigned_districts lad ON d.district = lad.district 
        JOIN student_details sd ON d.district = sd.district 
        WHERE lad.district IS NULL
    ";
    if ($filter_region) {
        $queryUnassigned .= " AND d.region = ?";
    }
    $stmtUnassigned = $pdo->prepare($queryUnassigned);
    $stmtUnassigned->execute($filter_region ? [$filter_region] : []);
    $unassigned_districts = [];
    while ($row = $stmtUnassigned->fetch()) {
        $region       = $row['region'];
        $district     = $row['district'];
        $student_name = $row['first_name'] . ' ' . $row['surname'];
        $unassigned_districts[$region][$district][] = $student_name;
    }

} catch (PDOException $e) {
    die("Database query error: " . htmlspecialchars($e->getMessage(), ENT_QUOTES, 'UTF-8'));
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>District Assignment Status</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body {
            background-color: #eef1f7;
            font-family: 'Inter', sans-serif;
        }
        .container {
            max-width: 1200px;
        }
        .card {
            border: none;
            border-radius: 12px;
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1);
            margin-bottom: 20px;
            overflow: hidden;
        }
        .table {
            margin-bottom: 0;
        }
        h2 {
            color: #2c3e50;
            font-weight: bold;
            text-align: center;
            margin-bottom: 30px;
        }
    </style>
</head>
<body>

<?php include 'admin_header.php'; ?>

<div class="container mt-5">
    <h2>District Assignment Status</h2>
    <form method="GET" class="mb-4">
        <div class="row">
            <div class="col-md-6">
                <label for="region" class="form-label">Filter by Region:</label>
                <select name="region" id="region" class="form-select" onchange="this.form.submit()">
                    <option value="">All Regions</option>
                    <?php foreach ($regions as $region): ?>
                        <option 
                            value="<?php echo htmlspecialchars($region, ENT_QUOTES, 'UTF-8'); ?>" 
                            <?php echo $filter_region == $region ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($region, ENT_QUOTES, 'UTF-8'); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-6">
                <label class="form-label">Filter by Type:</label>
                <div class="d-flex">
                    <div class="form-check me-3">
                        <input 
                            type="radio" 
                            class="form-check-input" 
                            name="type" 
                            value="assigned" 
                            id="assigned"
                            <?php echo $filter_type == 'assigned' ? 'checked' : ''; ?>
                            onchange="this.form.submit()">
                        <label class="form-check-label" for="assigned">Assigned Districts</label>
                    </div>
                    <div class="form-check">
                        <input 
                            type="radio" 
                            class="form-check-input" 
                            name="type" 
                            value="unassigned" 
                            id="unassigned"
                            <?php echo $filter_type == 'unassigned' ? 'checked' : ''; ?>
                            onchange="this.form.submit()">
                        <label class="form-check-label" for="unassigned">Unassigned Districts</label>
                    </div>
                </div>
            </div>
        </div>
    </form>
    
    <div class="row">
        <div class="col-md-12">
            <?php if ($filter_type == 'assigned'): ?>
                <?php foreach ($districts_by_region as $region => $districts): ?>
                    <div class="card">
                        <div class="card-header">
                            Assigned Districts - <?php echo htmlspecialchars($region, ENT_QUOTES, 'UTF-8'); ?>
                        </div>
                        <div class="card-body">
                            <table class="table table-bordered">
                                <thead>
                                    <tr>
                                        <th>District</th>
                                        <th>Assigned Lecturer</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($districts as $districtData): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($districtData['district'], ENT_QUOTES, 'UTF-8'); ?></td>
                                            <td><?php echo htmlspecialchars($districtData['lecturer_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php elseif ($filter_type == 'unassigned'): ?>
                <div class="card">
                    <div class="card-header">Unassigned Districts</div>
                    <div class="card-body">
                        <table class="table table-bordered">
                            <thead>
                                <tr>
                                    <th>District</th>
                                    <th>Students</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($unassigned_districts as $region => $districts): ?>
                                    <?php foreach ($districts as $district => $students): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($district, ENT_QUOTES, 'UTF-8'); ?></td>
                                            <td>
                                                <?php
                                                    echo implode(', ', array_map(
                                                        fn($name) => htmlspecialchars($name, ENT_QUOTES, 'UTF-8'),
                                                        $students
                                                    ));
                                                ?>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>
</body>
</html>
