<?php
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

include '../config/database.php';

/**
 * 1. Handle "Return" logic if form is submitted.
 */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'return_mapping') {
    $lecturerId = $_POST['lecturer_id'];

    // Perform whatever action you'd like to revert/return the mapping
    // For example, setting `mapping_submitted` back to 0:
    $updateStmt = $pdo->prepare("UPDATE lecturer_login SET mapping_submitted = 0 WHERE id = :lecturer_id");
    $updateStmt->execute(['lecturer_id' => $lecturerId]);

    // Optionally, you can set a success message to display after reloading the page
    $_SESSION['flash_message'] = "Mapping returned successfully.";
    // No need to redirect to another page; simply refresh this page to see updated data
}

/**
 * 2. Fetch all lecturers with submitted mappings
 */
$stmt = $pdo->prepare('
    SELECT 
        l.id AS lecturer_id,
        l.name AS lecturer_name,
        l.department,
        l.mapping_submitted,
        GROUP_CONCAT(DISTINCT lp.preferred_region) AS preferred_regions,
        SUM(m.distance) AS total_distance,
        SUM(m.nights) AS total_nights
    FROM lecturer_login l
    LEFT JOIN supervision_mapping m ON l.id = m.lecturer_id
    LEFT JOIN lecturer_preferences lp ON l.id = lp.lecturer_id
    WHERE l.mapping_submitted = 1
    GROUP BY l.id
    ORDER BY l.name
');
$stmt->execute();
$lecturers = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Lecturer Mapping</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
</head>
<body>
<?php include 'admin_header.php'; // Include header ?>

<div class="container mt-4">
    <h2>Lecturer Mapping</h2>

    <!-- Display a success message if one exists -->
    <?php if(isset($_SESSION['flash_message'])): ?>
        <div class="alert alert-success">
            <?php 
                echo $_SESSION['flash_message']; 
                unset($_SESSION['flash_message']); // Clear after displaying
            ?>
        </div>
    <?php endif; ?>

    <?php if (empty($lecturers)): ?>
        <p>No lecturers or mappings found.</p>
    <?php else: ?>
        <table class="table table-striped">
            <thead>
                <tr>
                    <th>Lecturer Name</th>
                    <th>Department</th>
                    <th>Preferred Regions</th>
                    <th>Total Distance (km)</th>
                    <th>Total Nights</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($lecturers as $lecturer): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($lecturer['lecturer_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                        <td><?php echo htmlspecialchars($lecturer['department'], ENT_QUOTES, 'UTF-8'); ?></td>
                        <td><?php echo htmlspecialchars($lecturer['preferred_regions'], ENT_QUOTES, 'UTF-8'); ?></td>
                        <td><?php echo htmlspecialchars($lecturer['total_distance'], ENT_QUOTES, 'UTF-8'); ?></td>
                        <td><?php echo htmlspecialchars($lecturer['total_nights'], ENT_QUOTES, 'UTF-8'); ?></td>
                        <td>
                            <!-- View Button to Open Modal -->
                            <button type="button" class="btn btn-outline-primary btn-sm" data-bs-toggle="modal" 
                                    data-bs-target="#pdfModal" 
                                    data-lecturer-id="<?php echo $lecturer['lecturer_id']; ?>">
                                <i class="fas fa-eye"></i> View
                            </button>
                            <!-- Return Button (submit form to the same page) -->
                            <form method="POST" style="display:inline;">
                                <input type="hidden" name="lecturer_id" value="<?php echo $lecturer['lecturer_id']; ?>">
                                <input type="hidden" name="action" value="return_mapping">
                                <button type="submit" class="btn btn-outline-success btn-sm">
                                    <i class="fas fa-arrow-left"></i> Return
                                </button>
                            </form>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    <?php endif; ?>
</div>

<!-- Modal to Display PDF -->
<div class="modal fade" id="pdfModal" tabindex="-1" aria-labelledby="pdfModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="pdfModalLabel">Supervision Details</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <iframe id="pdfFrame" src="" style="width: 100%; height: 500px;" frameborder="0"></iframe>
            </div>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/@popperjs/core@2.10.2/dist/umd/popper.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.min.js"></script>
<script>
    // JavaScript to dynamically load PDF in modal
    document.addEventListener('DOMContentLoaded', function () {
        var pdfModal = document.getElementById('pdfModal');
        pdfModal.addEventListener('show.bs.modal', function (event) {
            var button = event.relatedTarget; // Button that triggered the modal
            var lecturerId = button.getAttribute('data-lecturer-id'); // Extract lecturer ID from data attribute

            // Set the iframe source to load the PDF
            var iframe = document.getElementById('pdfFrame');
            iframe.src = 'view_mapping_pdf.php?lecturer_id=' + lecturerId;
        });

        pdfModal.addEventListener('hidden.bs.modal', function () {
            var iframe = document.getElementById('pdfFrame');
            iframe.src = ''; // Clear the iframe source when modal is closed
        });
    });
</script>
</body>
</html>
