<?php
// Start output buffering to prevent accidental output
ob_start();
require '../vendor/autoload.php'; // Ensure you have PHPSpreadsheet installed
include '../config/database.php';

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Worksheet\PageSetup;

// Fetch all lecturers and their assigned districts
$query = '
    SELECT 
        ll.id AS lecturer_id,
        ll.name AS lecturer_name,
        lad.district AS assigned_district,
        lad.region AS assigned_region
    FROM lecturer_login ll
    JOIN lecturer_assigned_districts lad ON ll.id = lad.lecturer_id
    ORDER BY ll.name ASC, lad.region ASC, lad.district ASC';

$stmt = $pdo->prepare($query);
$stmt->execute();
$lecturerDistricts = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch all students with "submitted" status
$query = '
    SELECT 
        id AS student_id,
        lecturer_id,
        district,
        region,
        first_name,
        surname,
        organisation_name,
        location
    FROM student_details 
    WHERE status = "submitted"
    ORDER BY lecturer_id ASC, region ASC, district ASC, surname ASC';

$stmt = $pdo->prepare($query);
$stmt->execute();
$submittedStudents = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Organize data by lecturer and region
$lecturerData = [];
$totalStudentsAssigned = 0;

foreach ($lecturerDistricts as $row) {
    $lecturerId = $row['lecturer_id'];
    $lecturerName = ucwords(strtolower($row['lecturer_name'])); // Proper case
    $region = ucwords(strtolower($row['assigned_region']));
    $district = ucwords(strtolower($row['assigned_district']));

    if (!isset($lecturerData[$lecturerId])) {
        $lecturerData[$lecturerId] = [
            'name' => $lecturerName,
            'regions' => [],
            'total_students' => 0
        ];
    }

    if (!isset($lecturerData[$lecturerId]['regions'][$region])) {
        $lecturerData[$lecturerId]['regions'][$region] = [];
    }

    $lecturerData[$lecturerId]['regions'][$region][$district] = [];
}

// Assign students under their correct lecturer, region, and district
foreach ($submittedStudents as $student) {
    $lecturerId = $student['lecturer_id'];
    $region = ucwords(strtolower($student['region']));
    $district = ucwords(strtolower($student['district']));

    if (isset($lecturerData[$lecturerId]['regions'][$region][$district])) {
        $lecturerData[$lecturerId]['regions'][$region][$district][] = [
            'name' => ucwords(strtolower($student['first_name'] . ' ' . $student['surname'])), // Proper case
            'organisation' => $student['organisation_name'],
            'location' => $student['location']
        ];
        $lecturerData[$lecturerId]['total_students']++;
        $totalStudentsAssigned++;
    }
}

// Count total students with "submitted" status
$totalSubmittedStudents = count($submittedStudents);

// Count total students with "submitted" status but NOT assigned to any lecturer
$totalUnassignedStudents = $totalSubmittedStudents - $totalStudentsAssigned;

// Create Excel Spreadsheet
$spreadsheet = new Spreadsheet();

// Use the default sheet for Summary
$summarySheet = $spreadsheet->getActiveSheet();
$summarySheet->setTitle("Summary");

// Set Summary Sheet to Landscape Mode
$summarySheet->getPageSetup()->setOrientation(PageSetup::ORIENTATION_LANDSCAPE);

// Set Summary Headers
$summarySheet->setCellValue('A1', 'Summary of Assigned Students');
$summarySheet->getStyle('A1')->getFont()->setBold(true)->setSize(14);
$summarySheet->mergeCells('A1:B1');

$summarySheet->setCellValue('A3', 'Total Students with Submitted Status:');
$summarySheet->setCellValue('B3', $totalSubmittedStudents);
$summarySheet->getStyle('A3')->getFont()->setBold(true);

$summarySheet->setCellValue('A4', 'Total Students Assigned to Lecturers:');
$summarySheet->setCellValue('B4', $totalStudentsAssigned);
$summarySheet->getStyle('A4')->getFont()->setBold(true);

$summarySheet->setCellValue('A5', 'Total Students Not Assigned to Any Lecturer:');
$summarySheet->setCellValue('B5', $totalUnassignedStudents);
$summarySheet->getStyle('A5')->getFont()->setBold(true);

// Add Lecturer-wise Student Count
$summarySheet->setCellValue('A7', 'Lecturer Name');
$summarySheet->setCellValue('B7', 'Number of Assigned Students');
$summarySheet->getStyle('A7:B7')->getFont()->setBold(true);
$summaryRow = 8;

foreach ($lecturerData as $lecturerId => $lecturer) {
    $summarySheet->setCellValue("A$summaryRow", $lecturer['name']);
    $summarySheet->setCellValue("B$summaryRow", $lecturer['total_students']);
    $summaryRow++;
}

// Auto-Size Columns for Summary
foreach (range('A', 'B') as $col) {
    $summarySheet->getColumnDimension($col)->setAutoSize(true);
}

// Create Separate Sheet for Each Lecturer
foreach ($lecturerData as $lecturerId => $lecturer) {
    $lecturerName = $lecturer['name'];

    // Create a new worksheet for the lecturer
    $sheet = new \PhpOffice\PhpSpreadsheet\Worksheet\Worksheet($spreadsheet, $lecturerName);
    $spreadsheet->addSheet($sheet);

    // Set Lecturer Sheet to Landscape Mode
    $sheet->getPageSetup()->setOrientation(PageSetup::ORIENTATION_LANDSCAPE);

    $row = 1;

    // Populate Data per Region and District
    foreach ($lecturer['regions'] as $region => $districts) {
        $sheet->setCellValue("A$row", strtoupper($region)); // Region Header
        $sheet->getStyle("A$row")->getFont()->setBold(true)->setSize(14);
        $row++;

        foreach ($districts as $district => $students) {
            $sheet->setCellValue("A$row", "District: " . strtoupper($district)); // District Header
            $sheet->getStyle("A$row")->getFont()->setBold(true)->setSize(12);
            $row++;

            // Table Headers
            $sheet->setCellValue("A$row", "Student Name");
            $sheet->setCellValue("B$row", "Organisation");
            $sheet->setCellValue("C$row", "Location");
            $sheet->getStyle("A$row:C$row")->getFont()->setBold(true);
            $row++;

            if (!empty($students)) {
                foreach ($students as $student) {
                    $sheet->setCellValue("A$row", $student['name']);
                    $sheet->setCellValue("B$row", $student['organisation']);
                    $sheet->setCellValue("C$row", $student['location']);
                    $row++;
                }
            } else {
                $sheet->setCellValue("A$row", "No Students Assigned");
                $sheet->getStyle("A$row")->getFont()->setItalic(true);
                $row++;
            }

            $row += 2;
        }
    }

    // Add Total Students Assigned at the Bottom
    $sheet->setCellValue("A$row", "Total Students Assigned: " . $lecturer['total_students']);
    $sheet->getStyle("A$row")->getFont()->setBold(true)->setSize(12);

    foreach (range('A', 'C') as $col) {
        $sheet->getColumnDimension($col)->setAutoSize(true);
    }
}

// Set active sheet to the summary sheet
$spreadsheet->setActiveSheetIndex(0);

// Set headers for download
header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
header('Content-Disposition: attachment; filename="Assigned_Students_' . date('Y-m-d') . '.xlsx"');
header('Cache-Control: max-age=0');

// Generate the Excel file and output it
$writer = new Xlsx($spreadsheet);
$writer->save('php://output');
exit();
