<?php
require '../vendor/autoload.php';

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Reader\Xlsx as XlsxReader;

session_start();
if (!isset($_SESSION['admin_id'])) {
    header('Location: login');
    exit;
}
include 'admin_header.php';
include '../config/database.php';

// Fetch regions and districts for the filter
$stmt = $pdo->prepare('SELECT DISTINCT region FROM districts');
$stmt->execute();
$regions = $stmt->fetchAll(PDO::FETCH_ASSOC);

$stmt = $pdo->prepare('SELECT * FROM districts');
$stmt->execute();
$districts = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Handle lecturer assignment
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['assign_lecturer'])) {
    $lecturer_id = htmlspecialchars($_POST['lecturer_id'], ENT_QUOTES, 'UTF-8');
    $student_id = htmlspecialchars($_POST['student_id'], ENT_QUOTES, 'UTF-8');

    $stmt = $pdo->prepare('UPDATE student_details SET lecturer_id = ? WHERE id = ?');
    $stmt->execute([$lecturer_id, $student_id]);

    header('Location: assign_lecturers');
    exit;
}

// Handle lecturer unassignment
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['unassign_lecturer'])) {
    $student_id = htmlspecialchars($_POST['student_id'], ENT_QUOTES, 'UTF-8');

    $stmt = $pdo->prepare('UPDATE student_details SET lecturer_id = NULL WHERE id = ?');
    $stmt->execute([$student_id]);

    header('Location: assign_lecturers');
    exit;
}

// Handle export to Excel
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['export_excel'])) {
    $stmt = $pdo->prepare('SELECT sd.*, u.index_number, l.name AS lecturer_name FROM student_details sd 
                            JOIN users u ON sd.user_id = u.id
                            LEFT JOIN lecturer_login l ON sd.lecturer_id = l.id');
    $stmt->execute();
    $students = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $spreadsheet = new Spreadsheet();
    $sheet = $spreadsheet->getActiveSheet();

    $sheet->setCellValue('A1', 'No.');
    $sheet->setCellValue('B1', 'Name');
    $sheet->setCellValue('C1', 'Index Number');
    $sheet->setCellValue('D1', 'Telephone Number');
    $sheet->setCellValue('E1', 'Name of Organisation');
    $sheet->setCellValue('F1', 'Location');
    $sheet->setCellValue('G1', 'District');
    $sheet->setCellValue('H1', 'Region');
    $sheet->setCellValue('I1', 'Supervisor');

    $row = 2;
    foreach ($students as $index => $student) {
        $sheet->setCellValue('A' . $row, $index + 1);
        $sheet->setCellValue('B' . $row, htmlspecialchars($student['first_name'] . ' ' . $student['surname'], ENT_QUOTES, 'UTF-8'));
        $sheet->setCellValue('C' . $row, htmlspecialchars($student['index_number'], ENT_QUOTES, 'UTF-8'));
        $sheet->setCellValue('D' . $row, htmlspecialchars($student['telephone'], ENT_QUOTES, 'UTF-8'));
        $sheet->setCellValue('E' . $row, htmlspecialchars($student['organisation_name'], ENT_QUOTES, 'UTF-8'));
        $sheet->setCellValue('F' . $row, htmlspecialchars($student['location'], ENT_QUOTES, 'UTF-8'));
        $sheet->setCellValue('G' . $row, htmlspecialchars($student['district'], ENT_QUOTES, 'UTF-8'));
        $sheet->setCellValue('H' . $row, htmlspecialchars($student['region'], ENT_QUOTES, 'UTF-8'));
        $sheet->setCellValue('I' . $row, $student['lecturer_name'] ? htmlspecialchars(initializeOtherNames($student['lecturer_name']), ENT_QUOTES, 'UTF-8') : 'Not Assigned');
        $row++;
    }

    $writer = new Xlsx($spreadsheet);
    $fileName = 'Students_Supervisors.xlsx';
    $filePath = sys_get_temp_dir() . '/' . $fileName;
    $writer->save($filePath);

    header('Content-Description: File Transfer');
    header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    header('Content-Disposition: attachment; filename="' . htmlspecialchars(basename($filePath), ENT_QUOTES, 'UTF-8') . '"');
    header('Content-Transfer-Encoding: binary');
    header('Expires: 0');
    header('Cache-Control: must-revalidate');
    header('Pragma: public');
    header('Content-Length: ' . filesize($filePath));
    ob_clean();
    flush();
    readfile($filePath);
    unlink($filePath);
    exit;
}

// Handle import from Excel
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_FILES['excel_file'])) {
    $file = $_FILES['excel_file']['tmp_name'];
    $reader = new XlsxReader();
    $spreadsheet = $reader->load($file);
    $sheet = $spreadsheet->getActiveSheet();

    $rows = $sheet->toArray();
    foreach ($rows as $index => $row) {
        if ($index == 0) continue; // Skip header row
        $index_number = htmlspecialchars($row[2], ENT_QUOTES, 'UTF-8');
        $lecturer_name = htmlspecialchars($row[8], ENT_QUOTES, 'UTF-8');

        if (!empty($lecturer_name)) {
            // Fetch lecturer ID based on lecturer name
            $stmt = $pdo->prepare('SELECT id FROM lecturer_login WHERE name = ?');
            $stmt->execute([$lecturer_name]);
            $lecturer = $stmt->fetch(PDO::FETCH_ASSOC);

            if ($lecturer) {
                $lecturer_id = $lecturer['id'];

                // Update student with lecturer ID
                $stmt = $pdo->prepare('UPDATE student_details sd JOIN users u ON sd.user_id = u.id SET sd.lecturer_id = ? WHERE u.index_number = ?');
                $stmt->execute([$lecturer_id, $index_number]);
            }
        }
    }

    header('Location: assign_lecturers');
    exit;
}

// Fetch students with optional filters
$regionFilter = isset($_GET['region']) ? htmlspecialchars($_GET['region'], ENT_QUOTES, 'UTF-8') : '';
$districtFilter = isset($_GET['district']) ? htmlspecialchars($_GET['district'], ENT_QUOTES, 'UTF-8') : '';
$sort_column = isset($_GET['sort']) ? htmlspecialchars($_GET['sort'], ENT_QUOTES, 'UTF-8') : 'district';
$sort_order = isset($_GET['order']) && $_GET['order'] == 'desc' ? 'desc' : 'asc';

$query = "SELECT sd.*, u.index_number, l.name AS lecturer_name FROM student_details sd 
          JOIN users u ON sd.user_id = u.id
          LEFT JOIN lecturer_login l ON sd.lecturer_id = l.id
          WHERE 1";

if ($regionFilter) {
    $query .= " AND sd.region = :region";
}
if ($districtFilter) {
    $query .= " AND sd.district = :district";
}

$query .= " ORDER BY $sort_column $sort_order";
$stmt = $pdo->prepare($query);

if ($regionFilter) {
    $stmt->bindParam(':region', $regionFilter);
}
if ($districtFilter) {
    $stmt->bindParam(':district', $districtFilter);
}

$stmt->execute();
$students = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch lecturers
$stmt = $pdo->prepare('SELECT l.id, l.name, lad.district FROM lecturer_login l 
                        JOIN lecturer_assigned_districts lad ON l.id = lad.lecturer_id');
$stmt->execute();
$lecturers = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Group lecturers by district
$lecturers_by_district = [];
foreach ($lecturers as $lecturer) {
    $lecturers_by_district[htmlspecialchars($lecturer['district'], ENT_QUOTES, 'UTF-8')][] = $lecturer;
}

// Function to initialize other names
function initializeOtherNames($name) {
    $names = explode(' ', $name);
    $initials = '';
    for ($i = 0; $i < count($names) - 1; $i++) {
        $initials .= strtoupper($names[$i][0]) . '.';
    }
    return htmlspecialchars($initials . ' ' . end($names), ENT_QUOTES, 'UTF-8');
}
?>

<div class="container mt-4">

    <h2>Assign Lecturers to Students</h2>
    <div class="mb-4">
        <form method="POST" action="assign_lecturers.php" class="d-inline">
            <button type="submit" name="export_excel" class="btn ex-btn btn-sm">Export to Excel</button>
        </form>
    </div>

    <form method="GET" action="assign_lecturers.php" class="mb-4">
        <div class="row">
            <div class="col-md-4">
                <label for="regionFilter" class="form-label">Filter by Region</label>
                <select name="region" id="regionFilter" class="form-select" onchange="this.form.submit()">
                    <option value="">All Regions</option>
                    <?php foreach ($regions as $region): ?>
                        <option value="<?php echo htmlspecialchars($region['region'], ENT_QUOTES, 'UTF-8'); ?>" <?php echo $regionFilter == htmlspecialchars($region['region'], ENT_QUOTES, 'UTF-8') ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($region['region'], ENT_QUOTES, 'UTF-8'); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-4">
                <label for="districtFilter" class="form-label">Filter by District</label>
                <select name="district" id="districtFilter" class="form-select" onchange="this.form.submit()">
                    <option value="">All Districts</option>
                    <?php foreach ($districts as $district): ?>
                        <option value="<?php echo htmlspecialchars($district['district'], ENT_QUOTES, 'UTF-8'); ?>" <?php echo $districtFilter == htmlspecialchars($district['district'], ENT_QUOTES, 'UTF-8') ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($district['district'], ENT_QUOTES, 'UTF-8'); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
        </div>
    </form>
    <table class="table table-striped">
        <thead>
            <tr>
                <th>No.</th>
                <th>Name</th>
                <th>Index Number</th>
                <th>Telephone Number</th>
                <th>Name of Organisation</th>
                <th>Exact Location</th>
                <th><a href="?sort=district&order=<?php echo htmlspecialchars($sort_order == 'asc' ? 'desc' : 'asc', ENT_QUOTES, 'UTF-8'); ?>">District <i class="fas fa-sort"></i></a></th>
                <th><a href="?sort=region&order=<?php echo htmlspecialchars($sort_order == 'asc' ? 'desc' : 'asc', ENT_QUOTES, 'UTF-8'); ?>">Region <i class="fas fa-sort"></i></a></th>
                <th>Supervisor</th>
                <th>Action</th>
            </tr>
        </thead>
        <tbody>
            <?php $no = 1; ?>
            <?php foreach ($students as $student): ?>
                <tr>
                    <td><?php echo $no++; ?></td>
                    <td><?php echo htmlspecialchars($student['first_name'] . ' ' . $student['surname'], ENT_QUOTES, 'UTF-8'); ?></td>
                    <td><?php echo htmlspecialchars($student['index_number'], ENT_QUOTES, 'UTF-8'); ?></td>
                    <td><?php echo htmlspecialchars($student['telephone'], ENT_QUOTES, 'UTF-8'); ?></td>
                    <td><?php echo htmlspecialchars($student['organisation_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                    <td><?php echo htmlspecialchars($student['location'], ENT_QUOTES, 'UTF-8'); ?></td>
                    <td><?php echo htmlspecialchars($student['district'], ENT_QUOTES, 'UTF-8'); ?></td>
                    <td><?php echo htmlspecialchars($student['region'], ENT_QUOTES, 'UTF-8'); ?></td>
                    <td><?php echo $student['lecturer_name'] ? htmlspecialchars(initializeOtherNames($student['lecturer_name']), ENT_QUOTES, 'UTF-8') : 'Not Assigned'; ?></td>
                    <td>
                        <?php if ($student['lecturer_name']): ?>
                            <form method="POST" action="assign_lecturers.php" style="display:inline;">
                                <input type="hidden" name="student_id" value="<?php echo htmlspecialchars($student['id'], ENT_QUOTES, 'UTF-8'); ?>">
                                <button type="submit" name="unassign_lecturer" class="btn btn-outline-danger btn-sm"><i class="fas fa-times"></i></button>
                            </form>
                        <?php else: ?>
                            <button class="btn btn-outline-success btn-sm assign-supervisor" data-student-id="<?php echo htmlspecialchars($student['id'], ENT_QUOTES, 'UTF-8'); ?>" data-student-name="<?php echo htmlspecialchars($student['first_name'] . ' ' . $student['surname'], ENT_QUOTES, 'UTF-8'); ?>" data-student-district="<?php echo htmlspecialchars($student['district'], ENT_QUOTES, 'UTF-8'); ?>"><i class="fas fa-check"></i></button>
                        <?php endif; ?>
                    </td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
</div>

<!-- Modal -->
<div class="modal fade" id="assignSupervisorModal" tabindex="-1" aria-labelledby="assignSupervisorModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="assignSupervisorModalLabel">Assign Supervisor</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <form id="assignSupervisorForm" method="POST" action="assign_lecturers.php">
                    <input type="hidden" name="student_id" id="modalStudentId">
                    <div class="mb-3">
                        <label for="modalStudentName" class="form-label">Student Name</label>
                        <input type="text" class="form-control" id="modalStudentName" readonly>
                    </div>
                    <div class="mb-3">
                        <label for="modalLecturerId" class="form-label">Select Supervisor</label>
                        <select name="lecturer_id" id="modalLecturerId" class="form-select" required>
                            <option value="">Select Supervisor</option>
                            <!-- Options will be populated dynamically -->
                        </select>
                    </div>
                    <button type="submit" name="assign_lecturer" class="btn btn-outline-success"><i class="fas fa-check"></i></button>
                </form>
            </div>
        </div>
    </div>
</div>

<?php include 'admin_footer.php'; ?>

<script>
document.querySelectorAll('.assign-supervisor').forEach(button => {
    button.addEventListener('click', function() {
        const studentId = this.getAttribute('data-student-id');
        const studentName = this.getAttribute('data-student-name');
        const studentDistrict = this.getAttribute('data-student-district');
        const modalLecturerId = document.getElementById('modalLecturerId');
        
        // Set student details in modal
        document.getElementById('modalStudentId').value = studentId;
        document.getElementById('modalStudentName').value = studentName;
        
        // Clear previous options
        modalLecturerId.innerHTML = '<option value="">Select Supervisor</option>';
        
        // Add lecturers options based on student district
        const lecturersByDistrict = <?php echo json_encode($lecturers_by_district); ?>;
        if (lecturersByDistrict[studentDistrict]) {
            lecturersByDistrict[studentDistrict].forEach(lecturer => {
                const option = document.createElement('option');
                option.value = lecturer.id;
                option.textContent = lecturer.name;
                modalLecturerId.appendChild(option);
            });
        }
        
        // Show modal
        const assignSupervisorModal = new bootstrap.Modal(document.getElementById('assignSupervisorModal'));
        assignSupervisorModal.show();
    });
});

document.getElementById('excelFileInput').addEventListener('change', function() {
    document.getElementById('importForm').submit();
});
</script>
