<?php
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

include '../config/database.php';

// Fetch all students from the database
$stmt = $pdo->prepare('SELECT * FROM student_details');
$stmt->execute();
$students = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch distinct groups from the users table
$groupStmt = $pdo->prepare('SELECT DISTINCT `group` FROM users');
$groupStmt->execute();
$groups = $groupStmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch distinct regions from the student_details table
$regionStmt = $pdo->prepare('SELECT DISTINCT region FROM student_details');
$regionStmt->execute();
$regions = $regionStmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Student List</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="admin_style.css" rel="stylesheet">
    <!-- FontAwesome for icons -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <style>
        .container {
            max-width: 1200px;
        }

        .filter-row {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 15px;
        }

        .form-select, .btn {
            max-width: 200px;
            margin-right: 10px;
        }

        .btn-outline-tomato {
            border-color: tomato;
            color: tomato;
        }

        .btn-outline-tomato:hover {
            background-color: tomato;
            color: white;
        }

        .table {
            margin-top: 15px;
        }
    </style>
</head>
<body>

<?php include 'admin_header.php'; ?>

<div class="container mt-4">
    <h2 class="mb-3">All Students</h2>
    
    <div class="filter-row">
        <div class="d-flex">
            <select id="groupFilter" class="form-select">
                <option value="">Filter by Group</option>
                <?php foreach ($groups as $group): ?>
                    <option value="<?php echo htmlspecialchars($group['group'], ENT_QUOTES, 'UTF-8'); ?>"><?php echo htmlspecialchars($group['group'], ENT_QUOTES, 'UTF-8'); ?></option>
                <?php endforeach; ?>
            </select>
            <select id="regionFilter" class="form-select">
                <option value="">Filter by Region</option>
                <?php foreach ($regions as $region): ?>
                    <option value="<?php echo htmlspecialchars($region['region'], ENT_QUOTES, 'UTF-8'); ?>"><?php echo htmlspecialchars($region['region'], ENT_QUOTES, 'UTF-8'); ?></option>
                <?php endforeach; ?>
            </select>
            <button id="filterButton" class="btn btn-outline-tomato">Filter</button>
        </div>
        <button class="btn btn-outline-tomato" onclick="confirmDownload()">Download Letters</button>
    </div>

    <?php if (empty($students)): ?>
        <p>No students found.</p>
    <?php else: ?>
        <table class="table table-striped" id="studentTable">
            <thead>
                <tr>
                    <th>Full Name</th>
                    <th>Email</th>
                    <th>Telephone</th>
                    <th>Course</th>
                    <th>Organisation</th>
                    <th>Region</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($students as $student): ?>
                    <tr data-group="<?php echo htmlspecialchars($student['group'], ENT_QUOTES, 'UTF-8'); ?>" data-region="<?php echo htmlspecialchars($student['region'], ENT_QUOTES, 'UTF-8'); ?>">
                        <td><?php echo htmlspecialchars($student['first_name'], ENT_QUOTES, 'UTF-8') . ' ' . htmlspecialchars($student['surname'], ENT_QUOTES, 'UTF-8'); ?></td>
                        <td><?php echo htmlspecialchars($student['email'], ENT_QUOTES, 'UTF-8'); ?></td>
                        <td><?php echo htmlspecialchars($student['telephone'], ENT_QUOTES, 'UTF-8'); ?></td>
                        <td><?php echo htmlspecialchars($student['course'], ENT_QUOTES, 'UTF-8'); ?></td>
                        <td><?php echo htmlspecialchars($student['organisation_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                        <td><?php echo htmlspecialchars($student['region'], ENT_QUOTES, 'UTF-8'); ?></td>
                        <td><i class="fas fa-eye" style="cursor:pointer;" data-bs-toggle="modal" data-bs-target="#studentDetailsModal" onclick="showStudentDetails(<?php echo htmlspecialchars(json_encode($student), ENT_QUOTES, 'UTF-8'); ?>)"></i></td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    <?php endif; ?>
</div>

<!-- Modal for Student Details -->
<div class="modal fade" id="studentDetailsModal" tabindex="-1" aria-labelledby="studentDetailsModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="studentDetailsModalLabel">Student Details</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body" id="studentDetailsContent">
                <!-- Details will be injected here by JavaScript -->
            </div>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/@popperjs/core@2.10.2/dist/umd/popper.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.min.js"></script>
<script>
    document.addEventListener('DOMContentLoaded', function () {
        document.getElementById('filterButton').addEventListener('click', filterTable);
    });

    function filterTable() {
        var group = document.getElementById('groupFilter').value;
        var region = document.getElementById('regionFilter').value;
        var rows = document.querySelectorAll('#studentTable tbody tr');

        rows.forEach(function (row) {
            var rowGroup = row.getAttribute('data-group');
            var rowRegion = row.getAttribute('data-region');
            var show = true;

            if (group && rowGroup !== group) {
                show = false;
            }
            if (region && rowRegion !== region) {
                show = false;
            }

            row.style.display = show ? '' : 'none';
        });
    }

    function showStudentDetails(student) {
        var content = `
            <p><strong>Surname:</strong> ${student.surname}</p>
            <p><strong>First Name:</strong> ${student.first_name}</p>
            <p><strong>Other Names:</strong> ${student.other_names}</p>
            <p><strong>Telephone:</strong> ${student.telephone}</p>
            <p><strong>Email:</strong> ${student.email}</p>
            <p><strong>Course:</strong> ${student.course}</p>
            <p><strong>Organisation:</strong> ${student.organisation_name}</p>
            <p><strong>Months:</strong> ${student.months}</p>
            <p><strong>Location:</strong> ${student.location}</p>
            <p><strong>GPS Location:</strong> ${student.gps_location}</p>
            <p><strong>Region:</strong> ${student.region}</p>
            <p><strong>District:</strong> ${student.district}</p>
            ${student.acceptance_letter ? `<p><strong>Acceptance Letter:</strong> <a href="../uploads/${student.acceptance_letter}" target="_blank">View</a></p>` : ''}
        `;
        document.getElementById('studentDetailsContent').innerHTML = content;
    }

    function confirmDownload() {
        if (confirm('Are you sure you want to download all acceptance letters?')) {
            window.location.href = 'download_letters.php';
        }
    }
</script>
</body>
</html>

