<?php
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}
include '../config/database.php';

// Fetch distinct regions for filtering
$query = 'SELECT DISTINCT region FROM student_details WHERE district IS NOT NULL AND district != ""';
$stmt = $pdo->prepare($query);
$stmt->execute();
$regions = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch distinct districts for filtering
$query = 'SELECT DISTINCT district FROM student_details WHERE district IS NOT NULL AND district != ""';
$stmt = $pdo->prepare($query);
$stmt->execute();
$districts = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get the selected filters
$regionFilter = isset($_GET['region_filter']) ? htmlspecialchars($_GET['region_filter'], ENT_QUOTES, 'UTF-8') : '';
$districtFilter = isset($_GET['district_filter']) ? htmlspecialchars($_GET['district_filter'], ENT_QUOTES, 'UTF-8') : '';
$groupFilter = isset($_GET['group_filter']) && $_GET['group_filter'] === 'Teaching Practice' ? 'Teaching Practice' : '';
$unassignedFilter = isset($_GET['unassigned_filter']) && $_GET['unassigned_filter'] === 'on';

// Fetch students with optional filters
$query = '
    SELECT sd.id AS student_id, sd.first_name, sd.surname, sd.district, sd.region, u.index_number, sd.lecturer_id, u.group
    FROM student_details sd
    JOIN users u ON sd.user_id = u.id
    WHERE sd.district IS NOT NULL AND sd.district != ""';

// Apply filters
if ($regionFilter) {
    $query .= ' AND sd.region = :region_filter';
}
if ($districtFilter) {
    $query .= ' AND sd.district = :district_filter';
}
if ($groupFilter) {
    $query .= ' AND u.group = :group_filter';  // Fetch from users table
}
if ($unassignedFilter) {
    $query .= ' AND sd.lecturer_id IS NULL';
}

$stmt = $pdo->prepare($query);

// Bind parameters for filtering
if ($regionFilter) {
    $stmt->bindParam(':region_filter', $regionFilter);
}
if ($districtFilter) {
    $stmt->bindParam(':district_filter', $districtFilter);
}
if ($groupFilter) {
    $stmt->bindParam(':group_filter', $groupFilter);
}
$stmt->execute();
$students = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch lecturers, their assigned districts, and profile images
$query = 'SELECT ll.id AS lecturer_id, ll.name, ll.profile_image, lad.district
          FROM lecturer_login ll
          JOIN lecturer_assigned_districts lad ON ll.id = lad.lecturer_id';
$stmt = $pdo->prepare($query);
$stmt->execute();
$lecturers = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Organize lecturers by district
$lecturersByDistrict = [];
foreach ($lecturers as $lecturer) {
    $lecturersByDistrict[$lecturer['district']][] = $lecturer;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Assign Students to Lecturers</title>
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.1.1/css/all.min.css">
    <link rel="stylesheet" href="button.css">

    <style>
        /* Ensure buttons have no rounded corners */
        .btn {
            border-radius: 0;
        }
    </style>
</head>
<body>

<?php include 'admin_header.php'; ?>

<div class="container mt-4">
    <h2>Assign Students to Lecturers</h2>

    <!-- Filters -->
    <form method="GET" action="assign_students.php" class="mb-4">
        <div class="row">
            <div class="col-md-4">
                <select name="region_filter" class="form-select" onchange="this.form.submit()">
                    <option value="">All Regions</option>
                    <?php foreach ($regions as $region): ?>
                        <option value="<?php echo htmlspecialchars($region['region'], ENT_QUOTES, 'UTF-8'); ?>" <?php echo $regionFilter === $region['region'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($region['region'], ENT_QUOTES, 'UTF-8'); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-4">
                <select name="district_filter" class="form-select" onchange="this.form.submit()">
                    <option value="">All Districts</option>
                    <?php foreach ($districts as $district): ?>
                        <option value="<?php echo htmlspecialchars($district['district'], ENT_QUOTES, 'UTF-8'); ?>" <?php echo $districtFilter === $district['district'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($district['district'], ENT_QUOTES, 'UTF-8'); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-4">
                <div class="form-check">
                    <input class="form-check-input" type="checkbox" name="group_filter" value="Teaching Practice" id="groupFilter" onchange="this.form.submit()" <?php echo $groupFilter === 'Teaching Practice' ? 'checked' : ''; ?>>
                    <label class="form-check-label" for="groupFilter">
                        Teaching Practice
                    </label>
                </div>
                <div class="form-check">
                    <input class="form-check-input" type="checkbox" name="unassigned_filter" id="unassignedFilter" onchange="this.form.submit()" <?php echo $unassignedFilter ? 'checked' : ''; ?>>
                    <label class="form-check-label" for="unassignedFilter">
                        Show Unassigned Students
                    </label>
                </div>
            </div>
        </div>
    </form>

    <form id="assignmentForm" method="POST" action="process_assignment.php">
        <table class="table table-striped">
            <thead>
                <tr>
                    <th>Index Number</th>
                    <th>Student Name</th>
                    <th>Chosen District</th>
                    <th>Available Lecturers</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($students as $student): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($student['index_number']); ?></td>
                        <td><?php echo htmlspecialchars($student['first_name'] . ' ' . $student['surname']); ?></td>
                        <td><?php echo htmlspecialchars($student['district']); ?></td>
                        <td>
                            <div class="d-flex align-items-center">
                                <select name="lecturer_assignment[<?php echo htmlspecialchars($student['student_id']); ?>]" class="form-select me-2" <?php echo $student['lecturer_id'] ? 'disabled' : ''; ?>>
                                    <option value="">Select Lecturer</option>
                                    <?php 
                                    // Show all lecturers for Central Region
                                    if ($student['region'] === 'Central Region') {
                                        foreach ($lecturers as $lecturer) {
                                            echo '<option value="'.htmlspecialchars($lecturer['lecturer_id']).'" '.($student['lecturer_id'] == $lecturer['lecturer_id'] ? 'selected' : '').'>'.htmlspecialchars($lecturer['name']).'</option>';
                                        }
                                    } elseif (isset($lecturersByDistrict[$student['district']])) {
                                        foreach ($lecturersByDistrict[$student['district']] as $lecturer) {
                                            echo '<option value="'.htmlspecialchars($lecturer['lecturer_id']).'" '.($student['lecturer_id'] == $lecturer['lecturer_id'] ? 'selected' : '').'>'.htmlspecialchars($lecturer['name']).'</option>';
                                        }
                                    } else {
                                        echo '<option value="" disabled>No lecturers available for this district</option>';
                                    }
                                    ?>
                                </select>
                                <?php if ($student['lecturer_id']): ?>
                                    <button type="button" class="btn btn-sm btn-outline-danger undo-assignment" data-student-id="<?php echo htmlspecialchars($student['student_id']); ?>"><i class="fas fa-undo"></i></button>
                                <?php endif; ?>
                            </div>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>

        <button type="submit" class="btn btn-primary">Assign Lecturers to Students</button>
    </form>
</div>

<!-- Bootstrap JS and Popper.js -->
<script src="https://cdn.jsdelivr.net/npm/@popperjs/core@2.10.2/dist/umd/popper.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.min.js"></script>
<script>
    // JavaScript to handle the undo functionality
    document.querySelectorAll('.undo-assignment').forEach(button => {
        button.addEventListener('click', function() {
            const studentId = this.getAttribute('data-student-id');
            
            fetch('undo_assignment.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({ studentId })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('Assignment undone successfully.');
                    window.location.reload();
                } else {
                    alert('Error undoing assignment.');
                }
            })
            .catch(error => console.error('Error:', error));
        });
    });
</script>
</body>
</html>
